"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const dns_validated_certificate_1 = require("../lib/dns-validated-certificate");
test('creates CloudFormation Custom Resource', () => {
    const stack = new core_1.Stack();
    const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
        zoneName: 'example.com',
    });
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Certificate', {
        domainName: 'test.example.com',
        hostedZone: exampleDotComZone,
    });
    expect(stack).toHaveResource('AWS::CloudFormation::CustomResource', {
        DomainName: 'test.example.com',
        ServiceToken: {
            'Fn::GetAtt': [
                'CertificateCertificateRequestorFunction5E845413',
                'Arn',
            ],
        },
        HostedZoneId: {
            Ref: 'ExampleDotCom4D1B83AA',
        },
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.certificateRequestHandler',
        Runtime: 'nodejs10.x',
        Timeout: 900,
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyName: 'CertificateCertificateRequestorFunctionServiceRoleDefaultPolicy3C8845BC',
        Roles: [
            {
                Ref: 'CertificateCertificateRequestorFunctionServiceRoleC04C13DA',
            },
        ],
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        'acm:RequestCertificate',
                        'acm:DescribeCertificate',
                        'acm:DeleteCertificate',
                    ],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'route53:GetChange',
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'route53:changeResourceRecordSets',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':route53:::hostedzone/',
                                { Ref: 'ExampleDotCom4D1B83AA' },
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('adds validation error on domain mismatch', () => {
    const stack = new core_1.Stack();
    const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
        zoneName: 'hello.com',
    });
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
        domainName: 'example.com',
        hostedZone: helloDotComZone,
    });
    expect(() => {
        assert_1.SynthUtils.synthesize(stack);
    }).toThrow(/DNS zone hello.com is not authoritative for certificate domain name example.com/);
});
test('does not try to validate unresolved tokens', () => {
    const stack = new core_1.Stack();
    const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
        zoneName: core_1.Token.asString('hello.com'),
    });
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
        domainName: 'hello.com',
        hostedZone: helloDotComZone,
    });
    assert_1.SynthUtils.synthesize(stack); // does not throw
});
test('test root certificate', () => {
    const stack = new core_1.Stack();
    const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
        zoneName: 'example.com',
    });
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
        domainName: 'example.com',
        hostedZone: exampleDotComZone,
    });
    expect(stack).toHaveResource('AWS::CloudFormation::CustomResource', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CertCertificateRequestorFunction98FDF273',
                'Arn',
            ],
        },
        DomainName: 'example.com',
        HostedZoneId: {
            Ref: 'ExampleDotCom4D1B83AA',
        },
    });
});
test('works with imported zone', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack', {
        env: { account: '12345678', region: 'us-blue-5' },
    });
    const imported = aws_route53_1.HostedZone.fromLookup(stack, 'ExampleDotCom', {
        domainName: 'mydomain.com',
    });
    // WHEN
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
        domainName: 'mydomain.com',
        hostedZone: imported,
        route53Endpoint: 'https://api.route53.xxx.com',
    });
    // THEN
    expect(stack).toHaveResource('AWS::CloudFormation::CustomResource', {
        ServiceToken: {
            'Fn::GetAtt': [
                'CertCertificateRequestorFunction98FDF273',
                'Arn',
            ],
        },
        DomainName: 'mydomain.com',
        HostedZoneId: 'DUMMY',
        Route53Endpoint: 'https://api.route53.xxx.com',
    });
});
test('works with imported role', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack', {
        env: { account: '12345678', region: 'us-blue-5' },
    });
    const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
        zoneName: 'hello.com',
    });
    const role = iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::account-id:role/role-name');
    // WHEN
    new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
        domainName: 'hello.com',
        hostedZone: helloDotComZone,
        customResourceRole: role,
    });
    // THEN
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Role: 'arn:aws:iam::account-id:role/role-name',
    });
});
//# sourceMappingURL=data:application/json;base64,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