# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mate3', 'mate3.sunspec', 'mate3.sunspec.scripts', 'mate3.tests']

package_data = \
{'': ['*'],
 'mate3.sunspec': ['doc/*'],
 'mate3.tests': ['known_systems/chinezbrun/*']}

install_requires = \
['fixedint>=0.1.6,<0.2.0', 'loguru>=0.5.3,<0.6.0', 'pymodbus>=2.4,<3.0']

entry_points = \
{'console_scripts': ['mate3 = mate3.main:main']}

setup_kwargs = {
    'name': 'mate3',
    'version': '0.6.2',
    'description': 'Outback Mate3s Python Library',
    'long_description': '# Outback Mate 3 & 3s Python library & command line interface\n\n[![PyPI version](https://badge.fury.io/py/mate3.svg)](https://badge.fury.io/py/mate3)\n\nThis Python library aims to provide complete support for all Outback devices connected to a Mate3/Mate3s (or AXS port?) via Modbus. You can:\n\n- Read values in Python - and then do whatever you want with them e.g. monitoring/alerting/dynamic power management etc.\n- Write values - i.e. control your Outback system with Python.\n- (Hopefully) avoid ever having to know about Modbus at all. Just use the Python API or the CLI.\n- (Hopefully) get access to the full Outback spec in a \'user friendly\' form. You don\'t need to know about the magic numbers in Enums or Bitfields (or the SunSpec), or how to interpret fault codes, and nor do you have to worry about twos-complements and other such things.\n- Co-develop without giving access to your system. That is, you can \'dump\' a snapshot of your system and others can then interact with it (within reason) as if it were a real Mate3 over Modbus - which is great for testing/debugging/etc.\n\nTested on Python 3.7. May work on 3.6.\n\n## Installation\n\nThe recommended installation is as follows:\n\n```sh\npip install mate3\n```\n\nAfter this you should be able to run the `mate3` command. To access your Mate it must be connected to your local network using its ethernet port.\n\n## Background info you probably should know ...\n\nReading this will help you understand this libary and how to interact with your Mate.\n\n### Modbus\n\nHopefully, you don\'t need to worry about Modbus at all - this library should abstract that away for you. The key thing to note is that Modbus is a communication protocol, and this library works by interacting with the Mate3 physical devices using synchronous messages. So:\n\n- The information isn\'t \'live\' - it\'s only the latest state since we last read the values. Generally, you should be calling `read` or `write` before/after any operation you make.\n- Don\'t over-communicate! If you start doing too many `read`s or `write`s you might brick the Modbus interface of your Mate (requiring a reboot to fix). As a rule of thumb, you probably don\'t want to be reading more frequently than once per second (and even then, preferably only specific fields, and not the whole lot). Since it\'s a communication protocol (and it\'s not actually clear what the latency is inherent in the Mate), there\'s not much point reading faster that this anyway.\n- Given the above, you might want to use the caching options in the `Mate3Client`, which can allow you to completely avoid interacting with/bricking your Mate while you\'re developing code etc. It\'s really tedious having to restart it every time your have a bug in your code.\n- Weird things happen when encoding stuff into Modbus. Hopefully you\'ll never notice this, but if you see things where your `-1` is appearing as `65535` then yeh, that may be it.\n\n### SunSpec & Outback & Modbus\n\nYou can check out the details of how Outback implements Modbus in [./mate3/sunspec/doc](./mate3/sunspec/doc), but the key things to note are:\n\n- SunSpec is a generic Modbus implementation for distributed energy systems include e.g. solar. There\'s a bunch of existing definitions for what e.g. charge controllers, inverters, etc. should be.\n- Outback use these, but they have their own additional information they include - which they refer to as \'configuration\' definitions (generally as that\'s where the writeable fields live i.e. things you can change). Generally, when you\'re using this library you might see e.g. `charge_controller.config.absorb_volts`. Here the `charge_controller` is the SunSpec block, and we add on a special `config` field which is actually a pointer to the Outback configuration block. This is to try to abstract away the implementation details so you don\'t have to worry about their being multiple charge controller things, etc.\n\n### Pseudo-glossary\n\nWords are confusing. For now, take the below as a rough guide:\n  - `Field` - this is a definition of a field e.g. `absorb_volts` is `Uint16` with units of `"Volts"` etc.\n  - `Model` - This is generally referring to a specific Modbus \'block\' - which is really just a collection of fields that are generally aligned to a specific device e.g. an inverter model will have an output KWH field, which a charge controller model won\'t. (Again, it\'s confusing here as Outback generally have two models per device.) In the case above `charge_controller` represents one (SunSpec) model, and `charge_controller.config` another (Outback) model. \n  - `Device` - this is meant to represent a physical device and is basically our way of wrapping the Outback config model with the SunSpec one.\n  - `FieldValue` - this is kind of like a `Field` but with data (read from Modbus) included i.e. "the value of the field". It includes some nice things too like auto-scaling variables (\'cos floats aren\'t a thing) and simple `read` or `write` APIs.\n\n## More documentation?\n\nAt this stage, it doesn\'t exist - the best documentation is the code and [the examples](./examples), though this only works well for those who know Python. A few other quick tips:\n\n- Turn intellisense on! There\'s a bunch of typing in this library, so it\'ll make your life much easier e.g. for finding all the fields accessible from your charge controller, etc.\n- [./mate3/sunspec/models.py](./mate3/sunspec/models.py) has all of the key definitions for every model, including all the fields (each of which has name/units/description/etc.). Error flags and enums are properly defined there too.\n\n## Using the library\n\nMore documentation is needed (see above), but you can get a pretty code idea from [./examples/getting_started.py](./examples/getting_started.py), copied (somewhat) below.\n\n```python\n# Creating a client allows you to interface with the Mate. It also does a read of all devices connected to it (via the\n# hub) on initialisation:\nwith Mate3Client("...") as client:\n        # What\'s the system name?\n        mate = client.devices.mate3\n        print(mate.system_name)\n        # >>> FieldValue[system_name] | Mode.RW | Implemented | Value: OutBack Power Technologies | Read @ 2021-01-01 17:50:54.373077\n        \n        # Get the battery voltage. Note that it\'s auto-scaled appropriately.\n        fndc = client.devices.fndc\n        print(fndc.battery_voltage)\n        # >>> FieldValue[battery_voltage] | Mode.R | Implemented | Scale factor: -1 | Unscaled value: 506 | Value: 50.6 | ...\n         Read @ 2021-01-01 17:50:54.378941\n\n        # Get the (raw) values for the same device type on different ports.\n        inverters = client.devices.single_phase_radian_inverters\n        for port, inverter in inverters.items():\n            print(f"Output KW for inverter on port {port} is {inverter.output_kw.value}")\n        # >>> Output KW for inverter on port 1 is 0.7\n        # >>> Output KW for inverter on port 2 is 0.0\n\n        # Values aren\'t \'live\' - they\'re only updated whenever you initialise the client, call client.update_all() or\n        # re-read a particular value. Here\'s how we re-read the battery voltage. Note the change in the last_read field\n        time.sleep(0.1)\n        fndc.battery_voltage.read()\n        print(fndc.battery_voltage)\n        # >>> FieldValue[battery_voltage] | Mode.R | Implemented | Scale factor: -1 | Unscaled value: 506 | Value: 50.6 | Read @ 2021-01-01 17:50:54.483401\n\n        # Nice. Modbus fields that aren\'t implemented are easy to identify:\n        print(mate.alarm_email_enable.implemented)\n        # >>> False\n\n        # We can write new values to the device too. Note that we don\'t need to worry about scaling etc.\n        # WARNING: this will actually write stuff to your mate - see the warning below!\n        mate.system_name.write("New system name")\n        print(mate.system_name)\n        # >>>  FieldValue[system_name] | Mode.RW | Implemented | Value: New system name | Read @ 2021-01-01 17:50:54.483986\n\n        # All the fields and options are well defined so e.g. for enums you can see valid options e.g:\n        print(list(mate.ags_generator_type.field.options))\n        # >>> [<ags_generator_type.AC Gen: 0>, <ags_generator_type.DC Gen: 1>, <ags_generator_type.No Gen: 2>]\n\n        # In this case these are normal python Enums, so you can access them as expected, and assign them:\n        mate.ags_generator_type.write(mate.ags_generator_type.field.options["DC Gen"])\n        # >>> ags_generator_type.DC Gen\n```\n\n## Using the command line interface (CLI)\n\nA simple CLI is available, with four main sub-commands:\n\n- `read` - reads all of the values from the Mate3 and prints to stdout in a variety of formats.\n- `write` - writes values to the Mate3. (If you\'re doing anything serious you should use the python API.)\n- `devices` - shows the connected devices.\n- `dump` - dumps all of the raw modbus values to a (JSON) file in a format compatible with `CachingModbusClient` which you can then share with others to help in debugging any problems you may have. \n\nFor each you can access the help (i.e. `mate3 <cmd> -h`) for more information.\n\n## Warnings\n\nFirst, the big one:\n\n> **WARNING!** Please make sure you read [the license](https://github.com/adamcharnock/mate3/blob/master/LICENSE) before using any of the `write` functionality. You could easily damage your equipment by setting incorrect values (directly or indirectly).\n\nIn addition, there are other edges cases that may cause problems, mostly related to if a device is re-assigned a new port. For example, you have two inverters, read some values, then switch their ports over in the Hub before writing some values - which may now go to the \'wrong\' one. For now, it\'s safest not to do that, unless you restart the `Mate3Client` each time. On that note, the recommended approach if you need to poll over time is to recreate the `Mate3Client` on every poll (as opposed to re-using one), as that\'ll help avoid these (or other) issues. There are exceptions to this rule, but you should know why you\'re breaking it before you do so.\n\n## Troubleshooting\n\nSome ideas (which can be helpful for issues)\n\n### Set log-level to DEBUG\n\nSee `mate3 -h` for the CLI, otherwise the following (or similar) for python code:\n\n```python\nfrom loguru import logger\nlogger.remove()\nlogger.add(sys.stderr, level="DEBUG")\n```\n\n### List the devices\n\n```sh\n$ mate3 devices --host ...\nname                                               address    port\n----                                               -------    ----\nMate3                                              40069      None\nChargeController                                   40986      4\nChargeControllerConfiguration                      41014      4\n...\n```\nAre they all there?\n\n### Create a dump of the raw modbus values\n\nSee `mate3 dump -h`. You can send the resulting JSON file to someone to help debug. (Just note that it includes all the data about the Mate, e.g. any passwords etc.)\n\n## Writing data to Postgres\n\nSee [./examples/postgres_monitor/README.md](./examples/posgres_monitor/README.md)\n\n## Contributing\n\nSee [./CONTRIBUTING.md](./CONTRIBUTING.md)\n\n## Credits\n\nThis was originally a heavily refactored version of\n[basrijn\'s Outback_Mate3 library](https://github.com/basrijn/Outback_Mate3), though has largely been completely rewritten since. Thanks anyway basrijn!\n',
    'author': 'Adam Charnock',
    'author_email': 'adam@adamcharnock.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/adamcharnock/mate3/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
