# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['gencompose', 'convcompose']
install_requires = \
['click>=7.1.2,<8.0.0', 'pyyaml>=5.3.1,<6.0.0']

entry_points = \
{'console_scripts': ['gen-compose = gencompose:main',
                     'gen-compose-convert = convcompose:main']}

setup_kwargs = {
    'name': 'gen-compose',
    'version': '1.1.0',
    'description': 'Key generator for macos keybinding system',
    'long_description': '# Compose Key On Macos\n\n`gen-compose` - generates _compose key_ keybindings for macos.  \n\n> A compose key (sometimes called multi key) is a key on a computer keyboard that indicates that the following (usually 2 or more) keystrokes trigger the insertion of an alternate character, typically a precomposed character or a symbol.\n> https://en.wikipedia.org/wiki/Compose_key\n\nCompose keys lets you insert complex character by entering multiple characters in a succession:\n\n![screencast demonstration of compose key](./demo.gif)\n\ne.g. `<compose_key> + d + o + w + n` will insert `👇`\n\nMac os doesn\'t come with a compose key feature built-in. However there\'s a short hack to make it work:\n\n1. Keys can be rebound in mac via `~/Library/KeyBindings/DefaultKeyBinding.dict` dictionary file.\n2. The rebound keys can be chained like compose keys e.g. pressing `abcd` can be made to insert `AlphaBetaCharlieDad`\n3. Modifier keys cannot be rebound\n\nWith these three rules we can replicate compose key and even set it to work with a mod key!\n\n## Install\n\n`gen-compose` can be installed via python manager with py3.6+ versions:\n\n```\n$ pip3 install --user gen-compose\n$ gen-compose --help\nUsage: gen-compose [OPTIONS] [MAPPINGS]...\n\n  Generate macos rebind file from compose json mapping\n\nOptions:\n  --key TEXT  key to use as compose key  [default: §]\n  -r, --raw   just keymap without prefix\n  --help      Show this message and exit.\n```\n\n## Preconfig\n\n\n1. First lets fix modifier key issue by forcing modifier to be a character. For example to use `right_options` key we need to use [karabiner elements] and remap it to some unused key like `non_us_backslash`:\n![karabiner compose screenshot](./karabiner-compose.png)\n\n2. Now we have the compose key ready: if we click right_options it should insert `§` character  \n    However we cannot compose anything yet as we have no compose mappings yet. For that we need to modify keybindings dictionary located in `~/Library/KeyBindings/DefaultKeyBinding.dict`.  \n    It\'s written in some cryptic hard to edit format and here\'s where `gen-compose` comes in and lets you write `yaml` files instead!\n\n## Usage\n\n1. Create yaml mappings file (e.g. `mappings/readme.yaml`):\n    ```yaml\n    cat: "(^≗ω≗^)"\n    "+1": 👍\n    "-1": 👍\n    ":(": "my face is sad"\n    ```\n   This map defines key combinations and texts that will be inserted, e.g. `<compose_key><plus><number 1>` will insert thumbs up.  \n   _note: see [mappings](./mappings) directory for some built in mappings_\n2. Using `gen-compose` we generated `.dict` keybind file file from our yaml configuration:\n    ```shell\n    $ gen-compose mappings/readme.yaml\n    {"§" = {\n      "c" = {\n        "a" = {\n          "t" = ("insertText:", "(^≗ω≗^)");\n        };\n      };\n      "+" = {\n        "1" = ("insertText:", "👍");\n      };\n      "-" = {\n        "1" = ("insertText:", "👍");\n      };\n      ":" = {\n        "(" = ("insertText:", "my face is sad");\n      };\n    };}\n    ```\n   _note: multiple mappings can be used to generate a single keymap:_ `$ gen-compose map1.yaml map2.yaml`\n3. Now save it directly to keybinds file:\n    ```shell\n    $ gen-compose mappings/readme.yaml > ~/Library/KeyBindings/DefaultKeyBinding.dict\n    ```\n4. Restart your programs and type `§+1` and you\'ll see `👍`!\n5. Customize your own mapping or see `/mappings` for some existing configurations and have fun!\n\n\n_note: Some programs need a hard reboot to take in the map, like `kill -9` sort of reboot to start working._\n\n## Using XCompose Mappings\n\nLinux\'s xcompose mappings are supported via experimental conversion:\n\n```\n$ gen-compose-conver xcompose --help\nUsage: gen-compose-convert xcompose [OPTIONS] [FILES]...\n\n  Convert xcompose file, that follows format like: <Multi_key> <parenleft>\n  <period> <1> <parenright>: "⑴"\n\nOptions:\n  -c, --keep-comments  keep inline comments\n  --help               Show this message and exit.\n```\n\nFor example:\n\n```\n$ cat mappings/example.compose\n<Multi_key> <B> <bar> : "₿" U20BF  # BITCOIN SIGN\n$ gen-compose-convert xcompose mappings/example.compose --keep-comments > example.yaml\n$ cat example.yaml\n"B|": "₿"  # BITCOIN SIGN\n```\n\n\n## Notes and Issues\n\n* There\'s no way to really debug this other than trial and error and making sure applications are restarted after every update.  \n* When multiple keymaps (eg `compose-gen map1.yaml map2.yal`) provided they will be joined and updated in order they are provided.\n* Keymap does not allow completely overlapping keys. e.g. `egg` and `eggplant` will break because it\'s impossible to map two overlapping values as keymap functionality has no timeout (i.e. break if nothing is typed after a second and insert `egg`, instead it will wait indefinitely for you to finish `eggplant`)  \n    If overlap happens `compose-gen` will choose the shorter form and discard the long one.  \n    To fix this add end terminator character such as space: `egg<space>` rather than `egg` in your mappings.\n* `DefaultKeyBinding.dict` map doesn\'t work in some popular programs: \n    - JetBrain editors such as PyCharm, see: https://youtrack.jetbrains.com/issue/IDEA-127470\n    - QT apps such as Qutebrowser, see (and vote): https://bugreports.qt.io/browse/QTBUG-393\n\n\n\n#### Related Resources\n\nhttps://github.com/gnarf/osx-compose-key  \nhttp://lolengine.net/blog/2012/06/17/compose-key-on-os-x  \nhttp://bob.cakebox.net/osxcompose.php  \nhttps://github.com/ttscoff/KeyBindings\n\n[karabiner elements]: https://karabiner-elements.pqrs.org/\n',
    'author': 'Granitosaurus',
    'author_email': 'wraptile@pm.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Granitosaurus/macos-compose',
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
