"""
Module defining the list sampler object.
"""

import logging
from contextlib import suppress

from scisample.base_sampler import BaseSampler
from scisample.utils import test_for_uniform_lengths, ParameterMixIn

LOG = logging.getLogger(__name__)


class ListSampler(BaseSampler, ParameterMixIn):
    """
    Class defining basic list sampling.

    .. code:: yaml

        sampler:
            type: list
            constants:
                X1: 20
            parameters:
                X2: [5, 10]
                X3: [5, 10]

    Entries in the ``constants`` dictionary will be added to all samples.
    Entries in the ``parameters`` block will be matched one to one.  The
    result of the above block would be:

    .. code:: python

        [{X1: 20, X2: 5, X3: 5}, {X1: 20, X2: 10, X3: 10}]
    """
    def __init__(self, data):
        """
        Initialize the sampler.

        :param data: Dictionary of sampler data.
        """
        super().__init__(data)
        self.check_validity()

    def check_validity(self):
        super().check_validity()

        self._check_variables_existence()
        self._check_variables_for_dups()

        with suppress(KeyError):
            test_for_uniform_lengths(self._parsed_parameters.items())

    @property
    def parameters(self):
        """
        Return a of list of the parameters being generated by the
        sampler.
        """
        return self._parameters_constants_parameters_only()

    def get_samples(self):
        """
        Get samples from the sampler.

        This returns samples as a list of dictionaries, with the
        sample variables as the keys:

        .. code:: python

            [{'b': 0.89856, 'a': 1}, {'b': 0.923223, 'a': 1}, ... ]
        """
        if self._samples is not None:
            return self._samples

        self._samples = []

        num_samples = 1

        with suppress(KeyError):
            for key, value in self._parsed_parameters.items():
                num_samples = len(value)
                break

        for i in range(num_samples):
            new_sample = {}

            with suppress(KeyError):
                new_sample.update(self.data['constants'])

            with suppress(KeyError):
                for key, value in self._parsed_parameters.items():
                    new_sample[key] = value[i]

            self._samples.append(new_sample)

        return self._samples
