import openai

def OpenAI_inference_list(
        reviews,
        api_key: str,
        model="gpt-3.5-turbo",
        temperature=0.1,
        max_tokens=100,
        stream=False,
        max_input_tokens=300
    ):
    openai.api_key = api_key
    
    for review_data in reviews:
        content = review_data[0].get('text-message', '')
        sentiment = review_data[1]
        
        if len(content) > max_input_tokens:
            content = content[:max_input_tokens] + "..."
        
        chat_completion = openai.ChatCompletion.create(
            model=model,
            messages=[
                {
                    "role": "system",
                    "content": "Given the review and sentiment, provide insights on the customer's feelings about the product."
                },
                {
                    "role": "user",
                    "content": f"REVIEW: {content}\nSENTIMENT: {sentiment}\nQUESTION: How does the customer feel about the product?"
                }
            ],
            temperature=temperature,
            max_tokens=max_tokens,
            stream=stream
        )
        
        # Extract and append the OpenAI review to the current review data
        openai_review = chat_completion['choices'][0]['message']['content']
        review_data.append({"openai-review": openai_review})

    return reviews

def summarize_reviews_openai(
        reviews,  # List of lists containing reviews and corresponding sentiment data
        KEY: str,  # OpenAI API key for authentication
        max_tokens: int = 150,  # Maximum number of tokens to generate for the summary
        temperature: float = 0.3,  # Sampling temperature for text generation (lower values = more deterministic)
        top_p: float = 1,  # Top-p sampling value for controlling diversity in generated text
        model_id: str = "gpt-3.5-turbo",  # Model ID for OpenAI LLM used for inference
        stream: bool = False  # Whether to stream the response from the API (default is False)
    ):
    """
    Function that takes a list of reviews and their corresponding sentiment data, 
    and generates a summary of all reviews using the OpenAI model.
    
    Parameters:
    - reviews: List[List[Dict]]
        A list of lists where each sublist contains:
        - A dictionary with the key 'text-message' representing the review content.
        - A dictionary with sentiment data, including 'label' (e.g., 'positive', 'neutral') and 'score' (a confidence score).
    
    - KEY: str
        The OpenAI API key required to authenticate and make requests to the OpenAI API.
    
    - max_tokens: int (default = 150)
        The maximum number of tokens the model can generate for the summary.
    
    - temperature: float (default = 0.3)
        The temperature setting for the model, controlling the randomness of the output. Lower values result in more deterministic responses.
    
    - top_p: float (default = 1)
        Top-p sampling controls the diversity of the generated text. A value of 1 means no restriction on sampling.
    
    - model_id: str (default = "gpt-3.5-turbo")
        The specific model ID for the OpenAI model to use for inference.
    
    - stream: bool (default = False)
        Whether or not to enable streaming responses from the API.
    
    Returns:
    - summary: str
        A summary of the reviews generated by the OpenAI model.
    """
    
    # Set up OpenAI API key
    openai.api_key = KEY
    
    # Combine all the reviews into a single string for summarization
    combined_reviews = "\n".join([f"REVIEW: {review[0]['text-message']} (Sentiment: {review[1]['label']}, Score: {review[1]['score']})" for review in reviews])
    
    # Call the OpenAI API to generate a summary based on the combined reviews
    response = openai.ChatCompletion.create(
        model=model_id,  # Specify the model ID
        messages=[
            {
                "role": "system",
                "content": "Summarize the overall sentiment and key points from the reviews provided below."
            },
            {
                "role": "user",
                "content": f"Here are the reviews:\n{combined_reviews}\nPlease provide a concise summary of the overall sentiment and main feedback points."
            }
        ],
        temperature=temperature,  # Set the temperature
        top_p=top_p,  # Set the top-p value
        max_tokens=max_tokens,  # Limit the number of output tokens
        stream=stream  # Enable or disable streaming
    )
    
    # Extract the generated summary from the API response
    summary = response['choices'][0]['message']['content']
    
    return summary