from transformers import pipeline , AutoModelForCausalLM , AutoTokenizer

def LocalLLM_inference_list(
        reviews,
        model_name="TinyLlama/TinyLlama-1.1B-Chat-v1.0",
        Local_General_LLM_kwargs=None,
        max_input_tokens=300,
        device_map='cpu'
    ):
    """
    Function to perform LLM inference on a list of reviews using a local model.
    
    Parameters:
    - reviews: List[List[Dict]]
        A list of lists where each sublist contains:
        - A dictionary with the key 'text-message' representing the review content.
        - A dictionary with sentiment data (e.g., label and score).
    
    - model_name: str
        The local model name used for inference.
    
    - Local_General_LLM_kwargs: dict
        Keyword arguments for LLM inference (e.g., temperature, top_p).
    
    - max_input_tokens: int
        Maximum number of tokens to consider from the input review content (input truncation).
    
    - device_map: str
        Device configuration for model loading (e.g., 'auto', 'cpu').
    
    Returns:
    - reviews: List[List[Dict]]
        The input list with each sublist now having an additional dictionary containing the generated local LLM review.
    """
    
    # Default LLM arguments if none are provided
    if Local_General_LLM_kwargs is None:
        Local_General_LLM_kwargs = {'temperature': 0.1, 'top_p': 1}
    
    # Initialize the model with the given settings
    llm_pipeline = pipeline("text-generation", model=model_name, device_map=device_map, **Local_General_LLM_kwargs)
    
    for review_data in reviews:
        content = review_data[0].get('text-message', '')
        sentiment = review_data[1]
        
        if len(content) > max_input_tokens:
            content = content[:max_input_tokens] + "..."
        
        # Generate the response using the local LLM
        response = llm_pipeline(f"REVIEW: {content}\nSENTIMENT: {sentiment}\nQUESTION: How does the customer feel about the product?")[0]['generated_text']
        
        # Append the generated review to the current review data
        review_data.append({"local-llm-review": response})
    
    return reviews

def summarize_reviews_local(
        reviews,  # List of lists containing reviews and corresponding sentiment data
        model_path: str,  # Path to the local LLM model
        max_tokens: int = 2048,  # Maximum number of tokens to generate for the summary
        temperature: float = 0.3,  # Sampling temperature for text generation (lower values = more deterministic)
        top_p: float = 1.0,  # Top-p sampling value for controlling diversity in generated text
    ):
    """
    Function that takes a list of reviews and their corresponding sentiment data, 
    and generates a summary of all reviews using a local LLM.

    Parameters:
    - reviews: List[List[Dict]]
        A list of lists where each sublist contains:
        - A dictionary with the key 'text-message' representing the review content.
        - A dictionary with sentiment data, including 'label' (e.g., 'positive', 'neutral') and 'score' (a confidence score).
    
    - model_path: str
        The path to the local LLM model directory.
    
    - max_tokens: int (default = 150)
        The maximum number of tokens the model can generate for the summary.
    
    - temperature: float (default = 0.3)
        The temperature setting for the model, controlling the randomness of the output. Lower values result in more deterministic responses.
    
    - top_p: float (default = 1.0)
        Top-p sampling controls the diversity of the generated text. A value of 1.0 means no restriction on sampling.
    
    Returns:
    - summary: str
        A summary of the reviews generated by the local LLM.
    """

    # Load the local model and tokenizer
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model = AutoModelForCausalLM.from_pretrained(model_path)

    # Combine all the reviews into a single string for summarization
    combined_reviews = "\n".join([f"REVIEW: {review[0]['text-message']} (Sentiment: {review[1]['label']}, Score: {review[1]['score']})" for review in reviews])

    # Prepare the input prompt for the model
    prompt = f"Here are the reviews:\n{combined_reviews}\nPlease provide a concise summary of the overall sentiment and main feedback points."

    # Generate the summary using the local model
    inputs = tokenizer(prompt, return_tensors="pt", max_length=512, truncation=True)
    outputs = model.generate(
        inputs.input_ids,
        max_length=max_tokens,
        temperature=temperature,
        top_p=top_p,
        do_sample=True
    )

    # Decode the output and return the summary
    summary = tokenizer.decode(outputs[0], skip_special_tokens=True)

    return summary