"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotebookPlatform = exports.SSOIdentityType = exports.StudioAuthMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_emr_1 = require("aws-cdk-lib/aws-emr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const utils_1 = require("../utils");
const notebook_platform_helpers_1 = require("./notebook-platform-helpers");
const emr_eks_cluster_1 = require("../emr-eks-platform/emr-eks-cluster");
/**
 * Enum to define authentication mode for Amazon EMR Studio
 */
var StudioAuthMode;
(function (StudioAuthMode) {
    StudioAuthMode["IAM"] = "IAM";
    StudioAuthMode["SSO"] = "SSO";
})(StudioAuthMode = exports.StudioAuthMode || (exports.StudioAuthMode = {}));
/**
 * Enum to define the type of identity Type in EMR studio
 */
var SSOIdentityType;
(function (SSOIdentityType) {
    SSOIdentityType["USER"] = "USER";
    SSOIdentityType["GROUP"] = "GROUP";
})(SSOIdentityType = exports.SSOIdentityType || (exports.SSOIdentityType = {}));
/**
 * A CDK construct to create a notebook infrastructure based on Amazon EMR Studio and assign users to it
 *
 * This construct is initialized through a constructor that takes as argument an interface defined in {@link NotebookPlatformProps}
 * The construct has a method to add users {@link addUser} the method take as argument {@link NotebookUserOptions}
 *
 * Resources deployed:
 *
 * * An S3 Bucket used by EMR Studio to store the Jupyter notebooks
 * * A KMS encryption Key used to encrypt an S3 bucket used by EMR Studio to store jupyter notebooks
 * * An EMR Studio service Role as defined here, and allowed to access the S3 bucket and KMS key created above
 * * An EMR Studio User Role as defined here - The policy template which is leveraged is the Basic one from the Amazon EMR Studio documentation
 * * Multiple EMR on EKS Managed Endpoints, each for a user or a group of users
 * * An execution role to be passed to the Managed endpoint from a policy provided by the user
 * * Multiple Session Policies that are used to map an EMR Studio user or group to a set of resources they are allowed to access. These resources are:
 *   * EMR Virtual Cluster - created above
 *   * ManagedEndpoint
 *
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: 'arn:aws:iam::012345678912:role/Admin-Admin',
 *   eksClusterName: 'cluster',
 * });
 *
 * const notebookPlatform = new NotebookPlatform(stack, 'platform-notebook', {
 *   emrEks: emrEks,
 *   eksNamespace: 'platformns',
 *   studioName: 'platform',
 *   studioAuthMode: StudioAuthMode.SSO,
 * });
 *
 * // If the S3 bucket is encrypted, add policy to the key for the role
 * const policy1 = new ManagedPolicy(stack, 'MyPolicy1', {
 *   statements: [
 *     new PolicyStatement({
 *       resources: <BUCKET ARN(s)>,
 *       actions: ['s3:*'],
 *     }),
 *     new PolicyStatement({
 *       resources: [
 *         stack.formatArn({
 *           account: Aws.ACCOUNT_ID,
 *           region: Aws.REGION,
 *           service: 'logs',
 *           resource: '*',
 *           arnFormat: ArnFormat.NO_RESOURCE_NAME,
 *         }),
 *       ],
 *       actions: [
 *         'logs:*',
 *       ],
 *     }),
 *   ],
 * });
 *
 * notebookPlatform.addUser([{
 *   identityName: 'user1',
 *   identityType: SSOIdentityType.USER,
 *   notebookManagedEndpoints: [{
 *     emrOnEksVersion: EmrVersion.V6_9,
 *     executionPolicy: policy1,
 *   }],
 * }]);
 *
 * ```
 */
class NotebookPlatform extends tracked_construct_1.TrackedConstruct {
    /**
     * @public
     * Constructs a new instance of the DataPlatform class
     * @param {Construct} scope the Scope of the AWS CDK Construct
     * @param {string} id the ID of the AWS CDK Construct
     * @param {NotebookPlatformProps} props the DataPlatformNotebooks [properties]{@link NotebookPlatformProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_ENG_PLATFORM_ID,
        };
        super(scope, id, trackedConstructProps);
        this.studioServicePolicy = [];
        this.studioUserPolicy = [];
        this.studioSubnetList = [];
        this.managedEndpointExecutionPolicyArnMapping = new Map();
        this.authMode = props.studioAuthMode;
        this.vcNamespace = props.eksNamespace ? props.eksNamespace : 'default';
        if (props.idpArn !== undefined) {
            this.federatedIdPARN = props.idpArn;
        }
        //Create encryption key to use with cloudwatch loggroup and S3 bucket storing notebooks and
        this.notebookPlatformEncryptionKey = singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey');
        this.emrVirtualClusterName = 'emr-vc-' + utils_1.Utils.stringSanitizer(props.studioName);
        this.emrEks = props.emrEks;
        //Get the list of private subnets in VPC
        this.studioSubnetList = this.emrEks.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        }).subnetIds;
        //Create a virtual cluster a give it a name of 'emr-vc-'+studioName provided by user
        this.emrVirtCluster = this.emrEks.addEmrVirtualCluster(this, {
            createNamespace: true,
            eksNamespace: props.eksNamespace,
            name: utils_1.Utils.stringSanitizer(this.emrVirtualClusterName),
        });
        //Create a security group to be attached to the studio workspaces
        this.workSpaceSecurityGroup = new aws_ec2_1.SecurityGroup(this, `workspaceSecurityGroup-${props.studioName}`, {
            vpc: this.emrEks.eksCluster.vpc,
            securityGroupName: 'workSpaceSecurityGroup-' + props.studioName,
            allowAllOutbound: false,
        });
        //Tag workSpaceSecurityGroup to be used with EMR Studio
        aws_cdk_lib_1.Tags.of(this.workSpaceSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
        //Create a security group to be attached to the engine for EMR
        //This is mandatory for Amazon EMR Studio although we are not using EMR on EC2
        this.engineSecurityGroup = new aws_ec2_1.SecurityGroup(this, `engineSecurityGroup-${props.studioName}`, {
            vpc: this.emrEks.eksCluster.vpc,
            securityGroupName: 'engineSecurityGroup-' + props.studioName,
            allowAllOutbound: false,
        });
        //Tag engineSecurityGroup to be used with EMR Studio
        aws_cdk_lib_1.Tags.of(this.engineSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
        //Create S3 bucket to store EMR Studio workspaces
        //Bucket is kept after destroying the construct
        this.workspacesBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            bucketName: 'workspaces-bucket-' + utils_1.Utils.stringSanitizer(props.studioName),
            encryptionKey: this.notebookPlatformEncryptionKey,
            serverAccessLogsPrefix: `${props.studioName}-workspace`,
        });
        this.notebookPlatformEncryptionKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [new aws_iam_1.ServicePrincipal('elasticmapreduce.amazonaws.com')],
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'kms:Encrypt*',
                'kms:Decrypt*',
                'kms:ReEncrypt*',
                'kms:GenerateDataKey*',
                'kms:Describe*',
            ],
            conditions: {
                ArnEquals: {
                    'aws:s3:arn': `arn:aws:s3:::${'ara-workspaces-bucket-' + aws_cdk_lib_1.Aws.ACCOUNT_ID + '-' + utils_1.Utils.stringSanitizer(props.studioName)}/*`,
                },
            },
            resources: ['*'],
        }), false);
        //Create a Managed policy for Studio service role
        this.studioServicePolicy.push(aws_iam_1.ManagedPolicy.fromManagedPolicyArn(this, `StudioServiceManagedPolicy-${props.studioName}`, notebook_platform_helpers_1.createStudioServiceRolePolicy(this, this.notebookPlatformEncryptionKey.keyArn, this.workspacesBucket.bucketName, props.studioName)));
        //Create a role for the Studio
        this.studioServiceRole = new aws_iam_1.Role(this, `studioServiceRole-${props.studioName}`, {
            assumedBy: new aws_iam_1.ServicePrincipal(NotebookPlatform.STUDIO_PRINCIPAL),
            roleName: 'studioServiceRole+' + utils_1.Utils.stringSanitizer(props.studioName),
            managedPolicies: this.studioServicePolicy,
        });
        // Create an EMR Studio user role only if the user uses SSO as authentication mode
        if (props.studioAuthMode === 'SSO') {
            //Get Managed policy for Studio user role and put it in an array to be assigned to a user role
            this.studioUserPolicy.push(aws_iam_1.ManagedPolicy.fromManagedPolicyArn(this, `StudioUserManagedPolicy-${props.studioName}`, notebook_platform_helpers_1.createStudioUserRolePolicy(this, props.studioName, this.studioServiceRole.roleName)));
            //Create a role for the EMR Studio user, this roles is further restricted by session policy for each user
            this.studioUserRole = new aws_iam_1.Role(this, `studioUserRole-${props.studioName}`, {
                assumedBy: new aws_iam_1.ServicePrincipal(NotebookPlatform.STUDIO_PRINCIPAL),
                roleName: 'studioUserRole+' + utils_1.Utils.stringSanitizer(props.studioName),
                managedPolicies: this.studioUserPolicy,
            });
        }
        // Create the EMR Studio
        this.studioInstance = new aws_emr_1.CfnStudio(this, `Studio-${props.studioName}`, {
            authMode: props.studioAuthMode,
            defaultS3Location: 's3://' + this.workspacesBucket.bucketName + '/',
            engineSecurityGroupId: this.engineSecurityGroup.securityGroupId,
            name: props.studioName,
            serviceRole: this.studioServiceRole.roleArn,
            subnetIds: this.studioSubnetList,
            userRole: this.studioUserRole ? this.studioUserRole.roleArn : undefined,
            vpcId: this.emrEks.eksCluster.vpc.vpcId,
            workspaceSecurityGroupId: this.workSpaceSecurityGroup.securityGroupId,
            idpAuthUrl: props.idpAuthUrl ? props.idpAuthUrl : undefined,
            idpRelayStateParameterName: props.idpRelayStateParameterName ? props.idpRelayStateParameterName : undefined,
        });
        // Set the Studio URL and Studio Id this is used in session Mapping for
        // EMR Studio when {@linkcode addFederatedUsers} or {@linkcode addSSOUsers} are called
        this.studioName = props.studioName;
        //Set the Studio Id to use for SessionMapping
        this.studioId = this.studioInstance.attrStudioId;
        new aws_cdk_lib_1.CfnOutput(this, `URL for EMR Studio: ${this.studioName}`, {
            value: this.studioInstance.attrUrl,
        });
        /*//Return EMR Studio URL as CfnOutput
        if (this.nestedStackParent != undefined) {
          new CfnOutput(this.nestedStackParent, `URL for EMR Studio: ${this.studioName}`, {
            value: this.studioInstance.attrUrl,
          });
        }*/
    }
    /**
     * @public
     * Method to add users, take a list of userDefinition and will create a managed endpoints for each user
     * and create an IAM Policy scoped to the list managed endpoints
     * @param {NotebookUserOptions} userList list of users
     */
    addUser(userList) {
        //Initialize the managedEndpointArns
        //Used to store the arn of managed endpoints after creation for each users
        //This is used to update the IAM policy
        let managedEndpointArns = [];
        //let managedEndpointObjects: Map<string, CustomResource> = new Map <string, CustomResource> ();
        let iamRolePolicy;
        let iamUserList = [];
        //Loop through each user and create its managed endpoint(s) as defined by the user
        for (let user of userList) {
            //For each policy create a role and then pass it to addManageEndpoint to create an endpoint
            user.notebookManagedEndpoints.forEach((notebookManagedEndpoint, index) => {
                if (!this.managedEndpointExecutionPolicyArnMapping.has(notebookManagedEndpoint.managedEndpointName)) {
                    //For each user or group, create a new managedEndpoint
                    //ManagedEndpoint ARN is used to update and scope the session policy of the user or group
                    let emrOnEksVersion = user.notebookManagedEndpoints[index].emrOnEksVersion;
                    let configOverride = user.notebookManagedEndpoints[index].configurationOverrides;
                    let managedEndpoint = this.emrEks.addManagedEndpoint(this, `${this.studioName}${utils_1.Utils.stringSanitizer(notebookManagedEndpoint.managedEndpointName)}`, {
                        managedEndpointName: `${this.studioName}-${notebookManagedEndpoint.managedEndpointName}`,
                        virtualClusterId: this.emrVirtCluster.attrId,
                        executionRole: this.emrEks.createExecutionRole(this, `${user.identityName}${index}`, notebookManagedEndpoint.executionPolicy, this.vcNamespace, `${notebookManagedEndpoint.managedEndpointName}-execRole`),
                        emrOnEksVersion: emrOnEksVersion ? emrOnEksVersion : NotebookPlatform.DEFAULT_EMR_VERSION,
                        configurationOverrides: configOverride ? configOverride : undefined,
                    });
                    managedEndpoint.node.addDependency(this.emrEks);
                    //Get the Security Group of the ManagedEndpoint which is the Engine Security Group
                    let engineSecurityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, `engineSecurityGroup${user.identityName}${index}`, managedEndpoint.getAttString('securityGroup'));
                    aws_cdk_lib_1.Tags.of(engineSecurityGroup).add('for-use-by-analytics-reference-architecture', 'true');
                    let vpcCidrBlock = this.emrEks.eksCluster.vpc.vpcCidrBlock;
                    //Update workspace Security Group to allow outbound traffic on port 18888 toward Engine Security Group
                    this.workSpaceSecurityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(vpcCidrBlock), aws_ec2_1.Port.tcp(18888), 'Allow traffic to EMR');
                    this.engineSecurityGroup?.addIngressRule(aws_ec2_1.Peer.ipv4(vpcCidrBlock), aws_ec2_1.Port.tcp(18888), 'Allow traffic from EMR Studio');
                    this.workSpaceSecurityGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
                    //Tag the Security Group of the ManagedEndpoint to be used with EMR Studio
                    aws_cdk_lib_1.Tags.of(engineSecurityGroup).add('for-use-with-amazon-emr-managed-policies', 'true');
                    //Add the managedendpointArn to @managedEndpointExcutionPolicyArnMapping
                    //This is to avoid the creation an endpoint with the same policy twice
                    //Save resources and reduce the deployment time
                    // TODO check the emr version is the same => to be fixed on a later commit need to solve adding a tuple to a JS map
                    this.managedEndpointExecutionPolicyArnMapping.set(notebookManagedEndpoint.managedEndpointName, managedEndpoint.getAttString('arn'));
                    //Push the managedendpoint arn to be used in to build the policy to attach to it
                    managedEndpointArns.push(managedEndpoint.getAttString('arn'));
                }
                else {
                    managedEndpointArns.push(this.managedEndpointExecutionPolicyArnMapping.get(notebookManagedEndpoint.managedEndpointName));
                }
            });
            if (this.authMode === 'IAM' && this.federatedIdPARN === undefined) {
                //Create the role policy and gets its ARN
                iamRolePolicy = notebook_platform_helpers_1.createIAMRolePolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                let iamUser = user.iamUser ? user.iamUser : aws_iam_1.User.fromUserName(this, `IAMUSER-${user.identityName}`, user.identityName);
                iamRolePolicy.attachToUser(iamUser);
            }
            else if (this.authMode === 'IAM' && this.federatedIdPARN != undefined) {
                //Create the role policy and gets its ARN
                iamRolePolicy = notebook_platform_helpers_1.createIAMRolePolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                notebook_platform_helpers_1.createIAMFederatedRole(this, iamRolePolicy, this.federatedIdPARN, user.identityName, this.studioId);
            }
            else if (this.authMode === 'SSO') {
                //Create the session policy and gets its ARN
                let sessionPolicyArn = notebook_platform_helpers_1.createUserSessionPolicy(this, user, this.studioServiceRole.roleName, managedEndpointArns, this.studioId);
                if (user.identityType == 'USER' || user.identityType == 'GROUP') {
                    //Map a session to user or group
                    new aws_emr_1.CfnStudioSessionMapping(this, 'studioUser' + user.identityName + user.identityName, {
                        identityName: user.identityName,
                        identityType: user.identityType,
                        sessionPolicyArn: sessionPolicyArn,
                        studioId: this.studioId,
                    });
                }
                else {
                    throw new Error(`identityType should be either USER or GROUP not ${user.identityType}`);
                }
            }
        }
        return iamUserList;
    }
}
exports.NotebookPlatform = NotebookPlatform;
_a = JSII_RTTI_SYMBOL_1;
NotebookPlatform[_a] = { fqn: "aws-analytics-reference-architecture.NotebookPlatform", version: "2.7.0" };
NotebookPlatform.DEFAULT_EMR_VERSION = emr_eks_cluster_1.EmrVersion.V6_8;
NotebookPlatform.STUDIO_PRINCIPAL = 'elasticmapreduce.amazonaws.com';
//# sourceMappingURL=data:application/json;base64,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