"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeFormationAdmin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const utils_1 = require("../utils");
/**
 * An AWS Lake Formation administrator with privileges to do all the administration tasks in AWS Lake Formation.
 * The principal is an Amazon IAM user or role and is added/removed to the list of AWS Lake Formation administrator
 * via the Data Lake Settings API.
 * Creation/deleting first retrieves the current list of administrators and then add/remove the principal to this list.
 * These steps are done outside of any transaction. Concurrent modifications between retrieving and updating can lead to inconsistent results.
 */
class LakeFormationAdmin extends constructs_1.Construct {
    /**
     * Construct a new instance of LakeFormationAdmin.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {LakeFormationAdminProps} props the LakeFormationAdminProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.catalogId = props.catalogId || aws_cdk_lib_1.Aws.ACCOUNT_ID;
        this.principal = props.principal;
        // Check if the principal is an Amazon IAM Role or User and extract the arn and name
        const principalArn = this.principal.roleArn ? this.principal.roleArn : this.principal.userArn;
        const principalName = this.principal.roleName ? this.principal.roleName : this.principal.userName;
        const stack = aws_cdk_lib_1.Stack.of(this);
        // AWS Lambda function for the AWS CDK Custom Resource responsible for creating the AWS Lake Formation tag
        const createLfAdminFn = new pre_bundled_function_1.PreBundledFunction(this, 'lfAdminCreateFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'lake-formation/resources/lambdas/admin',
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        createLfAdminFn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    service: 'lakeformation',
                    resource: 'catalog',
                    resourceName: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
            actions: [
                'lakeformation:PutDataLakeSettings',
                'lakeformation:GetDataLakeSettings',
            ],
        }));
        // AWS CDK Custom Resource Provider for creating the AWS Lake Formation tag
        const createLfAdminCrp = new custom_resources_1.Provider(this, 'lfAdminCreateCrp', {
            onEventHandler: createLfAdminFn,
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        new aws_cdk_lib_1.CustomResource(this, 'lfAdminCreateCr', {
            serviceToken: createLfAdminCrp.serviceToken,
            properties: {
                CatalogId: this.catalogId,
                PrincipalArn: principalArn,
                PrincipalName: principalName,
            },
        });
    }
    /**
    * Adds the CDK execution role to LF admins
    * It requires default cdk bootstrap
    */
    static addCdkExecRole(scope, name) {
        // get the CDK execution role
        const cdkRole = utils_1.Utils.getCdkExecRole(scope, name);
        // make the CDK execution role a Lake Formation admin
        return new LakeFormationAdmin(scope, name, {
            principal: cdkRole,
        });
    }
}
exports.LakeFormationAdmin = LakeFormationAdmin;
_a = JSII_RTTI_SYMBOL_1;
LakeFormationAdmin[_a] = { fqn: "aws-analytics-reference-architecture.LakeFormationAdmin", version: "2.7.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFrZS1mb3JtYXRpb24tYWRtaW4uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbGFrZS1mb3JtYXRpb24vbGFrZS1mb3JtYXRpb24tYWRtaW4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsaUNBQWlDO0FBRWpDLDZDQUE4RTtBQUM5RSxpREFBb0U7QUFDcEUsdURBQWlEO0FBQ2pELG1EQUFxRDtBQUNyRCxtRUFBd0Q7QUFDeEQsMkNBQXVDO0FBQ3ZDLHlFQUFvRTtBQUNwRSxvQ0FBaUM7QUFrQmpDOzs7Ozs7R0FNRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsc0JBQVM7SUFvQi9DOzs7Ozs7T0FNRztJQUVILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBOEI7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksaUJBQUcsQ0FBQyxVQUFVLENBQUM7UUFDbkQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBRWpDLG9GQUFvRjtRQUNwRixNQUFNLFlBQVksR0FBSSxJQUFJLENBQUMsU0FBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFFLElBQUksQ0FBQyxTQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUUsSUFBSSxDQUFDLFNBQW1CLENBQUMsT0FBTyxDQUFDO1FBQy9ILE1BQU0sYUFBYSxHQUFJLElBQUksQ0FBQyxTQUFtQixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUUsSUFBSSxDQUFDLFNBQW1CLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBRSxJQUFJLENBQUMsU0FBbUIsQ0FBQyxRQUFRLENBQUM7UUFFbkksTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsMEdBQTBHO1FBQzFHLE1BQU0sZUFBZSxHQUFHLElBQUkseUNBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3RFLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7WUFDM0IsUUFBUSxFQUFFLHdDQUF3QztZQUNsRCxPQUFPLEVBQUUsaUJBQWlCO1lBQzFCLFlBQVksRUFBRSx3QkFBYSxDQUFDLE9BQU87WUFDbkMsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUM5QixDQUFDLENBQUM7UUFFSCxlQUFlLENBQUMsZUFBZSxDQUFDLElBQUkseUJBQWUsQ0FBQztZQUNsRCxTQUFTLEVBQUU7Z0JBQ1QsS0FBSyxDQUFDLFNBQVMsQ0FBQztvQkFDZCxPQUFPLEVBQUUsZUFBZTtvQkFDeEIsUUFBUSxFQUFFLFNBQVM7b0JBQ25CLFlBQVksRUFBRSxpQkFBRyxDQUFDLFVBQVU7b0JBQzVCLFNBQVMsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQjtpQkFDekMsQ0FBQzthQUNIO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLG1DQUFtQztnQkFDbkMsbUNBQW1DO2FBQ3BDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSiwyRUFBMkU7UUFDM0UsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLDJCQUFRLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQzlELGNBQWMsRUFBRSxlQUFlO1lBQy9CLFlBQVksRUFBRSx3QkFBYSxDQUFDLE9BQU87U0FDcEMsQ0FBQyxDQUFDO1FBRUgsK0ZBQStGO1FBQy9GLElBQUksNEJBQWMsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDMUMsWUFBWSxFQUFFLGdCQUFnQixDQUFDLFlBQVk7WUFDM0MsVUFBVSxFQUFFO2dCQUNWLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztnQkFDekIsWUFBWSxFQUFFLFlBQVk7Z0JBQzFCLGFBQWEsRUFBRSxhQUFhO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQTdFRDs7O01BR0U7SUFDSyxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQWdCLEVBQUUsSUFBWTtRQUV6RCw2QkFBNkI7UUFDN0IsTUFBTSxPQUFPLEdBQUcsYUFBSyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFbEQscURBQXFEO1FBQ3JELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQ3pDLFNBQVMsRUFBRSxPQUFPO1NBQ25CLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBZkgsZ0RBZ0ZDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTUlULTBcblxuaW1wb3J0IHsgQXdzLCBTdGFjaywgRHVyYXRpb24sIEFybkZvcm1hdCwgQ3VzdG9tUmVzb3VyY2UgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBJUm9sZSwgSVVzZXIsIFBvbGljeVN0YXRlbWVudCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgUnVudGltZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCB7IFByb3ZpZGVyIH0gZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IFByZUJ1bmRsZWRGdW5jdGlvbiB9IGZyb20gJy4uL2NvbW1vbi9wcmUtYnVuZGxlZC1mdW5jdGlvbic7XG5pbXBvcnQgeyBVdGlscyB9IGZyb20gJy4uL3V0aWxzJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciB0aGUgbGFrZUZvcm1hdGlvbkFkbWluIENvbnN0cnVjdFxuICovXG5cbmV4cG9ydCBpbnRlcmZhY2UgTGFrZUZvcm1hdGlvbkFkbWluUHJvcHMge1xuICAvKipcbiAgKiBUaGUgcHJpbmNpcGFsIHRvIGRlY2xhcmUgYXMgYW4gQVdTIExha2UgRm9ybWF0aW9uIGFkbWluaXN0cmF0b3JcbiAgKi9cbiAgcmVhZG9ubHkgcHJpbmNpcGFsOiBJUm9sZSB8IElVc2VyO1xuICAvKipcbiAgKiBUaGUgY2F0YWxvZyBJRCB0byBjcmVhdGUgdGhlIGFkbWluaXN0cmF0b3IgaW5cbiAgKiBAZGVmYXVsdCAtIFRoZSBhY2NvdW50IElEXG4gICovXG4gIHJlYWRvbmx5IGNhdGFsb2dJZD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBbiBBV1MgTGFrZSBGb3JtYXRpb24gYWRtaW5pc3RyYXRvciB3aXRoIHByaXZpbGVnZXMgdG8gZG8gYWxsIHRoZSBhZG1pbmlzdHJhdGlvbiB0YXNrcyBpbiBBV1MgTGFrZSBGb3JtYXRpb24uXG4gKiBUaGUgcHJpbmNpcGFsIGlzIGFuIEFtYXpvbiBJQU0gdXNlciBvciByb2xlIGFuZCBpcyBhZGRlZC9yZW1vdmVkIHRvIHRoZSBsaXN0IG9mIEFXUyBMYWtlIEZvcm1hdGlvbiBhZG1pbmlzdHJhdG9yXG4gKiB2aWEgdGhlIERhdGEgTGFrZSBTZXR0aW5ncyBBUEkuXG4gKiBDcmVhdGlvbi9kZWxldGluZyBmaXJzdCByZXRyaWV2ZXMgdGhlIGN1cnJlbnQgbGlzdCBvZiBhZG1pbmlzdHJhdG9ycyBhbmQgdGhlbiBhZGQvcmVtb3ZlIHRoZSBwcmluY2lwYWwgdG8gdGhpcyBsaXN0LlxuICogVGhlc2Ugc3RlcHMgYXJlIGRvbmUgb3V0c2lkZSBvZiBhbnkgdHJhbnNhY3Rpb24uIENvbmN1cnJlbnQgbW9kaWZpY2F0aW9ucyBiZXR3ZWVuIHJldHJpZXZpbmcgYW5kIHVwZGF0aW5nIGNhbiBsZWFkIHRvIGluY29uc2lzdGVudCByZXN1bHRzLlxuICovXG5leHBvcnQgY2xhc3MgTGFrZUZvcm1hdGlvbkFkbWluIGV4dGVuZHMgQ29uc3RydWN0IHtcblxuICAvKipcbiAgKiBBZGRzIHRoZSBDREsgZXhlY3V0aW9uIHJvbGUgdG8gTEYgYWRtaW5zXG4gICogSXQgcmVxdWlyZXMgZGVmYXVsdCBjZGsgYm9vdHN0cmFwXG4gICovXG4gIHB1YmxpYyBzdGF0aWMgYWRkQ2RrRXhlY1JvbGUoc2NvcGU6IENvbnN0cnVjdCwgbmFtZTogc3RyaW5nKSB7XG5cbiAgICAvLyBnZXQgdGhlIENESyBleGVjdXRpb24gcm9sZVxuICAgIGNvbnN0IGNka1JvbGUgPSBVdGlscy5nZXRDZGtFeGVjUm9sZShzY29wZSwgbmFtZSk7XG5cbiAgICAvLyBtYWtlIHRoZSBDREsgZXhlY3V0aW9uIHJvbGUgYSBMYWtlIEZvcm1hdGlvbiBhZG1pblxuICAgIHJldHVybiBuZXcgTGFrZUZvcm1hdGlvbkFkbWluKHNjb3BlLCBuYW1lLCB7XG4gICAgICBwcmluY2lwYWw6IGNka1JvbGUsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgY2F0YWxvZ0lkOiBzdHJpbmc7XG4gIHB1YmxpYyBwcmluY2lwYWw6IElSb2xlIHwgSVVzZXI7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhIG5ldyBpbnN0YW5jZSBvZiBMYWtlRm9ybWF0aW9uQWRtaW4uXG4gICAqIEBwYXJhbSB7Q29uc3RydWN0fSBzY29wZSB0aGUgU2NvcGUgb2YgdGhlIENESyBDb25zdHJ1Y3RcbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIHRoZSBJRCBvZiB0aGUgQ0RLIENvbnN0cnVjdFxuICAgKiBAcGFyYW0ge0xha2VGb3JtYXRpb25BZG1pblByb3BzfSBwcm9wcyB0aGUgTGFrZUZvcm1hdGlvbkFkbWluUHJvcHMgcHJvcGVydGllc1xuICAgKiBAYWNjZXNzIHB1YmxpY1xuICAgKi9cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTGFrZUZvcm1hdGlvbkFkbWluUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5jYXRhbG9nSWQgPSBwcm9wcy5jYXRhbG9nSWQgfHwgQXdzLkFDQ09VTlRfSUQ7XG4gICAgdGhpcy5wcmluY2lwYWwgPSBwcm9wcy5wcmluY2lwYWw7XG5cbiAgICAvLyBDaGVjayBpZiB0aGUgcHJpbmNpcGFsIGlzIGFuIEFtYXpvbiBJQU0gUm9sZSBvciBVc2VyIGFuZCBleHRyYWN0IHRoZSBhcm4gYW5kIG5hbWVcbiAgICBjb25zdCBwcmluY2lwYWxBcm4gPSAodGhpcy5wcmluY2lwYWwgYXMgSVJvbGUpLnJvbGVBcm4gPyAodGhpcy5wcmluY2lwYWwgYXMgSVJvbGUpLnJvbGVBcm4gOiAodGhpcy5wcmluY2lwYWwgYXMgSVVzZXIpLnVzZXJBcm47XG4gICAgY29uc3QgcHJpbmNpcGFsTmFtZSA9ICh0aGlzLnByaW5jaXBhbCBhcyBJUm9sZSkucm9sZU5hbWUgPyAodGhpcy5wcmluY2lwYWwgYXMgSVJvbGUpLnJvbGVOYW1lIDogKHRoaXMucHJpbmNpcGFsIGFzIElVc2VyKS51c2VyTmFtZTtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICAvLyBBV1MgTGFtYmRhIGZ1bmN0aW9uIGZvciB0aGUgQVdTIENESyBDdXN0b20gUmVzb3VyY2UgcmVzcG9uc2libGUgZm9yIGNyZWF0aW5nIHRoZSBBV1MgTGFrZSBGb3JtYXRpb24gdGFnXG4gICAgY29uc3QgY3JlYXRlTGZBZG1pbkZuID0gbmV3IFByZUJ1bmRsZWRGdW5jdGlvbih0aGlzLCAnbGZBZG1pbkNyZWF0ZUZuJywge1xuICAgICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM185LFxuICAgICAgY29kZVBhdGg6ICdsYWtlLWZvcm1hdGlvbi9yZXNvdXJjZXMvbGFtYmRhcy9hZG1pbicsXG4gICAgICBoYW5kbGVyOiAnbGFtYmRhLm9uX2V2ZW50JyxcbiAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfREFZLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygyMCksXG4gICAgfSk7XG5cbiAgICBjcmVhdGVMZkFkbWluRm4uYWRkVG9Sb2xlUG9saWN5KG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIHN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgc2VydmljZTogJ2xha2Vmb3JtYXRpb24nLFxuICAgICAgICAgIHJlc291cmNlOiAnY2F0YWxvZycsXG4gICAgICAgICAgcmVzb3VyY2VOYW1lOiBBd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdsYWtlZm9ybWF0aW9uOlB1dERhdGFMYWtlU2V0dGluZ3MnLFxuICAgICAgICAnbGFrZWZvcm1hdGlvbjpHZXREYXRhTGFrZVNldHRpbmdzJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgLy8gQVdTIENESyBDdXN0b20gUmVzb3VyY2UgUHJvdmlkZXIgZm9yIGNyZWF0aW5nIHRoZSBBV1MgTGFrZSBGb3JtYXRpb24gdGFnXG4gICAgY29uc3QgY3JlYXRlTGZBZG1pbkNycCA9IG5ldyBQcm92aWRlcih0aGlzLCAnbGZBZG1pbkNyZWF0ZUNycCcsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBjcmVhdGVMZkFkbWluRm4sXG4gICAgICBsb2dSZXRlbnRpb246IFJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcblxuICAgIC8vIENyZWF0ZSBhbiBBV1MgQ0RLIEN1c3RvbSBSZXNvdXJjZSBmb3Igc3RhcnRpbmcgdGhlIHNvdXJjZSBjcmF3bGVyIGFuZCB3YWl0aW5nIGZvciBjb21wbGV0aW9uXG4gICAgbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdsZkFkbWluQ3JlYXRlQ3InLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IGNyZWF0ZUxmQWRtaW5DcnAuc2VydmljZVRva2VuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBDYXRhbG9nSWQ6IHRoaXMuY2F0YWxvZ0lkLFxuICAgICAgICBQcmluY2lwYWxBcm46IHByaW5jaXBhbEFybixcbiAgICAgICAgUHJpbmNpcGFsTmFtZTogcHJpbmNpcGFsTmFtZSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cbn0iXX0=