"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksCluster = exports.EmrVersion = exports.Autoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const emr_managed_endpoint_1 = require("./emr-managed-endpoint");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const K8sRoleBinding = require("./resources/k8s/rbac/emr-containers-role-binding.json");
const K8sRole = require("./resources/k8s/rbac/emr-containers-role.json");
const emr_eks_cluster_helpers_1 = require("./emr-eks-cluster-helpers");
const singleton_launch_template_1 = require("../singleton-launch-template");
const emr_eks_nodegroup_asg_tag_1 = require("./emr-eks-nodegroup-asg-tag");
const emr_eks_job_template_1 = require("./emr-eks-job-template");
/**
 * The different autoscaler available with EmrEksCluster
 */
var Autoscaler;
(function (Autoscaler) {
    Autoscaler["KARPENTER"] = "KARPENTER";
    Autoscaler["CLUSTER_AUTOSCALER"] = "CLUSTER_AUTOSCALER";
})(Autoscaler = exports.Autoscaler || (exports.Autoscaler = {}));
/**
 * The different EMR versions available on EKS
 */
var EmrVersion;
(function (EmrVersion) {
    EmrVersion["V6_9"] = "emr-6.9.0-latest";
    EmrVersion["V6_8"] = "emr-6.8.0-latest";
    EmrVersion["V6_7"] = "emr-6.7.0-latest";
    EmrVersion["V6_6"] = "emr-6.6.0-latest";
    EmrVersion["V6_5"] = "emr-6.5.0-latest";
    EmrVersion["V6_4"] = "emr-6.4.0-latest";
    EmrVersion["V6_3"] = "emr-6.3.0-latest";
    EmrVersion["V6_2"] = "emr-6.2.0-latest";
    EmrVersion["V5_33"] = "emr-5.33.0-latest";
    EmrVersion["V5_32"] = "emr-5.32.0-latest";
})(EmrVersion = exports.EmrVersion || (exports.EmrVersion = {}));
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
class EmrEksCluster extends tracked_construct_1.TrackedConstruct {
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.EMR_EKS_TRACKING_ID,
        };
        super(scope, id, trackedConstructProps);
        this.clusterName = props.eksClusterName ?? EmrEksCluster.DEFAULT_CLUSTER_NAME;
        //Define EKS cluster logging
        const eksClusterLogging = [
            aws_eks_1.ClusterLoggingTypes.API,
            aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
            aws_eks_1.ClusterLoggingTypes.SCHEDULER,
            aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
            aws_eks_1.ClusterLoggingTypes.AUDIT,
        ];
        //Set the autoscaler mechanism flag
        this.isKarpenter = props.autoscaling == Autoscaler.KARPENTER ? true : false;
        this.defaultNodes = props.defaultNodes == undefined ? true : props.defaultNodes;
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = new aws_iam_1.Role(this, 'ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.nodegroupAsgTagsProviderServiceToken = new emr_eks_nodegroup_asg_tag_1.EmrEksNodegroupAsgTagProvider(this, 'AsgTagProvider', {
            eksClusterName: this.clusterName,
        }).provider.serviceToken;
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.jobTemplateProviderToken = new emr_eks_job_template_1.EmrEksJobTemplateProvider(this, 'jobTemplateProvider').provider.serviceToken;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            this.eksCluster = new aws_eks_1.Cluster(scope, `${this.clusterName}Cluster`, {
                defaultCapacity: 0,
                clusterName: this.clusterName,
                version: props.kubernetesVersion || EmrEksCluster.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer ?? undefined,
            });
            //Create VPC flow log for the EKS VPC
            let eksVpcFlowLogLogGroup = new aws_logs_1.LogGroup(this, 'eksVpcFlowLogLogGroup', {
                logGroupName: `/aws/emr-eks-vpc-flow/${this.clusterName}`,
                encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey'),
                retention: aws_logs_1.RetentionDays.ONE_WEEK,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            //Allow vpc flowlog to access KMS key to encrypt logs
            singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey').addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`)],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                conditions: {
                    ArnLike: {
                        'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:*`,
                    },
                },
                resources: ['*'],
            }));
            //Setup the VPC flow logs
            const iamRoleforFlowLog = new aws_iam_1.Role(this, 'iamRoleforFlowLog', {
                assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
            this.eksCluster.vpc.addFlowLog('eksVpcFlowLog', {
                destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(eksVpcFlowLogLogGroup, iamRoleforFlowLog),
            });
            //Setting up the cluster with the required controller
            emr_eks_cluster_helpers_1.eksClusterSetup(this, this, props.eksAdminRoleArn);
            //Deploy the right autoscaler using the flag set earlier 
            if (this.isKarpenter) {
                this.karpenterChart = emr_eks_cluster_helpers_1.karpenterSetup(this.eksCluster, this.clusterName, this, props.karpenterVersion || EmrEksCluster.DEFAULT_KARPENTER_VERSION);
            }
            else {
                const kubernetesVersion = props.kubernetesVersion ?? EmrEksCluster.DEFAULT_EKS_VERSION;
                emr_eks_cluster_helpers_1.clusterAutoscalerSetup(this.eksCluster, this.clusterName, this, kubernetesVersion);
            }
        }
        else {
            //Initialize with the provided EKS Cluster
            this.eksCluster = props.eksCluster;
        }
        //Check if the user want to use the default nodegroup and
        //Add the default nodegroup configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.CLUSTER_AUTOSCALER) {
            emr_eks_cluster_helpers_1.setDefaultManagedNodeGroups(this);
        }
        //Check if there user want to use the default Karpenter provisioners and
        //Add the defaults pre-configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.KARPENTER) {
            emr_eks_cluster_helpers_1.setDefaultKarpenterProvisioners(this);
        }
        ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: 's3-access-logs' });
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(this.eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        this.eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        this.eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
            awsServiceName: 'emr-containers.amazonaws.com',
        });
        this.eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: ['']
        });
        // Create an Amazon S3 Bucket for default podTemplate assets
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${this.clusterName.toLowerCase()}-emr-eks-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.clusterName}/pod-template`,
        };
        aws_cdk_lib_1.Tags.of(this.assetBucket).add('for-use-with', 'cdk-analytics-reference-architecture');
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
            roleName: 'araS3BucketDeploymentRole',
        });
        // Upload the default podTemplate to the Amazon S3 asset bucket
        this.uploadPodTemplate('defaultPodTemplates', path_1.join(__dirname, 'resources/k8s/pod-template'));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
        this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
        this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
        this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        // Set the custom resource provider service token here to avoid circular dependencies
        this.managedEndpointProviderServiceToken = new emr_managed_endpoint_1.EmrManagedEndpointProvider(this, 'ManagedEndpointProvider', {
            assetBucket: this.assetBucket,
        }).provider.serviceToken;
        // Provide the podTemplate location on Amazon S3
        new aws_cdk_lib_1.CfnOutput(this, 'podTemplateLocation', {
            description: 'Use podTemplates in Amazon EMR jobs from this Amazon S3 Location',
            value: this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}`),
        });
    }
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = props.eksClusterName || EmrEksCluster.DEFAULT_CLUSTER_NAME;
        let emrEksCluster;
        if (stack.node.tryFindChild(id) == undefined) {
            emrEksCluster = new EmrEksCluster(stack, id, props);
        }
        return stack.node.tryFindChild(id) || emrEksCluster;
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        const regex = /^[a-z0-9]+$/g;
        if (!eksNamespace.match(regex)) {
            throw new Error(`Namespace provided violates the constraints of Namespace naming ${eksNamespace}`);
        }
        const ns = options.createNamespace
            ? this.eksCluster.addManifest(`${options.name}Namespace`, {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: { name: eksNamespace },
            })
            : null;
        // deep clone the Role template object and replace the namespace
        const k8sRole = JSON.parse(JSON.stringify(K8sRole));
        k8sRole.metadata.namespace = eksNamespace;
        const role = this.eksCluster.addManifest(`${options.name}Role`, k8sRole);
        role.node.addDependency(this.emrServiceRole);
        if (ns)
            role.node.addDependency(ns);
        // deep clone the Role Binding template object and replace the namespace
        const k8sRoleBinding = JSON.parse(JSON.stringify(K8sRoleBinding));
        k8sRoleBinding.metadata.namespace = eksNamespace;
        const roleBinding = this.eksCluster.addManifest(`${options.name}RoleBinding`, k8sRoleBinding);
        roleBinding.node.addDependency(role);
        const virtCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace || 'default' } },
            },
            tags: [{
                    key: 'for-use-with',
                    value: 'cdk-analytics-reference-architecture',
                }],
        });
        virtCluster.node.addDependency(roleBinding);
        virtCluster.node.addDependency(this.emrServiceRole);
        if (ns)
            virtCluster.node.addDependency(ns);
        aws_cdk_lib_1.Tags.of(virtCluster).add('for-use-with', 'cdk-analytics-reference-architecture');
        return virtCluster;
    }
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope, id, options) {
        if (options.managedEndpointName.length > 64) {
            throw new Error(`error managed endpoint name length is greater than 64 ${id}`);
        }
        if (this.notebookDefaultConfig == undefined) {
            throw new Error('error empty configuration override is not supported on non-default nodegroups');
        }
        let jsonConfigurationOverrides;
        // TODO this need to be broadended to all possible emr configuration
        // try {
        //   //Check if the configOverride provided by user is valid
        //   let isConfigOverrideValid: boolean = validateSchema(JSON.stringify(configOverrideSchema), options.configurationOverrides);
        //   jsonConfigurationOverrides = isConfigOverrideValid ? options.configurationOverrides : this.notebookDefaultConfig;
        // } catch (error) {
        //   throw new Error(`The configuration override is not valid JSON : ${options.configurationOverrides}`);
        // }
        jsonConfigurationOverrides = options.configurationOverrides ? options.configurationOverrides : this.notebookDefaultConfig;
        // Create custom resource with async waiter until the Amazon EMR Managed Endpoint is created
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.managedEndpointProviderServiceToken,
            properties: {
                clusterId: options.virtualClusterId,
                executionRoleArn: options.executionRole.roleArn,
                endpointName: options.managedEndpointName,
                releaseLabel: options.emrOnEksVersion || EmrEksCluster.DEFAULT_EMR_VERSION,
                configurationOverrides: jsonConfigurationOverrides,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
   * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
   * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
   * If NVME local storage is used, the user_data is modified.
   * @param {string} id the CDK ID of the resource
   * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
   */
    addEmrEksNodegroup(id, props) {
        if (this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        // Get the subnet from Properties or one private subnet for each AZ
        const subnetList = props.subnet ? [props.subnet] : this.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        }).subnets;
        // Add Amazon SSM agent to the user data
        var userData = [
            'yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
            'systemctl enable amazon-ssm-agent',
            'systemctl start amazon-ssm-agent',
        ];
        var launchTemplateName = `EmrEksLaunch-${this.clusterName}`;
        // If the Nodegroup uses NVMe, add user data to configure them
        if (props.mountNvme) {
            userData = userData.concat([
                'INSTANCE_TYPE=$(ec2-metadata -t)',
                'if [[ $INSTANCE_TYPE == *"2xlarge"* ]]; then',
                'DEVICE="/dev/nvme1n1"',
                'mkfs.ext4 $DEVICE',
                'else',
                'yum install -y mdadm',
                'SSD_NVME_DEVICE_LIST=("/dev/nvme1n1" "/dev/nvme2n1")',
                'SSD_NVME_DEVICE_COUNT=${#SSD_NVME_DEVICE_LIST[@]}',
                'RAID_DEVICE=${RAID_DEVICE:-/dev/md0}',
                'RAID_CHUNK_SIZE=${RAID_CHUNK_SIZE:-512}  # Kilo Bytes',
                'FILESYSTEM_BLOCK_SIZE=${FILESYSTEM_BLOCK_SIZE:-4096}  # Bytes',
                'STRIDE=$((RAID_CHUNK_SIZE * 1024 / FILESYSTEM_BLOCK_SIZE))',
                'STRIPE_WIDTH=$((SSD_NVME_DEVICE_COUNT * STRIDE))',
                'mdadm --create --verbose "$RAID_DEVICE" --level=0 -c "${RAID_CHUNK_SIZE}" --raid-devices=${#SSD_NVME_DEVICE_LIST[@]} "${SSD_NVME_DEVICE_LIST[@]}"',
                'while [ -n "$(mdadm --detail "$RAID_DEVICE" | grep -ioE \'State :.*resyncing\')" ]; do',
                'echo "Raid is resyncing.."',
                'sleep 1',
                'done',
                'echo "Raid0 device $RAID_DEVICE has been created with disks ${SSD_NVME_DEVICE_LIST[*]}"',
                'mkfs.ext4 -m 0 -b "$FILESYSTEM_BLOCK_SIZE" -E "stride=$STRIDE,stripe-width=$STRIPE_WIDTH" "$RAID_DEVICE"',
                'DEVICE=$RAID_DEVICE',
                'fi',
                'systemctl stop docker',
                'mkdir -p /var/lib/kubelet/pods',
                'mount $DEVICE /var/lib/kubelet/pods',
                'chmod 750 /var/lib/docker',
                'systemctl start docker',
            ]);
            launchTemplateName = `EmrEksNvmeLaunch-${this.clusterName}`;
        }
        // Add headers and footers to user data
        const userDataMime = aws_cdk_lib_1.Fn.base64(`MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
${userData.join('\r\n')}

--==MYBOUNDARY==--\\
`);
        // Create a new LaunchTemplate or reuse existing one
        const lt = singleton_launch_template_1.SingletonCfnLaunchTemplate.getOrCreate(this, launchTemplateName, userDataMime);
        // Create one Amazon EKS Nodegroup per subnet
        subnetList.forEach((subnet, index) => {
            // Make the ID unique across AZ using the index of subnet in the subnet list
            const resourceId = `${id}-${index}`;
            const nodegroupName = props.nodegroupName ? `${props.nodegroupName}-${index}` : resourceId;
            // Add the user data to the NodegroupOptions
            const nodeGroupParameters = {
                ...props,
                ...{
                    launchTemplateSpec: {
                        id: lt.ref,
                        version: lt.attrLatestVersionNumber,
                    },
                    subnets: {
                        subnets: [subnet],
                    },
                    nodegroupName: nodegroupName,
                },
            };
            // Create the Amazon EKS Nodegroup
            this.addNodegroupCapacity(resourceId, nodeGroupParameters);
        });
    }
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId, options) {
        const nodegroup = this.eksCluster.addNodegroupCapacity(nodegroupId, options);
        // Adding the Amazon SSM policy
        nodegroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // Add tags for the Cluster Autoscaler IAM scoping
        aws_cdk_lib_1.Tags.of(nodegroup).add('eks:cluster-name', `${this.clusterName}`);
        // Add tags for the Cluster Autoscaler management
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/enabled', 'true', {
            applyToLaunchedInstances: true,
        });
        // Add tag for the AZ
        if (options.subnets && options.subnets.subnets) {
            aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/topology.kubernetes.io/zone', options.subnets.subnets[0].availabilityZone, {
                applyToLaunchedInstances: true,
            });
        }
        // Add tag for the lifecycle type (spot or on-demand)
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/eks.amazonaws.com/capacityType', (options.capacityType == aws_eks_1.CapacityType.SPOT) ? 'SPOT' : 'ON_DEMAND', {
            applyToLaunchedInstances: true,
        });
        // Iterate over labels and add appropriate tags
        if (options.labels) {
            for (const [key, value] of Object.entries(options.labels)) {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/label/${key}`, value, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Label${key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/label/${key}`,
                        tagValue: value,
                    },
                }).node.addDependency(nodegroup);
            }
        }
        // Iterate over taints and add appropriate tags
        if (options.taints) {
            options.taints.forEach((taint) => {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`, `${taint.value}:${taint.effect}`, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Taint${taint.key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`,
                        tagValue: `${taint.value}:${taint.effect}`,
                    },
                }).node.addDependency(nodegroup);
            });
        }
        return nodegroup;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, namespace, name) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(this, `${id}irsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + namespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: {
                PodTemplateAccess: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                's3:getObject',
                            ],
                            resources: [
                                stack.formatArn({
                                    region: '',
                                    account: '',
                                    service: 's3',
                                    resource: this.podTemplateLocation.bucketName,
                                    resourceName: `${this.podTemplateLocation.objectKey}/*`,
                                }),
                            ],
                        }),
                    ],
                }),
            },
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
        });
    }
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope, id, options) {
        // Create custom resource to execute the create job template boto3 call
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.jobTemplateProviderToken,
            properties: {
                name: options.name,
                jobTemplateData: options.jobTemplateData,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        if (!this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
exports.EmrEksCluster = EmrEksCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrEksCluster[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksCluster", version: "2.7.0" };
EmrEksCluster.DEFAULT_EMR_VERSION = EmrVersion.V6_8;
EmrEksCluster.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_21;
EmrEksCluster.DEFAULT_CLUSTER_NAME = 'data-platform';
EmrEksCluster.DEFAULT_KARPENTER_VERSION = 'v0.20.0';
//# sourceMappingURL=data:application/json;base64,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