"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomainTbacWorkflow = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
/**
 * This CDK Construct creates a TBAC workflow for Producer/Consumer account.
 * It is based on an AWS Step Functions state machine. It has the following steps:
 * * grants permissions to data domain LF tag to workflow role
 * * creates tables in database that is shared via LF tag
 * * creates Resource-Link(s) for database
 *
 * This Step Functions state machine is invoked from the Central Gov. account via EventBridge Event Bus.
 * It is initiatated in {@link DataDomain}, but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * const workflowRole = new Role(stack, 'myWorkflowRole', {
 *  assumedBy: ...
 * });
 *
 * new DataDomainTbacWorkflow(this, 'DataDomainWorkflow', {
 *  domainName: 'departmentName',
 *  eventBus: eventBus,
 *  workflowRole: workflowRole,
 *  centralAccountId: '1234567891011',
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
class DataDomainTbacWorkflow extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomainTbacWorkflow.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainWorkflowProps} props the DataDomainWorkflowProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Task to create a resource-link for shared table from central gov accunt
        const createDbResourceLink = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createDbResourceLink', {
            service: 'glue',
            action: 'createDatabase',
            iamResources: ['*'],
            parameters: {
                'DatabaseInput': {
                    'Name.$': "States.Format('rl-{}', $.detail.central_database_name)",
                    'TargetDatabase': {
                        'CatalogId': props.centralAccountId,
                        'DatabaseName.$': '$.detail.central_database_name',
                    },
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Trigger crawler workflow
        const triggerCrawler = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'triggerCrawler', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'database_name': aws_stepfunctions_1.JsonPath.stringAt("$.detail.database_name"),
                        'central_database_name': aws_stepfunctions_1.JsonPath.stringAt("$.detail.central_database_name"),
                        'table_names': aws_stepfunctions_1.JsonPath.stringAt("$.detail.table_names"),
                        'central_account_id': aws_stepfunctions_1.JsonPath.stringAt("$.detail.central_account_id"),
                        'lf_access_mode': aws_stepfunctions_1.JsonPath.stringAt("$.detail.lf_access_mode"),
                    }),
                    detailType: 'triggerCrawler',
                    eventBus: props.eventBus,
                    source: 'com.central.stepfunction',
                }]
        });
        // Trigger crawler workflow
        const grantCrawlerPermission = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'grantCrawlerPermission', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'central_database_name': aws_stepfunctions_1.JsonPath.stringAt("$.detail.central_database_name"),
                    }),
                    detailType: 'grantCrawlerPermission',
                    eventBus: props.eventBus,
                    source: 'com.central.stepfunction',
                }],
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Task to check if this account is data product owner (producer)
        const thisAccountIsProducer = new aws_stepfunctions_1.Choice(this, 'thisAccountIsProducer')
            .when(aws_stepfunctions_1.Condition.stringEquals('$.detail.producer_acc_id', aws_cdk_lib_1.Aws.ACCOUNT_ID), grantCrawlerPermission)
            .otherwise(new aws_stepfunctions_1.Pass(this, 'finishWorkflow'));
        grantCrawlerPermission.next(triggerCrawler);
        // Skip if Database resource link already exist
        createDbResourceLink.addCatch(thisAccountIsProducer, {
            errors: ['Glue.AlreadyExistsException'],
            resultPath: '$.Exception',
        }).next(thisAccountIsProducer);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'Workflow', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/tbac-workflow',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // State Machine workflow to create a resource link for TBAC database
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'tbacStateMachine', {
            definition: createDbResourceLink,
            role: props.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
    }
}
exports.DataDomainTbacWorkflow = DataDomainTbacWorkflow;
//# sourceMappingURL=data:application/json;base64,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