"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CentralGovernance = exports.LfAccessControlMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
;
const constructs_1 = require("constructs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const targets = require("aws-cdk-lib/aws-events-targets");
const lakeformation = require("aws-cdk-lib/aws-lakeformation");
const data_mesh_workflow_role_1 = require("./data-mesh-workflow-role");
const lake_formation_1 = require("../lake-formation");
const lake_formation_2 = require("../lake-formation");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const data_domain_1 = require("./data-domain");
/**
 * Enum to define access control mode in Lake Formation
 */
var LfAccessControlMode;
(function (LfAccessControlMode) {
    LfAccessControlMode["NRAC"] = "nrac";
    LfAccessControlMode["TBAC"] = "tbac";
})(LfAccessControlMode = exports.LfAccessControlMode || (exports.LfAccessControlMode = {}));
/**
 * This CDK Construct creates a Data Product registration workflow and resources for the Central Governance account.
 * It uses AWS Step Functions state machine to orchestrate the workflow:
 * * creates tables in AWS Glue Data Catalog
 * * shares tables to Data Product owner account (Producer)
 *
 * This construct also creates an Amazon EventBridge Event Bus to enable communication with Data Domain accounts (Producer/Consumer).
 *
 * This construct requires to use the default [CDK qualifier](https://docs.aws.amazon.com/cdk/v2/guide/bootstrapping.html) generated with the standard CDK bootstrap stack.
 * It ensures the right CDK execution role is used and granted Lake Formation administrator permissions so CDK can create Glue databases when registring a DataDomain.
 *
 * To register a DataDomain, the following information are required:
 * * The account Id of the DataDomain
 * * The secret ARN for the domain configuration available as a CloudFormation output when creating a {@link DataDomain}
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { CentralGovernance, LfTag } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'CentralGovStack');
 *
 * const tags: LfTag[] = [{key: 'tag1': values:['LfTagValue1', 'LfTagValue2']}]
 * const governance = new CentralGovernance(stack, 'myCentralGov', { tags });
 *
 * governance.registerDataDomain('Domain1', 'domain1Name', <DOMAIN_CONFIG_SECRET_ARN>);
 * ```
 */
class CentralGovernance extends constructs_1.Construct {
    /**
     * Construct a new instance of CentralGovernance.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CentralGovernanceProps} props the CentralGovernance properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lfTags = [];
        // Event Bridge event bus for the Central Governance account
        this.eventBus = new aws_events_1.EventBus(this, 'centralEventBus', {
            eventBusName: CentralGovernance.CENTRAL_BUS_NAME,
        });
        this.eventBus.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        this.cdkLfAdmin = lake_formation_2.LakeFormationAdmin.addCdkExecRole(scope, 'CdkLfAdmin');
        // Create LF tags in Central Governance account
        if (props) {
            if (props.lfTags) {
                this.lfTags = props.lfTags;
                this.lfTags.forEach(tag => new lakeformation.CfnTag(this, `CentralLfTag${tag.key}`, {
                    tagKey: tag.key,
                    tagValues: tag.values,
                }).node.addDependency(this.cdkLfAdmin));
            }
        }
        // Workflow role used by the state machine
        this.workflowRole = new data_mesh_workflow_role_1.DataMeshWorkflowRole(this, 'WorkflowRole').role;
        this.workflowRole.attachInlinePolicy(new aws_iam_1.Policy(this, 'sendEvents', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['events:Put*'],
                    resources: [this.eventBus.eventBusArn],
                }),
            ],
        }));
        // Task to create a table
        const createTable = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createTable', {
            service: 'glue',
            action: 'createTable',
            iamResources: ['*'],
            parameters: {
                'DatabaseName.$': `States.Format('{}-${CentralGovernance.DOMAIN_DATABASE_PREFIX}-{}', $.lf_access_mode, $.producer_acc_id)`,
                'TableInput': {
                    'Name.$': '$.tables.name',
                    'Owner.$': '$.producer_acc_id',
                    'StorageDescriptor': {
                        'Location.$': '$.tables.location'
                    }
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Grant SUPER permissions (and grantable) on product database and tables to Data Domain account
        const grantTablePermissions = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantTablePermissionsToProducer', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    "ALL"
                ],
                'PermissionsWithGrantOption': [
                    'ALL'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier.$': '$.producer_acc_id'
                },
                'Resource': {
                    'Table': {
                        'DatabaseName.$': `States.Format('{}-${CentralGovernance.DOMAIN_DATABASE_PREFIX}-{}', $.lf_access_mode ,$.producer_acc_id)`,
                        'Name.$': '$.tables.name',
                    },
                },
            },
            outputPath: '$.tables.name',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        // Trigger workflow in Data Domain account via Event Bridge
        const triggerProducer = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'triggerCreateResourceLinks', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'central_database_name': aws_stepfunctions_1.JsonPath.format("{}-{}-{}", aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"), CentralGovernance.DOMAIN_DATABASE_PREFIX, aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id")),
                        'central_account_id': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        'producer_acc_id': aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id"),
                        'database_name': aws_stepfunctions_1.JsonPath.format("{}-{}", aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"), CentralGovernance.DOMAIN_DATABASE_PREFIX),
                        'table_names': aws_stepfunctions_1.JsonPath.stringAt("$.map_result.flatten"),
                        'lf_access_mode': aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"),
                    }),
                    detailType: aws_stepfunctions_1.JsonPath.format("{}_createResourceLinks", aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id")),
                    eventBus: this.eventBus,
                    source: 'com.central.stepfunction'
                }]
        });
        // Iterate over multiple tables in parallel
        const tablesMapTask = new aws_stepfunctions_1.Map(this, 'forEachTable', {
            itemsPath: '$.tables',
            parameters: {
                'producer_acc_id.$': '$.producer_acc_id',
                'tables.$': '$$.Map.Item.Value',
                'lf_access_mode.$': '$.lf_access_mode',
            },
            resultSelector: {
                'flatten.$': '$[*]'
            },
            resultPath: '$.map_result',
        });
        // Check if LF access mode is NRAC
        const checkModeTask = new aws_stepfunctions_1.Choice(this, 'isModeNRAC')
            .when(aws_stepfunctions_1.Condition.stringEquals('$.lf_access_mode', LfAccessControlMode.NRAC), grantTablePermissions)
            .otherwise(new aws_stepfunctions_1.Pass(this, 'Pass', {
            outputPath: '$.tables.name',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        }));
        tablesMapTask.iterator(createTable.addCatch(checkModeTask, {
            errors: ['Glue.AlreadyExistsException'],
            resultPath: '$.CreateTableException',
        }).next(checkModeTask));
        tablesMapTask.next(triggerProducer);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'centralGov-stateMachine', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/workflow',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // State machine to register data product from Data Domain
        new aws_stepfunctions_1.StateMachine(this, 'RegisterDataProduct', {
            definition: tablesMapTask,
            role: this.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
    }
    /**
     * Registers a new Data Domain account in Central Governance account.
     * Each Data Domain account {@link DataDomain} has to be registered in Central Gov. account before it can participate in a mesh.
     *
     * It creates:
     * * A cross-account policy for Amazon EventBridge Event Bus to enable Data Domain to send events to Central Gov. account
     * * A Lake Formation data access role scoped down to the data domain products bucket
     * * A Glue Catalog Database to hold Data Products for this Data Domain
     * * A Rule to forward events to target Data Domain account.
     *
     * Object references are passed from the DataDomain account to the CentralGovernance account via a AWS Secret Manager secret and cross account access.
     * It includes the following JSON object:
     * ```json
     * {
     *   BucketName: 'clean-<ACCOUNT_ID>-<REGION>',
     *   Prefix: 'data-products',
     *   KmsKeyId: '<KMS_ID>,
     * }
     * ```
     *
     * @param {string} id the ID of the CDK Construct
     * @param {string} domainId the account ID of the DataDomain to register
     * @param {string} domainName the name of the DataDomain, i.e. Line of Business name
     * @param {string} domainSecretArn the full ARN of the secret used by producers to share references with the central governance
     * @param {LfAccessControlMode} lfAccessControlMode Lake Formation Access Control mode for the DataDomain
     * @access public
     */
    registerDataDomain(id, domainId, domainName, domainSecretArn, lfAccessControlMode) {
        // Import the data domain secret from it's full ARN
        const domainSecret = aws_secretsmanager_1.Secret.fromSecretCompleteArn(this, `${id}DomainSecret`, domainSecretArn);
        // Extract data domain references
        const domainBucket = domainSecret.secretValueFromJson('BucketName').unsafeUnwrap();
        const domainPrefix = domainSecret.secretValueFromJson('Prefix').unsafeUnwrap();
        const domainKey = domainSecret.secretValueFromJson('KmsKeyId').unsafeUnwrap();
        // Construct domain event bus ARN
        const dataDomainBusArn = `arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${domainId}:event-bus/${data_domain_1.DataDomain.DOMAIN_BUS_NAME}`;
        const lfModes = lfAccessControlMode ? [lfAccessControlMode] : [LfAccessControlMode.NRAC, LfAccessControlMode.TBAC];
        lfModes.forEach(mode => {
            // Create the database in Glue with datadomain mode+prefix+bucket, and metadata parameters
            new aws_cdk_lib_1.aws_glue.CfnDatabase(this, `${id}DataDomainDatabase-${mode}`, {
                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                databaseInput: {
                    description: `Database for data products in ${domainName} data domain. Account id: ${domainId}. LF Access Control mode: ${mode}`,
                    name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    locationUri: `s3://${domainBucket}/${domainPrefix}`,
                    parameters: {
                        'data_owner': domainId,
                        'data_owner_name': domainName,
                        'pii_flag': false,
                        'access_mode': mode,
                    }
                }
            }).node.addDependency(this.cdkLfAdmin);
            // Grant workflow role permissions to domain database
            new lakeformation.CfnPrincipalPermissions(this, `${id}WorkflowRoleDbAccess-${mode}`, {
                permissions: ['ALL'],
                permissionsWithGrantOption: [],
                principal: {
                    dataLakePrincipalIdentifier: this.workflowRole.roleArn,
                },
                resource: {
                    database: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}DataDomainDatabase-${mode}`));
        });
        // register the S3 location in Lake Formation and create data access role
        new lake_formation_1.LakeFormationS3Location(this, `${id}LFLocation`, {
            s3Location: {
                bucketName: domainBucket,
                objectKey: domainPrefix,
            },
            accountId: domainId,
            kmsKeyId: domainKey,
        });
        if (lfModes.includes(LfAccessControlMode.TBAC)) {
            const mode = LfAccessControlMode.TBAC;
            // Create LF tag for data domain
            new lakeformation.CfnTag(this, `${id}LfTag`, {
                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                tagValues: [domainName],
            }).node.addDependency(this.node.findChild(`${id}DataDomainDatabase-${mode}`));
            // Associate LF tag with data domain database
            new lakeformation.CfnTagAssociation(this, `${id}DbTagAssoc`, {
                resource: {
                    database: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    }
                },
                lfTags: [
                    {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                        tagValues: [domainName],
                    }
                ],
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // share data domain tag with domainId account
            new lakeformation.CfnPrincipalPermissions(this, `grantDataDomainTag`, {
                permissions: ['ASSOCIATE'],
                permissionsWithGrantOption: ['ASSOCIATE'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTag: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                        tagValues: [domainName]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // create LF tag policy for table resource
            new lakeformation.CfnPrincipalPermissions(this, `LFPolicyTable`, {
                permissions: ['ALL'],
                permissionsWithGrantOption: ['ALL'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTagPolicy: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceType: 'TABLE',
                        expression: [{
                                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                                tagValues: [domainName]
                            }]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // create LF tag policy for database resource
            new lakeformation.CfnPrincipalPermissions(this, `LFPolicyDatabase`, {
                permissions: ['CREATE_TABLE', 'DESCRIBE'],
                permissionsWithGrantOption: ['CREATE_TABLE', 'DESCRIBE'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTagPolicy: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceType: 'DATABASE',
                        expression: [{
                                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                                tagValues: [domainName]
                            }]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            if (this.lfTags) {
                var shareTagsDependency = new constructs_1.DependencyGroup();
                this.lfTags.forEach(tag => shareTagsDependency.add(this.node.findChild(`CentralLfTag${tag.key}`)));
                // Share all tags with domainId account
                this.lfTags.forEach(tag => {
                    new lakeformation.CfnPrincipalPermissions(this, `grantDataDomainTag${tag.key}`, {
                        permissions: ['ASSOCIATE'],
                        permissionsWithGrantOption: ['ASSOCIATE'],
                        principal: {
                            dataLakePrincipalIdentifier: domainId,
                        },
                        resource: {
                            lfTag: {
                                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                                tagKey: tag.key,
                                tagValues: tag.values
                            }
                        },
                    }).node.addDependency(shareTagsDependency);
                });
            }
        }
        // Cross-account policy to allow Data Domain account to send events to Central Gov. account event bus
        new aws_events_1.CfnEventBusPolicy(this, `${domainName}Policy`, {
            eventBusName: this.eventBus.eventBusName,
            statementId: `AllowDataDomainAccToPutEvents_${domainId}`,
            action: 'events:PutEvents',
            principal: domainId,
        });
        // Event Bridge Rule to trigger createResourceLinks workflow in target Data Domain account
        const rule = new aws_events_1.Rule(this, `${id}Rule`, {
            eventPattern: {
                source: ['com.central.stepfunction'],
                detailType: [`${domainId}_createResourceLinks`],
            },
            eventBus: this.eventBus,
        });
        rule.addTarget(new targets.EventBus(aws_events_1.EventBus.fromEventBusArn(this, `${id}DomainEventBus`, dataDomainBusArn)));
        rule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
    }
}
exports.CentralGovernance = CentralGovernance;
_a = JSII_RTTI_SYMBOL_1;
CentralGovernance[_a] = { fqn: "aws-analytics-reference-architecture.CentralGovernance", version: "2.7.0" };
CentralGovernance.CENTRAL_BUS_NAME = 'central-mesh-bus';
CentralGovernance.DOMAIN_DATABASE_PREFIX = 'data-domain';
CentralGovernance.DOMAIN_TAG_KEY = 'LoB';
//# sourceMappingURL=data:application/json;base64,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