"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreparedDataset = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * PreparedDataset is used by the [BatchReplayer]{@link BatchReplayer} to generate data in different targets.
 *
 * One of the startDatetime or offset parameter needs to be passed to the constructor:
 *  * StartDatetime is used for prepared datasets provided by the Analytics Reference Architecture because they are known during synthetize time.
 *  * Offset is used when a PreparedDataset is created from a CustomDataset because the startDatetime is not known during synthetize time.
 *
 * A PreparedDataset has following properties:
 *
 * 1. Data is partitioned by timestamp (a range in seconds). Each folder stores data within a given range.
 * There is no constraint on how long the timestamp range can be. But each file must not be larger than 100MB.
 * Creating new PreparedDataset requires to find the right balance between number of partitions and the amount of data read by each BatchReplayer (micro-)batch
 * The available PreparedDatasets have a timestamp range that fit the total dataset time range (see each dataset documentation below) to avoid having too many partitions.
 *
 * Here is an example:
 *
 * |- time_range_start=16000000000
 *
 *    |- file1.csv 100MB
 *
 *    |- file2.csv 50MB
 *
 * |- time_range_start=16000000300 // 5 minute range (300 sec)
 *
 *    |- file1.csv 1MB
 *
 * |- time_range_start=16000000600
 *
 *    |- file1.csv 100MB
 *
 *    |- file2.csv 100MB
 *
 *    |- whichever-file-name-is-fine-as-we-have-manifest-files.csv 50MB
 *
 * 2. It has a manifest CSV file with two columns: start and path. Start is the timestamp
 *
 * start        , path
 *
 * 16000000000  , s3://<path>/<to>/<folder>/time_range_start=16000000000/file1.csv
 *
 * 16000000000  , s3://<path>/<to>/<folder>/time_range_start=16000000000/file2.csv
 *
 * 16000000300  , s3://<path>/<to>/<folder>/time_range_start=16000000300/file1.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/file1.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/file2.csv
 *
 * 16000000600  , s3://<path>/<to>/<folder>/time_range_start=16000000600/whichever-file....csv
 *
 * If the stack is deployed in another region than eu-west-1, data transfer costs will apply.
 * The pre-defined PreparedDataset access is recharged to the consumer via Amazon S3 Requester Pay feature.
 */
class PreparedDataset {
    /**
     * Constructs a new instance of the Dataset class
     * @param {DatasetProps} props the DatasetProps
     * @access public
     */
    constructor(props) {
        if (props.startDatetime === undefined && props.offset === undefined) {
            throw new Error('[PreparedDataset] One of startDatetime or offset parameter must be passed');
        }
        if (props.startDatetime !== undefined && props.offset !== undefined) {
            throw new Error('[PreparedDataset] Only one of startDatetime or offset parameter must be passed');
        }
        this.startDateTime = props.startDatetime;
        this.offset = props.startDatetime ? PreparedDataset.getOffset(props.startDatetime) : props.offset;
        this.location = props.location;
        this.tableName = this.sqlTable();
        this.manifestLocation = props.manifestLocation;
        this.dateTimeColumnToFilter = props.dateTimeColumnToFilter;
        this.dateTimeColumnsToAdjust = props.dateTimeColumnsToAdjust;
    }
    /**
     * Calculate the offset in Seconds from the startDatetime value and the current datetime
     * @param {string} startDatetime the min datetime value used as reference for offset
     * @access private
     */
    static getOffset(startDatetime) {
        const now = new Date().getTime();
        const minDatetime = new Date(startDatetime).getTime();
        return Math.floor((now - minDatetime) / 1000).toString();
    }
    /**
     * Extract the last part of the object key (with / delimiter) and replace '-' with '_' for SQL compatibility
     * @access private
     */
    sqlTable() {
        const parsedPrefix = this.location.objectKey.split('/');
        const re = /\-/gi;
        return parsedPrefix[parsedPrefix.length - 1].replace(re, '_');
    }
}
exports.PreparedDataset = PreparedDataset;
_a = JSII_RTTI_SYMBOL_1;
PreparedDataset[_a] = { fqn: "aws-analytics-reference-architecture.PreparedDataset", version: "2.7.0" };
/**
 * The bucket name of the AWS Analytics Reference Architecture datasets.
 * Data transfer costs will aply if the stack is deployed in another region than eu-west-1.
 * The pre-defined PreparedDataset access is recharged to the consumer via Amazon S3 Requester Pay feature.
 */
PreparedDataset.DATASETS_BUCKET = 'aws-analytics-reference-architecture';
/**
 * The web sale dataset part of 1GB retail datasets.
 * The time range is one week from min(sale_datetime) to max(sale_datetime)
 *
 * | Column name           | Column type | Example                  |
 * |-----------------------|-------------|--------------------------|
 * | item_id               | bigint      | 3935                     |
 * | order_id              | bigint      | 81837                    |
 * | quantity              | bigint      | 65                       |
 * | wholesale_cost        | double      | 32.98                    |
 * | list_price            | double      | 47.82                    |
 * | sales_price           | double      | 36.34                    |
 * | ext_discount_amt      | double      | 2828.8                   |
 * | ext_sales_price       | double      | 2362.1                   |
 * | ext_wholesale_cost    | double      | 2143.7                   |
 * | ext_list_price        | double      | 3108.3                   |
 * | ext_tax               | double      | 0.0                      |
 * | coupon_amt            | double      | 209.62                   |
 * | ext_ship_cost         | double      | 372.45                   |
 * | net_paid              | double      | 2152.48                  |
 * | net_paid_inc_tax      | double      | 2152.48                  |
 * | net_paid_inc_ship     | double      | 442.33                   |
 * | net_paid_inc_ship_tax | double      | 442.33                   |
 * | net_profit            | double      | 8.78                     |
 * | bill_customer_id      | string      | AAAAAAAALNLFAAAA         |
 * | ship_customer_id      | string      | AAAAAAAALPPJAAAA         |
 * | warehouse_id          | string      | AAAAAAAABAAAAAAA         |
 * | promo_id              | string      | AAAAAAAAPCAAAAAA         |
 * | ship_delay            | string      | OVERNIGHT                |
 * | ship_mode             | string      | SEA                      |
 * | ship_carrier          | string      | GREAT EASTERN            |
 * | sale_datetime         | string      | 2021-01-06T15:00:19.373Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_WEB_SALE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/web-sale',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/web-sale-manifest.csv',
    },
    dateTimeColumnToFilter: 'sale_datetime',
    dateTimeColumnsToAdjust: ['sale_datetime'],
});
/**
 * The store sale dataset part of 1GB retail datasets.
 * The time range is one week from min(sale_datetime) to max(sale_datetime)
 *
 * | Column name        | Column type | Example                  |
 * |--------------------|-------------|--------------------------|
 * | item_id            | bigint      | 3935                     |
 * | ticket_id          | bigint      | 81837                    |
 * | quantity           | bigint      | 96                       |
 * | wholesale_cost     | double      | 21.15                    |
 * | list_price         | double      | 21.78                    |
 * | sales_price        | double      | 21.18                    |
 * | ext_discount_amt   | double      | 0.0                      |
 * | ext_sales_price    | double      | 2033.28                  |
 * | ext_wholesale_cost | double      | 2030.4                   |
 * | ext_list_price     | double      | 2090.88                  |
 * | ext_tax            | double      | 81.1                     |
 * | coupon_amt         | double      | 0.0                      |
 * | net_paid           | double      | 2033.28                  |
 * | net_paid_inc_tax   | double      | 2114.38                  |
 * | net_profit         | double      | 2.88                     |
 * | customer_id        | string      | AAAAAAAAEOIDAAAA         |
 * | store_id           | string      | AAAAAAAABAAAAAAA         |
 * | promo_id           | string      | AAAAAAAAEEAAAAAA         |
 * | sale_datetime      | string      | 2021-01-04T22:20:04.144Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_STORE_SALE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-sale',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-sale-manifest.csv',
    },
    dateTimeColumnToFilter: 'sale_datetime',
    dateTimeColumnsToAdjust: ['sale_datetime'],
});
/**
 * The customer dataset part of 1GB retail datasets.
 * The time range is one week from min(customer_datetime) to max(customer_datetime)
 *
 * | Column name       	| Column type 	| Example                    	|
 * |-------------------	|-------------	|----------------------------	|
 * | customer_id       	| string      	| AAAAAAAAHCLFOHAA           	|
 * | salutation        	| string      	| Miss                       	|
 * | first_name        	| string      	| Tina                       	|
 * | last_name         	| string      	| Frias                      	|
 * | birth_country     	| string      	| GEORGIA                    	|
 * | email_address     	| string      	| Tina.Frias@jdK4TZ1qJXB.org 	|
 * | birth_date        	| string      	| 1924-06-14                 	|
 * | gender            	| string      	| F                          	|
 * | marital_status    	| string      	| D                          	|
 * | education_status  	| string      	| 2 yr Degree                	|
 * | purchase_estimate 	| bigint      	| 2500                       	|
 * | credit_rating     	| string      	| Low Risk                   	|
 * | buy_potential     	| string      	| 1001-5000                  	|
 * | vehicle_count     	| bigint      	| 1                          	|
 * | lower_bound       	| bigint      	| 170001                     	|
 * | upper_bound       	| bigint      	| 180000                     	|
 * | address_id        	| string      	| AAAAAAAALAFINEAA           	|
 * | customer_datetime 	| string      	| 2021-01-19T08:07:47.140Z   	|
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_CUSTOMER = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-manifest.csv',
    },
    dateTimeColumnToFilter: 'customer_datetime',
    dateTimeColumnsToAdjust: ['customer_datetime'],
});
/**
 * The customer address dataset part of 1GB retail datasets.
 * It can be joined with customer dataset on address_id column.
 * The time range is one week from min(address_datetime) to max(address_datetime)
 *
 * | Column name      | Column type | Example                  |
 * |------------------|-------------|--------------------------|
 * | address_id       | string      | AAAAAAAAINDKAAAA         |
 * | city             | string      | Farmington               |
 * | county           | string      | Greeley County           |
 * | state            | string      | KS                       |
 * | zip              | bigint      | 69145                    |
 * | country          | string      | United States            |
 * | gmt_offset       | double      | -6.0                     |
 * | location_type    | string      | apartment                |
 * | street           | string      | 390 Pine South Boulevard |
 * | address_datetime | string      | 2021-01-03T02:25:52.826Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 *
 */
PreparedDataset.RETAIL_1_GB_CUSTOMER_ADDRESS = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-address',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/customer-address-manifest.csv',
    },
    dateTimeColumnToFilter: 'address_datetime',
    dateTimeColumnsToAdjust: ['address_datetime'],
});
/**
 * The item dataset part of 1GB retail datasets
 * The time range is one week from min(item_datetime) to max(item_datetime)
 *
 * | Column name   | Column type | Example                                        |
 * |---------------|-------------|------------------------------------------------|
 * |       item_id |      bigint |                                          15018 |
 * |     item_desc |      string | Even ready materials tell with a ministers; un |
 * |         brand |      string |                                 scholarmaxi #9 |
 * |         class |      string |                                        fishing |
 * |      category |      string |                                         Sports |
 * |      manufact |      string |                                    eseoughtpri |
 * |          size |      string |                                            N/A |
 * |         color |      string |                                        thistle |
 * |         units |      string |                                         Bundle |
 * |     container |      string |                                        Unknown |
 * |  product_name |      string |                          eingoughtbarantiought |
 * | item_datetime |      string |                       2021-01-01T18:17:56.718Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_ITEM = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/item',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/item-manifest.csv',
    },
    dateTimeColumnToFilter: 'item_datetime',
    dateTimeColumnsToAdjust: ['item_datetime'],
});
/**
 * The promo dataset part of 1GB retail datasets
 * The time range is one week from min(promo_datetime) to max(promo_datetime)
 *
 * | Column name     | Column type | Example                  |
 * |-----------------|-------------|--------------------------|
 * |        promo_id |      string |         AAAAAAAAHIAAAAAA |
 * |            cost |      double |                   1000.0 |
 * | response_target |      bigint |                        1 |
 * |      promo_name |      string |                     anti |
 * |         purpose |      string |                  Unknown |
 * |  start_datetime |      string | 2021-01-01 00:00:35.890Z |
 * |    end_datetime |      string | 2021-01-02 13:16:09.785Z |
 * |  promo_datetime |      string | 2021-01-01 00:00:16.104Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_PROMO = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/promo',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/promo-manifest.csv',
    },
    dateTimeColumnToFilter: 'promo_datetime',
    dateTimeColumnsToAdjust: ['promo_datetime'],
});
/**
 * The store dataset part of 1GB retail datasets
 * The time range is one week from min(store_datetime) to max(store_datetime)
 *
 * | Column name      | Column type | Example                  |
 * |------------------|-------------|--------------------------|
 * |         store_id |      string |         AAAAAAAAKAAAAAAA |
 * |       store_name |      string |                      bar |
 * | number_employees |      bigint |                      219 |
 * |      floor_space |      bigint |                  6505323 |
 * |            hours |      string |                 8AM-12AM |
 * |          manager |      string |             David Trahan |
 * |        market_id |      bigint |                       10 |
 * |   market_manager |      string |      Christopher Maxwell |
 * |             city |      string |                   Midway |
 * |           county |      string |        Williamson County |
 * |            state |      string |                       TN |
 * |              zip |      bigint |                    31904 |
 * |          country |      string |            United States |
 * |       gmt_offset |      double |                     -5.0 |
 * |   tax_percentage |      double |                      0.0 |
 * |           street |      string |            71 Cedar Blvd |
 * |   store_datetime |      string | 2021-01-01T00:00:00.017Z |
 *
 * The BatchReplayer adds two columns ingestion_start and ingestion_end
 */
PreparedDataset.RETAIL_1_GB_STORE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/store-manifest.csv',
    },
    dateTimeColumnToFilter: 'store_datetime',
    dateTimeColumnsToAdjust: ['store_datetime'],
});
/**
 * The store dataset part of 1GB retail datasets
 * The time range is one week from min(warehouse_datetime) to max(warehouse_datetime)
 *
 * | Column name        | Column type | Example                  |
 * |--------------------|-------------|--------------------------|
 * |       warehouse_id |      string |         AAAAAAAAEAAAAAAA |
 * |     warehouse_name |      string |               Operations |
 * |             street |      string |    461 Second Johnson Wy |
 * |               city |      string |                 Fairview |
 * |                zip |      bigint |                    35709 |
 * |             county |      string |        Williamson County |
 * |              state |      string |                       TN |
 * |            country |      string |            United States |
 * |         gmt_offset |      double |                     -5.0 |
 * | warehouse_datetime |      string | 2021-01-01T00:00:00.123Z |
 *
 */
PreparedDataset.RETAIL_1_GB_WAREHOUSE = new PreparedDataset({
    location: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/warehouse',
    },
    startDatetime: '2021-01-01T00:00:00.000Z',
    manifestLocation: {
        bucketName: PreparedDataset.DATASETS_BUCKET,
        objectKey: 'datasets/prepared/retail/1GB/warehouse-manifest.csv',
    },
    dateTimeColumnToFilter: 'warehouse_datetime',
    dateTimeColumnsToAdjust: ['warehouse_datetime'],
});
//# sourceMappingURL=data:application/json;base64,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