import { PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { Function, FunctionProps } from 'aws-cdk-lib/aws-lambda';
import { Construct } from 'constructs';
/**
 * The properties for the PreBundledFunction construct.
 * It extends existing FunctionProps as optional using `Partial`
 * (as we don't require `Code` prop)
 */
export interface PreBundledFunctionProps extends Partial<FunctionProps> {
    codePath: string;
    lambdaPolicyStatements?: PolicyStatement[];
}
/**
 * A Lambda function with prebundled dependencies.
 *
 * It changes of the code path by based on the environment that `cdk synth` is running on.
 *
 * This class is used together with a Projen custom task "copy-resources", and "pip-install".
 * The tasks will ensure that all Python and libraries files are available in "lib" folder,
 * with the same relative path.
 *
 * When this construct is being run in JSII, this file will be in `node_modules` folder
 * (as it's installed as a 3rd party library.) So we need to change reference based on __dirname.
 *
 * Additionally, the Lambda function is using a common Lambda layer with common dependencies when the runtime is Python.
 * Refer to [PreBundledLayer]{@link PreBundledLayer} construct for packaged Python dependencies.
 *
 *  * Usage example:
 * ```typescript
 * import { PreBundledFunction } from 'aws-analytics-reference-architecture';
 *
 * new PreBundledFunction(this, 'PreBundledFunction', {
 *   codePath: 'construct-dir/resources/lambdas/lambda_dir',
 *   lambdaPolicyStatements: findFilePathsFnPolicy,
 *   // you can use any property available in Function CDK Construct including
 *   memorySize: 1024,
 *   runtime: Runtime.PYTHON_3_8,
 *   handler: 'lambda-file-name.handler',
 *   logRetention: RetentionDays.ONE_WEEK,
 *   timeout: Duration.minutes(15),
 * });
 * ```
 */
export declare class PreBundledFunction extends Function {
    /**
     * Constructs a new instance of the PreBundledFunction construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {PreBundledFunctionProps} props the PreBundledFunction [properties]{@link PreBundledFunctionProps}
     */
    constructor(scope: Construct, id: string, props: PreBundledFunctionProps);
}
