"""
Some auxiliary entities not falling into other categories
"""

import collections
import configparser
import contextlib
import copy
import enum
import json
import logging
import os
import subprocess
import sys
import threading
import time
import traceback
import warnings
from typing import Any, List, Mapping, MutableMapping, Tuple, Optional

import stm32pio.core.settings

module_logger = logging.getLogger(__name__)  # this file logger


def _get_version_from_scm() -> str:
    try:
        import setuptools_scm  # setuptools_scm is the dev-only dependency
    except ImportError:
        return 'Portable (not-installed). See git tag'
    else:
        # Calculate the version in real-time from the Git repo state
        return setuptools_scm.get_version(root='../..', relative_to=__file__)

def get_version() -> str:
    """Retrieve the app version as string"""
    if sys.version_info >= (3, 8):
        import importlib.metadata
        try:
            # For modern Python use the package metadata (if we are installed). For this to be available the wheel build
            # should be done with setuptools_scm
            return importlib.metadata.version('stm32pio')
        except importlib.metadata.PackageNotFoundError:
            # stm32pio is not installed (i.e. running from sources)
            return _get_version_from_scm()
    else:
        try:
            # Version is stored in the stm32pio/core/version.py file auto-generated by setuptools_scm tool
            import stm32pio.core.version
        except ImportError:
            # Version file is not available, most likely we are not installed (i.e. running from sources)
            return _get_version_from_scm()
        else:
            return stm32pio.core.version.version


logging_levels = {  # for exposing the levels to the GUI
    logging.getLevelName(logging.CRITICAL): logging.CRITICAL,
    logging.getLevelName(logging.ERROR): logging.ERROR,
    logging.getLevelName(logging.WARNING): logging.WARNING,
    logging.getLevelName(logging.INFO): logging.INFO,
    logging.getLevelName(logging.DEBUG): logging.DEBUG,
    logging.getLevelName(logging.NOTSET): logging.NOTSET
}


def log_current_exception(logger: logging.Logger, show_traceback_threshold_level: int = logging.DEBUG):
    """
    Print format is:

        ExceptionName: message
        [optional] traceback

    We do not explicitly retrieve an exception info via sys.exc_info() as it immediately stores a reference to the
    current Python frame and/or variables causing some possible weird errors (objects are not GC'ed) and memory leaks.
    See https://cosmicpercolator.com/2016/01/13/exception-leaks-in-python-2-and-3/ for more information
    """
    exc_full_str = traceback.format_exc()
    exc_str = exc_full_str.splitlines()[-1]
    if exc_str.startswith('Exception') and not logger.isEnabledFor(show_traceback_threshold_level):
        exc_str = exc_str[len('Exception: '):]  # meaningless information
    exc_tb = ''.join(exc_full_str.splitlines(keepends=True)[:-1])
    logger.error(f'{exc_str}\n{exc_tb}' if logger.isEnabledFor(show_traceback_threshold_level) else exc_str)


class ProjectLoggerAdapter(logging.LoggerAdapter):
    """
    Use this as a logger for every project:

        self.logger = stm32pio.util.ProjectLoggerAdapter(logging.getLogger('some_singleton_projects_logger'),
                                                         { 'project_id': id(self) })

    It will automatically mix in 'project_id' (and any other property) to every LogRecord (whether you supply 'extra' in
    your log call or not)
    """
    def process(self, msg: Any, kwargs: MutableMapping[str, Any]) -> Tuple[Any, MutableMapping[str, Any]]:
        """Inject context data (both from the adapter and the log call)"""
        if 'extra' in kwargs:
            kwargs['extra'].update(self.extra)
        else:
            kwargs['extra'] = self.extra
        return msg, kwargs


# Currently available verbosity levels. Verbosity determines how every LogRecord will be formatted (regardless its
# logging level)
@enum.unique
class Verbosity(enum.IntEnum):
    NORMAL = enum.auto()
    VERBOSE = enum.auto()


# Do not add or remove any information from the message and simply pass it "as-is"
as_is_formatter = logging.Formatter('%(message)s')


class DispatchingFormatter(logging.Formatter):
    """
    The wrapper around the ordinary logging.Formatter allowing to have multiple formatters for different purposes.
    General arguments schema:

        {
            verbosity=Verbosity.NORMAL,
            general={
                Verbosity.NORMAL: logging.Formatter(...)
                Verbosity.VERBOSE: logging.Formatter(...)
                ...
            },
            special={
                'case_1': {
                    Verbosity.NORMAL: logging.Formatter(...)
                    ...
                },
                ...
            }
        }
    """

    # Mapping of logging formatters for "special". Currently, only "from_subprocess" is defined. It's good to hide such
    # implementation details as much as possible though they are still tweakable from the outer code
    special_formatters = {
        'from_subprocess': {
            level: as_is_formatter for level in Verbosity
        }
    }

    def __init__(self, *args, general: Mapping[Verbosity, logging.Formatter] = None,
                 special: Mapping[str, Mapping[Verbosity, logging.Formatter]] = None,
                 verbosity: Verbosity = Verbosity.NORMAL, **kwargs):

        super().__init__(*args, **kwargs)  # will be '%(message)s' if no arguments were given

        self.verbosity = verbosity
        self._warn_was_shown = False

        if general is not None:
            self.general = general
        else:
            warnings.warn("'general' argument for DispatchingFormatter was not provided. It contains formatters for "
                          "all the logging events except special ones and should be a dict with verbosity levels keys "
                          "and logging.Formatter values")
            self.general = {}

        if special is not None:
            self.special = special
        else:
            self.special = DispatchingFormatter.special_formatters  # use defaults


    def find_formatter_for(self, record: logging.LogRecord, verbosity: Verbosity) -> Optional[logging.Formatter]:
        """Determine and return the appropriate formatter"""
        special_formatter = next((self.special[case] for case in self.special.keys() if hasattr(record, case)), None)
        if special_formatter is not None:
            return special_formatter.get(verbosity)
        else:
            return self.general.get(verbosity)


    def format(self, record: logging.LogRecord) -> str:
        """Overridden method"""
        # Allows to specify a verbosity level on the per-record basis, not only globally
        formatter = self.find_formatter_for(record,
                                            record.verbosity if hasattr(record, 'verbosity') else self.verbosity)
        if formatter is not None:
            return formatter.format(record)
        else:
            if not self._warn_was_shown:
                self._warn_was_shown = True
                module_logger.warning("No formatter found, use default one hereinafter")
            return super().format(record)



class LogPipeRC:
    """Small class suitable for passing to the caller when the LogPipe context manager is invoked"""
    value = ''  # string accumulating all incoming messages

    def __init__(self, fd: int):
        self.pipe = fd  # writable half of os.pipe


class LogPipe(threading.Thread, contextlib.AbstractContextManager):
    """
    The thread combined with a context manager to provide a nice way to temporarily redirect something's stream output
    into the logging module. One straightforward application is to suppress subprocess STDOUT and/or STDERR streams and
    wrap them into the logging mechanism as it is now for any other message in your app. Also, store the incoming
    messages in the string for using it after an execution
    """

    def __init__(self, logger: logging.Logger, level: int, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.logger = logger
        self.level = level

        self.fd_read, self.fd_write = os.pipe()  # create 2 ends of the pipe and setup the reading one
        self.pipe_reader = os.fdopen(self.fd_read)

        self.rc = LogPipeRC(self.fd_write)  # RC stands for "remote control"

    def __enter__(self) -> LogPipeRC:
        """
        Activate the thread and return the consuming end of the pipe so the invoking code can use it to feed its
        messages from now on
        """
        self.start()
        return self.rc

    def run(self):
        """
        Routine of the thread, logging everything
        """
        for line in iter(self.pipe_reader.readline, ''):  # stops the iterator when empty string will occur
            self.rc.value += line  # accumulate the string
            self.logger.log(self.level, line.strip('\n'), extra={ 'from_subprocess': True })  # mark the message origin
        self.pipe_reader.close()

    def __exit__(self, exc_type, exc_val, exc_tb):
        """
        The exception will be passed forward, if present, so we don't need to do something with that. The following
        tear-down process will be done anyway
        """
        os.close(self.fd_write)


_pio_boards_cache: List[str] = []
_pio_boards_cache_lifetime: float = 5.0
_pio_boards_fetched_at: float = 0

def get_platformio_boards() -> List[str]:
    """
    Obtain the PlatformIO boards list. As we interested only in STM32 ones, cut off all of the others. Additionally,
    establish a short-time "cache" to prevent the overflooding with requests to subprocess.

    IMPORTANT NOTE: PlatformIO can go to the Internet from time to time when it decides that its cache is out of date.
    So it MAY take a long time to execute.
    """

    global _pio_boards_fetched_at, _pio_boards_cache, _pio_boards_cache_lifetime

    current_time = time.time()
    if len(_pio_boards_cache) == 0 or (current_time - _pio_boards_fetched_at > _pio_boards_cache_lifetime):
        # Windows 7, as usual, correctly works only with shell=True...
        result = subprocess.run(f"{stm32pio.core.settings.config_default['app']['platformio_cmd']} boards "
                                f"--json-output stm32cube", encoding='utf-8', shell=True, stdout=subprocess.PIPE,
                                check=True)
        _pio_boards_cache = [board['id'] for board in json.loads(result.stdout)]
        _pio_boards_fetched_at = current_time

    return copy.copy(_pio_boards_cache)


def cleanup_dict(mapping: Mapping[str, Any]) -> dict:
    """Recursively copy non-empty values to the new dictionary. Return this new dict"""
    cleaned = {}

    for key, value in mapping.items():
        if isinstance(value, collections.abc.Mapping):
            cleaned[key] = cleanup_dict(value)
        elif value is not None and value != '':
            cleaned[key] = value

    return cleaned


def configparser_to_dict(config: configparser.ConfigParser) -> dict:
    """Convert configparser.ConfigParser instance to a dictionary"""
    return {section: {key: value for key, value in config.items(section)} for section in config.sections()}
