"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addLambdaPermission = exports.singletonEventRole = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Obtain the Role for the EventBridge event
 *
 * If a role already exists, it will be returned. This ensures that if multiple
 * events have the same target, they will share a role.
 */
function singletonEventRole(scope, policyStatements) {
    const id = 'EventsRole';
    const existing = scope.node.tryFindChild(id);
    if (existing) {
        return existing;
    }
    const role = new iam.Role(scope, id, {
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    policyStatements.forEach(role.addToPolicy.bind(role));
    return role;
}
exports.singletonEventRole = singletonEventRole;
/**
 * Allows a Lambda function to be called from a rule
 */
function addLambdaPermission(rule, handler) {
    let scope;
    let node = handler.permissionsNode;
    if (rule instanceof core_1.Construct) {
        // Place the Permission resource in the same stack as Rule rather than the Function
        // This is to reduce circular dependency when the lambda handler and the rule are across stacks.
        scope = rule;
        node = rule.node;
    }
    const permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
    if (!node.tryFindChild(permissionId)) {
        handler.addPermission(permissionId, {
            scope,
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('events.amazonaws.com'),
            sourceArn: rule.ruleArn,
        });
    }
}
exports.addLambdaPermission = addLambdaPermission;
//# sourceMappingURL=data:application/json;base64,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