"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsQueue = void 0;
const iam = require("@aws-cdk/aws-iam");
/**
 * Use an SQS Queue as a target for Amazon EventBridge rules.
 *
 * @stability stable
 * @example
 *
 *    // publish to an SQS queue every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.SqsQueue(queue));
 */
class SqsQueue {
    /**
     * @stability stable
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a result from an EventBridge event.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/resource-based-policies-eventbridge.html#sqs-permissions
     * @stability stable
     */
    bind(rule, _id) {
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        }));
        return {
            id: '',
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,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