"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchLogGroup = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
/**
 * Use an AWS CloudWatch LogGroup as an event rule target.
 *
 * @stability stable
 */
class CloudWatchLogGroup {
    /**
     * @stability stable
     */
    constructor(logGroup, props = {}) {
        this.logGroup = logGroup;
        this.props = props;
    }
    /**
     * Returns a RuleTarget that can be used to log an event into a CloudWatch LogGroup.
     *
     * @stability stable
     */
    bind(_rule, _id) {
        // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
        const resourcePolicyId = `EventsLogGroupPolicy${_rule.node.uniqueId}`;
        const logGroupStack = cdk.Stack.of(this.logGroup);
        if (!this.logGroup.node.tryFindChild(resourcePolicyId)) {
            new log_group_resource_policy_1.LogGroupResourcePolicy(logGroupStack, resourcePolicyId, {
                policyStatements: [new iam.PolicyStatement({
                        effect: iam.Effect.ALLOW,
                        actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                        resources: [this.logGroup.logGroupArn],
                        principals: [new iam.ServicePrincipal('events.amazonaws.com')],
                    })],
            });
        }
        return {
            id: '',
            arn: logGroupStack.formatArn({
                service: 'logs',
                resource: 'log-group',
                sep: ':',
                resourceName: this.logGroup.logGroupName,
            }),
            input: this.props.event,
            targetResource: this.logGroup,
        };
    }
}
exports.CloudWatchLogGroup = CloudWatchLogGroup;
//# sourceMappingURL=data:application/json;base64,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