"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStream = void 0;
const iam = require("@aws-cdk/aws-iam");
const util_1 = require("./util");
/**
 * Use a Kinesis Stream as a target for AWS CloudWatch event rules.
 *
 * @stability stable
 * @example
 *
 *    // put to a Kinesis stream every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.KinesisStream(stream));
 */
class KinesisStream {
    /**
     * @stability stable
     */
    constructor(stream, props = {}) {
        this.stream = stream;
        this.props = props;
    }
    /**
     * Returns a RuleTarget that can be used to trigger this Kinesis Stream as a result from a CloudWatch event.
     *
     * @stability stable
     */
    bind(_rule, _id) {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['kinesis:PutRecord', 'kinesis:PutRecords'],
                resources: [this.stream.streamArn],
            })];
        return {
            id: '',
            arn: this.stream.streamArn,
            role: util_1.singletonEventRole(this.stream, policyStatements),
            input: this.props.message,
            targetResource: this.stream,
            kinesisParameters: this.props.partitionKeyPath ? { partitionKeyPath: this.props.partitionKeyPath } : undefined,
        };
    }
}
exports.KinesisStream = KinesisStream;
//# sourceMappingURL=data:application/json;base64,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