"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster.
 *
 * @stability stable
 */
class EcsTask {
    /**
     * @stability stable
     */
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        this.platformVersion = props.platformVersion;
        if (props.role) {
            const role = props.role;
            this.createEventRolePolicyStatements().forEach(role.addToPolicy.bind(role));
            this.role = role;
        }
        else {
            this.role = util_1.singletonEventRole(this.taskDefinition, this.createEventRolePolicyStatements());
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                cdk.Annotations.of(this.taskDefinition).addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events.
     *
     * @stability stable
     */
    bind(_rule, _id) {
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                platformVersion: this.platformVersion,
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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