"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsApi = void 0;
const path = require("path");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const sdk_api_metadata_generated_1 = require("./sdk-api-metadata.generated");
const util_1 = require("./util");
const awsSdkMetadata = sdk_api_metadata_generated_1.metadata;
/**
 * Use an AWS Lambda function that makes API calls as an event rule target.
 *
 * @stability stable
 */
class AwsApi {
    /**
     * @stability stable
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * Returns a RuleTarget that can be used to trigger this AwsApi as a result from an EventBridge event.
     *
     * @stability stable
     */
    bind(rule, id) {
        const handler = new lambda.SingletonFunction(rule, `${rule.node.id}${id}Handler`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'aws-api-handler')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: 'b4cf1abd-4e4f-4bc6-9944-1af7ccd9ec37',
            lambdaPurpose: 'AWS',
        });
        if (this.props.policyStatement) {
            handler.addToRolePolicy(this.props.policyStatement);
        }
        else {
            handler.addToRolePolicy(new iam.PolicyStatement({
                actions: [awsSdkToIamAction(this.props.service, this.props.action)],
                resources: ['*'],
            }));
        }
        // Allow handler to be called from rule
        util_1.addLambdaPermission(rule, handler);
        const input = {
            service: this.props.service,
            action: this.props.action,
            parameters: this.props.parameters,
            catchErrorPattern: this.props.catchErrorPattern,
            apiVersion: this.props.apiVersion,
        };
        return {
            id: '',
            arn: handler.functionArn,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: handler,
        };
    }
}
exports.AwsApi = AwsApi;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = awsSdkMetadata[srv].prefix || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
//# sourceMappingURL=data:application/json;base64,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