#!usr/bin/python3
"""
Convert a haplotypes table generated by SMAP haplotype-sites or SMAP
haplotype-window into pairwise genetic similarity/distance and/or locus information matrixes.
The output matrixes are created in customised, high-quality figures or in standard output
file formats for downstream data analyses
"""
from __future__ import annotations
import sys
from collections import defaultdict
import argparse
import numpy as np
import pandas as pd
from pandas._libs.missing import NAType
import logging
from .haplotype import (INDEX_COLUMNS,
                        REFERENCE_COLUMN_NAME,
                        LOCUS_COLUMN_NAME,
                        HAPLOTYPES_COLUMN_NAME,
                        _Table)
from pathlib import Path
from typing import Iterable, overload, Literal, TextIO, Sequence, Tuple
from itertools import combinations_with_replacement, product, combinations
from multiprocessing import Pool
from functools import partial
from enum import Enum, unique, auto
from .haplotype import CountMatrix, FrequencyMatrix
from pandas.api.types import is_integer_dtype
from pandas.api.extensions import register_series_accessor
from abc import ABC, abstractmethod
import matplotlib.pyplot as plt
import matplotlib as mpl

from .plotting import PLOT_ALL, PLOT_NOTHING, PLOT_SUMMARY, PlotLevel, heatmap
import random
logger = logging.getLogger(__name__)


class SimilairityCoefficient(ABC):
    """Base class for calculating relatedness between samples."""
    @abstractmethod
    def calculate(self,
                  sample1: pd.Series,
                  sample2: pd.Series,
                  bootstrap_info: pd.Series | None = None) -> float | NAType:
        ...

    @classmethod
    @abstractmethod
    def from_command_line_args(cls, parsed_args: argparse.Namespace):
        ...

    def __str__(self):
        return self.__class__.__name__


class LocusInformationCoefficient(SimilairityCoefficient):
    def __init__(self,
                 include_non_shared_loci: bool,
                 locus_information_criterion: Literal['Shared'] | Literal['Unique'],
                 informative_loci_as_proportion: bool,
                 partially_informative_loci: bool) -> None:
        """
        Calculate relatedness between samples based on the level of loci are shared or unique
        between the samples.

        :param include_non_shared_loci: Loci with data in only one out of two samples in
            each comparison are included in calculations."
        :type include_non_shared_loci: bool
        :param locus_information_criterion: Indicates the basis for what defines an 'informative'
            locus: do the loci share haplotypes or does the locus have unique haplotypes.
        :type locus_information_criterion: Literal['Shared'] | Literal['Unique']
        :param informative_loci_as_proportion: Calculate the informative loci as a
            proportion to the total loci.
        :type informative_loci_as_proportion: bool
        :param partially_informative_loci: Include loci in with at least one
            shared/unique haplotype.
        :type partially_informative_loci: bool
        """
        self._locus_information_criterion = locus_information_criterion
        self._informative_loci_as_proportion = informative_loci_as_proportion
        self._partially_informative_loci = partially_informative_loci
        self._include_non_shared_loci = include_non_shared_loci

    @property
    def locus_information_criterion(self) -> Literal['Shared'] | Literal['Unique']:
        return self._locus_information_criterion

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, LocusInformationCoefficient):
            return super().__eq__(other)
        return (self._locus_information_criterion == other._locus_information_criterion and
                self._informative_loci_as_proportion == other._informative_loci_as_proportion and
                self._partially_informative_loci == other._partially_informative_loci and
                self._include_non_shared_loci == other._include_non_shared_loci)

    @classmethod
    def from_command_line_args(cls, parsed_args: argparse.Namespace) -> LocusInformationCoefficient:
        """
        Fetch the options for calculating the locus information from the command-line
        arguments and construct a LocusInformationCoefficient object.
        """
        return cls(parsed_args.include_non_shared_loci,
                   parsed_args.locus_information_criterion,
                   parsed_args.informative_loci_as_proportion,
                   parsed_args.partially_informative_loci)

    def calculate(self, sample1: pd.Series, sample2: pd.Series) -> pd.Index:
        """
        Calculate the locus information data from two samples, based on the level
        of loci that are shared or unique between the samples. Samples are represented
        as pandas Series objects, and must have an identical index. The indices must
        be pandas MultiIndex objects, encoding three levels: the scaffold/chromosoms
        on which the locus is located, a unique id for the locus and a representation
        of the haplotype, unique within a locus. The levels correspond to the first
        three columns of the output generated by SMAP haplotype-sites or SMAP
        haplotype-window.

        :param sample1: haplotype abscence/presence information for the first sample.
        :type sample1: pd.Series
        :param sample2: haplotype abscence/presence information for the second sample.
        :type sample2: pd.Series
        :rtype: pd.Index
        """
        shared_alleles = sample1.smap.shared_alleles(sample2)
        unique_sample1 = sample1.smap.unique_alleles(sample2, include_non_shared_loci=False)
        unique_sample2 = sample2.smap.unique_alleles(sample1, include_non_shared_loci=False)
        # TODO: use enums and when python 3.10 becomes widespread: use new 'match' statement
        if self._locus_information_criterion == 'Shared':
            if self._partially_informative_loci:
                return shared_alleles.unique(level=LOCUS_COLUMN_NAME)
            unique_loci_sample1 = unique_sample1.droplevel(level=HAPLOTYPES_COLUMN_NAME)
            unique_loci_sample2 = unique_sample2.droplevel(level=HAPLOTYPES_COLUMN_NAME)
            return shared_alleles.droplevel(level=HAPLOTYPES_COLUMN_NAME)\
                                 .difference(unique_loci_sample1)\
                                 .difference(unique_loci_sample2)\
                                 .unique(level=LOCUS_COLUMN_NAME)
        elif self._locus_information_criterion == 'Unique':
            unique_in_either = unique_sample1.union(unique_sample2).unique(level=LOCUS_COLUMN_NAME)
            if self._partially_informative_loci:
                return unique_in_either
            return unique_in_either.difference(shared_alleles
                                               .droplevel(level=[REFERENCE_COLUMN_NAME,
                                                                 HAPLOTYPES_COLUMN_NAME]))\
                                   .unique(level=LOCUS_COLUMN_NAME)
        raise ValueError('--locus_information_criterion must be either "Shared" or "Unique".')


class HaplotypeBasedSimilairityCoefficient(SimilairityCoefficient, ABC):
    def __init__(self, include_non_shared_loci: bool = False,
                 calc_distance: bool = False,
                 distance_method: DistanceMethod | None = None) -> None:
        self._include_non_shared_loci = include_non_shared_loci
        if calc_distance and not distance_method:
            raise ValueError("'--distance' requires '--distance_method'.")
        self._distance_method = distance_method
        self._calc_distance = calc_distance
        super().__init__()

    @property
    def include_non_shared_loci(self) -> bool:
        return self._include_non_shared_loci

    @property
    def distance_method(self) -> DistanceMethod | None:
        return self._distance_method

    @property
    def calc_distance(self) -> bool:
        return self._calc_distance

    @classmethod
    def from_command_line_args(cls,
                               parsed_args: argparse.Namespace) -> \
            HaplotypeBasedSimilairityCoefficient:
        return cls(parsed_args.include_non_shared_loci,
                   parsed_args.calc_distance,
                   parsed_args.distance_method)
    @overload  # noqa: E301
    def calculate(self,
                  sample1: pd.Series,
                  sample2: pd.Series,
                  bootstrap_info: pd.Series | None = ...,
                  incremental: Literal[False] = ...) -> float | NAType: ...
    @overload  # noqa: E301
    def calculate(self,
                  sample1: pd.Series,
                  sample2: pd.Series,
                  bootstrap_info: pd.Series | None,
                  incremental: Literal[True]) -> pd.Series: ...
    @overload  # noqa: E301
    def calculate(self,
                  sample1: pd.Series,
                  sample2: pd.Series,
                  *,
                  incremental: Literal[True]) -> pd.Series: ...
    @overload  # noqa: E301
    def calculate(self,
                  sample1: pd.Series,
                  sample2: pd.Series,
                  bootstrap_info: pd.Series | None = ...,
                  incremental: bool = ...) -> pd.Series | float | NAType: ...
    def calculate(self,  # noqa: E301
                  sample1,
                  sample2,
                  bootstrap_info=None,
                  incremental=False):
        similarity = self._similarity(sample1,
                                      sample2,
                                      bootstrap_info=bootstrap_info,
                                      incremental=incremental)
        if incremental:
            similarity.name = 'similarities'
        if self._calc_distance:
            return self._similarity_to_distance(similarity)
        return similarity

    @overload
    @abstractmethod
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info: pd.Series | None = ...,
                    incremental: Literal[False] = ...) -> float | NAType: ...
    @overload  # noqa: E301
    @abstractmethod
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info: pd.Series | None,
                    incremental: Literal[True]) -> pd.Series: ...
    @overload  # noqa: E301
    @abstractmethod
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    *,
                    incremental: Literal[True]) -> pd.Series: ...
    @overload  # noqa: E301
    @abstractmethod
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    *,
                    incremental: bool) -> float | pd.Series | NAType: ...
    @abstractmethod  # noqa: E301
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info: pd.Series | None = None,
                    incremental: bool = False) -> float | pd.Series | NAType:
        ...

    @overload
    def _similarity_to_distance(self,
                                similarity: pd.Series) -> pd.Series: ...
    @overload  # noqa: E301
    def _similarity_to_distance(self,
                                similarity: float) -> float: ...
    @overload  # noqa: E301
    def _similarity_to_distance(self,
                                similarity: NAType) -> NAType: ...
    def _similarity_to_distance(self, similarity):  # noqa: E301
        if self._distance_method is DistanceMethod.EUCLIDEAN:
            return ((1 - similarity)**2)**(1/2)
        return 1 - similarity


class Jaccard(HaplotypeBasedSimilairityCoefficient):
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info=None,
                    incremental=False) -> float | int | NAType:
        shared_alleles = sample1.smap.number_of_shared_alleles(
            sample2, bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample1 = sample1.smap.number_of_unique_alleles(
            sample2, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample2 = sample2.smap.number_of_unique_alleles(
            sample1, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        denominator = (shared_alleles + unique_sample1 + unique_sample2)
        if incremental:
            result = shared_alleles / denominator.replace(0, pd.NA)
            return result
        if shared_alleles > 0 or unique_sample1 > 0 or unique_sample2 > 0:
            logger.debug(f"{sample1.name} and {sample2.name} - shared: {shared_alleles}, "
                         f"unique for {sample1.name}: {unique_sample1}, "
                         f"and unique for {sample2.name}: {unique_sample2}")
            return shared_alleles / denominator
        return pd.NA


class SorensenDice(HaplotypeBasedSimilairityCoefficient):
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info=None,
                    incremental=False) -> float | int | NAType:
        shared_alleles = sample1.smap.number_of_shared_alleles(
            sample2, bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample1 = sample1.smap.number_of_unique_alleles(
            sample2, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample2 = sample2.smap.number_of_unique_alleles(
            sample1, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        denominator = (2 * shared_alleles) + unique_sample1 + unique_sample2
        if incremental:
            result = (2 * shared_alleles) / denominator.replace(0, pd.NA)
            return result
        if denominator > 0:
            return (2 * shared_alleles) / denominator
        return pd.NA


class Ochiai(HaplotypeBasedSimilairityCoefficient):
    def _similarity(self,
                    sample1: pd.Series,
                    sample2: pd.Series,
                    bootstrap_info=None,
                    incremental=False) -> float | int | NAType:
        shared_alleles = sample1.smap.number_of_shared_alleles(
            sample2, bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample1 = sample1.smap.number_of_unique_alleles(
            sample2, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        unique_sample2 = sample2.smap.number_of_unique_alleles(
            sample1, self._include_non_shared_loci,
            bootstrap_info=bootstrap_info, incremental=incremental)
        denominator = ((shared_alleles + unique_sample1) *
                       (shared_alleles + unique_sample2))**(1/2)
        if incremental:
            result = shared_alleles / denominator.replace(0, pd.NA)
            return result
        if denominator > 0:
            return shared_alleles / denominator
        return pd.NA


@unique
class Coefficient(Enum):
    JACCARD = Jaccard
    SORENSEN_DICE = SorensenDice
    OCHIAI = auto()

    def from_command_line_args(self,
                               parsed_args: argparse.Namespace) -> SimilairityCoefficient:
        return self.value.from_command_line_args(parsed_args)


@unique
class DistanceMethod(Enum):
    INVERSED = auto()
    EUCLIDEAN = auto()

    def __str__(self) -> str:
        return self.name.lower()


@register_series_accessor('smap')
class SmapAccessor:
    def __init__(self, ser: pd.Series) -> None:
        self._ser = ser

    def haplotypes_with_counts(self) -> pd.Series:
        not_na = self.haplotypes_not_na()
        not_na_non_zero = not_na.loc[not_na > 0]
        return not_na_non_zero

    def haplotypes_not_na(self) -> pd.Series:
        return self._ser.replace(0, pd.NA).dropna()

    def unique_alleles(self,
                       other: pd.Series,
                       include_non_shared_loci: bool = False) -> pd.Index:
        ser1_with_counts = self.haplotypes_with_counts()
        ser2_with_counts = other.smap.haplotypes_with_counts()
        if include_non_shared_loci:
            return ser1_with_counts.index.difference(ser2_with_counts.index)
        ser1_loci_with_counts = ser1_with_counts.index.droplevel(HAPLOTYPES_COLUMN_NAME)
        ser2_loci_with_counts = ser2_with_counts.index.droplevel(HAPLOTYPES_COLUMN_NAME)
        shared_loci = ser1_loci_with_counts.intersection(ser2_loci_with_counts)
        ser1_with_counts = self._filter_loci(ser1_with_counts, shared_loci)
        ser2_with_counts = self._filter_loci(ser2_with_counts, shared_loci)
        unique_alleles = ser1_with_counts.index.difference(ser2_with_counts.index)
        return unique_alleles

    @staticmethod
    def _filter_loci(ser: pd.Series, loci: pd.Index) -> pd.DataFrame:
        ser_index_only_loci = ser.reset_index(level=HAPLOTYPES_COLUMN_NAME)
        shared_loci = ser_index_only_loci.loc[loci]
        shared_loci_with_haplotypes = shared_loci.set_index(HAPLOTYPES_COLUMN_NAME, append=True)
        return shared_loci_with_haplotypes

    def number_of_unique_alleles(self,
                                 other: pd.Series,
                                 include_non_shared_loci=False,
                                 bootstrap_info: pd.Series | None = None,
                                 incremental=False) -> int:
        unique_alleles = self.unique_alleles(other, include_non_shared_loci)
        if bootstrap_info is not None:
            unique_alleles_loci = unique_alleles.droplevel([REFERENCE_COLUMN_NAME,
                                                            HAPLOTYPES_COLUMN_NAME])
            unique_alleles.repeat(bootstrap_info.loc[unique_alleles_loci])
        if incremental:
            count_matrix = pd.DataFrame(index=self._ser.index, columns=['count']).fillna(0)
            count_matrix.loc[unique_alleles, 'count'] = 1
            incremental_sum = count_matrix.cumsum(skipna=True)
            # TODO: Sort argument from groupby does not seem to work in pandas 1.4.2,
            # so do the sort using reindex
            old_index_order = incremental_sum.index.unique(LOCUS_COLUMN_NAME)
            result = incremental_sum.groupby(LOCUS_COLUMN_NAME, sort=False).last(1)
            result_old_loci_order = result.reindex(old_index_order)['count']
            return result_old_loci_order
        return unique_alleles.size

    def number_of_shared_alleles(self,
                                 other: pd.Series,
                                 bootstrap_info: pd.Series | None = None,
                                 incremental=False) -> int:
        common_haplotypes = self.shared_alleles(other)
        if bootstrap_info is not None:
            common_haplotypes_loci = common_haplotypes.droplevel([REFERENCE_COLUMN_NAME,
                                                                  HAPLOTYPES_COLUMN_NAME])
            common_haplotypes = common_haplotypes.repeat(bootstrap_info.loc[common_haplotypes_loci])
        if incremental:
            count_matrix = pd.DataFrame(index=self._ser.index, columns=['count']).fillna(0)
            count_matrix.loc[common_haplotypes, 'count'] = 1
            incremental_sum = count_matrix.cumsum(skipna=True)
            # TODO: Sort argument from groupby does not seem to work in pandas 1.4.2,
            # so do the sort using reinde
            old_index_order = incremental_sum.index.unique(LOCUS_COLUMN_NAME)
            result = incremental_sum.groupby(LOCUS_COLUMN_NAME, sort=False).last(1)
            result_old_loci_order = result.reindex(old_index_order)['count']
            return result_old_loci_order
        return common_haplotypes.size

    def shared_alleles(self, other: pd.Series) -> pd.Index:
        assert self._ser.index.equals(other.index)
        self_other = self._ser.to_frame()
        self_other[other.name] = other.values
        counts_for_both_haplotypes = (self_other.fillna(0) > 0).all(axis=1)
        return counts_for_both_haplotypes[counts_for_both_haplotypes.eq(True)].index

    def shared_loci(self, other: pd.Series) -> pd.Index:
        assert self._ser.index.equals(other.index)
        self_other = self._ser.to_frame()
        self_other[other.name] = other.values
        counts_in_either_locus = (self_other > 0).groupby(LOCUS_COLUMN_NAME, sort=False)
        counts_for_both_loci = counts_in_either_locus.any().all(axis=1)
        return counts_for_both_loci[counts_for_both_loci.eq(True)].index

    def number_of_shared_loci(self, other: pd.Series, incremental=True) -> int:
        shared_loci = self.shared_loci(other)
        if incremental:
            count_matrix = pd.DataFrame(index=self._ser.index, columns=['count']).fillna(0)
            count_matrix_loci = count_matrix.droplevel([REFERENCE_COLUMN_NAME,
                                                        HAPLOTYPES_COLUMN_NAME])
            count_matrix_loci.loc[shared_loci, 'number_of_shared_loci'] = 1
            # Here, the sort=False does seem to work, so no reindex needed
            # (see other comments about the sort=False in groupby above).
            incremental_sum = count_matrix_loci.cumsum(skipna=True)
            result = incremental_sum.groupby(LOCUS_COLUMN_NAME, sort=False).last(1)
            return result[['number_of_shared_loci']]
        return shared_loci.size


class SimilarityMatrix(_Table):
    def write(self,
              format: Literal['Nexus'] | Literal['Phylip'],
              file_name: Path,
              mask: Literal['Upper'] | Literal['Lower'] | None) -> None:
        format_options = {
            'Nexus': self._write_nexus,
            'Phylip': self._write_phylip,
        }
        format_options[format](file_name, mask)

    @property
    def values(self):
        return self._df.fillna(np.nan).to_numpy("float32", copy=True)

    def _write_nexus(self, file_name: Path,
                     mask: Literal['Upper'] | Literal['Lower'] | None) -> None:
        samples = self._df.columns
        number_of_samples = len(samples)
        file_name_with_ext = str(file_name) + '.nex'
        mask_str = 'lower' if mask == 'Lower' else 'upper' if mask == 'Upper' else 'both'
        with open(file_name_with_ext, 'w+') as open_file:
            open_file.write('#nexus\nBEGIN Taxa;\n'
                            f'DIMENSIONS ntax={number_of_samples};\nTAXLABELS\n')
            open_file.write("\n".join(f"[{i}] '{sample}'"
                            for i, sample in enumerate(samples, start=1)))
            open_file.write(f"\n;\nEND; [Taxa]\nBEGIN Distances;\nDIMENSIONS ntax={len(samples)};"
                            f"\nFORMAT labels=left diagonal triangle={mask_str};\nMATRIX\n")
            open_file.write("\n".join(f"[{i}] '{sample}' {' '.join(self._df[sample].astype(str))}"
                            for i, sample in enumerate(samples, start=1)))
            open_file.write('\n;\nEND; [Distances]\n')

    def _write_phylip(self, file_name: Path, _: Literal['Upper'] | Literal['Lower'] | None) -> None:
        file_name_with_ext = str(file_name) + '.dist'
        self._df.to_csv(file_name_with_ext, sep='\t', na_rep='<NA>')

    def mask(self, which: Literal['Upper'] | Literal['Lower'] | str) -> None:
        shape = self.shape
        assert shape[0] == shape[1]
        nsamples = shape[0]
        mask = np.ones(self.shape, dtype='bool')
        mask[np.triu_indices(nsamples)] = False
        if which == 'Lower':
            self._df.mask(mask, other=pd.NA, inplace=True)
        elif which == 'Upper':
            self._df.mask(~mask, other=pd.NA, inplace=True)
        else:
            raise NotImplementedError("Values for 'which' argument must be "
                                      "either 'Upper' or 'Lower'.")


def filename_from_textio(open_file: TextIO) -> str:
    try:
        return open_file.name
    except AttributeError:
        return repr(open_file)


@overload
def file2list(open_file: TextIO,
              correct_keys: Iterable[str] | None = ...,
              allow_duplicates: bool = ...,
              required_columns: Literal[2] = ...) -> list[tuple[str, str]]: ...
@overload  # noqa: E302
def file2list(open_file: TextIO,
              correct_keys: Iterable[str] | None,
              allow_duplicates: bool,
              required_columns: Literal[1]) -> list[str]: ...
@overload  # noqa: E302
def file2list(open_file: TextIO,
              *,
              allow_duplicates: bool,
              required_columns: Literal[1]) -> list[str]: ...
@overload  # noqa: E302
def file2list(open_file: TextIO,
              correct_keys: Iterable[str] | None,
              *,
              required_columns: Literal[1]) -> list[str]: ...
@overload  # noqa: E302
def file2list(open_file: TextIO,
              *,
              required_columns: Literal[1]) -> list[str]: ...
def file2list(open_file: TextIO,  # noqa: E302
              correct_keys: Iterable[str] | None = None,
              allow_duplicates: bool = False,
              required_columns: int = 2):
    assert required_columns in (1, 2)
    result = list()
    file_name = filename_from_textio(open_file)
    logger.info("Parsing %s.", file_name)
    for line_number, line in enumerate(open_file):
        line_number += 1
        line = line.strip()
        if line:
            split_line = line.split("\t")
            result_key, *other_column = split_line
            if len(other_column) + 1 != required_columns:
                raise ValueError(f"Line {line_number} contains the incorrect number of entries: "
                                 f"expected {required_columns}, found {len(other_column) + 1}.")
            if correct_keys and result_key not in correct_keys:
                raise ValueError(f"Invalid value ({result_key}) on line {line_number} "
                                 f"in file {file_name}.")
            if not allow_duplicates and result_key in result:
                raise ValueError(f"Duplicate entry ({result_key}) on line {line_number} "
                                 f"in file {file_name}")
            result.append((result_key, *other_column) if other_column else result_key)
    return result


def parse_args(args: Sequence[str]) -> argparse.Namespace:
    logger.info('Parsing arguments.')
    parser = get_argument_parser()
    parsed_args = parser.parse_args(args)
    if any([parsed_args.locus_information_criterion,
            parsed_args.informative_loci_as_proportion,
            parsed_args.partially_informative_loci]) \
       and not parsed_args.output_locus_information:
        raise ValueError("Any of the options '--locus_information_criterion', "
                         "'--partial' or '--informative_loci_as_proportion' "
                         "require '--output_locus_information'.")
    parsed_args.similarity_coefficient = \
        parsed_args.similarity_coefficient\
                   .from_command_line_args(parsed_args)

    return parsed_args


def parse_new_sample_names_file(file: TextIO, current_names: Iterable[str]) -> dict[str, str]:
    try:
        result = file2list(file, correct_keys=current_names, required_columns=2)
    except ValueError as err:
        raise ValueError('Error while parsing sample names file. Please check that '
                         'there are no duplicate samples in the first column of the '
                         'file, and that all of these samples are also present in the '
                         'header of the haplotypes table.') from err
    return dict(result)


def create_relatedness_matrix(haplotypes: FrequencyMatrix | CountMatrix,
                              similarity_coefficient: SimilairityCoefficient,
                              bootstrap_info=None,
                              number_of_processes: int = 1) -> SimilarityMatrix:
    '''
    Create a relatedness matrix based on the haplotypes table.
    '''
    samples = haplotypes.sample_names
    # Make similarity matrix by performing pairwise comparisons between all samples.
    similarities = pd.DataFrame(index=samples, columns=samples)
    worker = partial(similarity_coefficient.calculate, bootstrap_info=bootstrap_info)
    sample_combonations = [(sample1, sample2) for (_, sample1), (_, sample2) in
                           combinations_with_replacement(haplotypes.iter_samples(), 2)]
    with Pool(number_of_processes) as mp_pool:
        results = mp_pool.starmap(worker, sample_combonations)
    result_with_names = zip(combinations_with_replacement(haplotypes.sample_names, 2), results)
    for (sample1_name, sample2_name), result in result_with_names:
        similarities.at[[sample1_name], [sample2_name]] = result
        similarities.at[[sample2_name], [sample1_name]] = result
    return SimilarityMatrix(similarities)


def create_information_matrix(haplotypes: FrequencyMatrix | CountMatrix,
                              similarity_coefficient: LocusInformationCoefficient,
                              processes: int = 1) -> \
                              Tuple[SimilarityMatrix, pd.DataFrame, pd.DataFrame]:
    '''
    Create a relatedness matrix based on the haplotypes table.
    '''
    samples = haplotypes.sample_names
    loci = haplotypes.loci_names

    # Initiate empty frames
    completely_unique_or_shared_loci = pd.DataFrame(index=samples, columns=samples)
    number_of_assessed_loci_per_sample_combination = pd.DataFrame(
        0, index=samples, columns=samples)
    counts_content, counts_columns = ([(0, 0)] * len(loci), ['count', 'informative'])
    counts_per_locus = pd.DataFrame(counts_content, index=pd.Index(loci, name='Locus'),
                                    columns=counts_columns)

    sample_combonations = [(sample1, sample2) for (_, sample1), (_, sample2) in
                           combinations_with_replacement(haplotypes.iter_samples(), 2)]
    worker = partial(locus_information_worker, similarity_coefficient)
    with Pool(processes) as mp_pool:
        results = mp_pool.starmap(worker, sample_combonations)

    for (sample1_name, sample2_name), result in \
            zip(combinations_with_replacement(samples, 2), results):
        informative_loci, shared_loci = result
        completely_unique_or_shared_loci.at[sample2_name, sample1_name] = informative_loci.size
        completely_unique_or_shared_loci.at[sample1_name, sample2_name] = informative_loci.size
        number_of_assessed_loci_per_sample_combination.at[sample2_name, sample1_name] = \
            shared_loci.size
        number_of_assessed_loci_per_sample_combination.at[sample1_name, sample2_name] = \
            shared_loci.size

        if sample1_name != sample2_name:
            counts_per_locus.loc[shared_loci, 'count'] += 1
            counts_per_locus.loc[informative_loci, 'informative'] += 1

    with pd.option_context('mode.use_inf_as_na', True):
        proportions = (counts_per_locus['informative'] / counts_per_locus['count']).fillna(0)
    counts_per_locus['proportion'] = proportions

    if similarity_coefficient.locus_information_criterion == "Unique":
        counts_per_locus['samples_with_unique'] = haplotypes.get_identifying_loci()

    return (SimilarityMatrix(completely_unique_or_shared_loci),
            number_of_assessed_loci_per_sample_combination,
            counts_per_locus)


def locus_information_worker(similarity_coefficient: LocusInformationCoefficient,
                             sample1: pd.Series,
                             sample2: pd.Series) -> Tuple[pd.Index, pd.Index]:
    # Calculate the similarity (= informative loci)
    informative_loci = similarity_coefficient.calculate(sample1, sample2)
    # Capture assesed loci
    shared_loci = sample1.smap.shared_loci(sample2)
    return informative_loci, shared_loci


def read_haplotype_table(file_location: Path) -> CountMatrix | FrequencyMatrix:
    # Import the haplotypes table.
    # Setting dtypes globally together with index_columns does not work, see
    # https://github.com/pandas-dev/pandas/pull/44632
    # set_index must be used.
    logger.info('Loading input file.')
    df = pd.read_csv(file_location, sep='\t',
                     parse_dates=False, dtype=str, header=0).set_index(INDEX_COLUMNS)
    convert_options = {"infer_objects": False, "convert_string": False, "convert_integer": True,
                       "convert_boolean": False, "convert_floating": False}
    try:
        df = df.apply(pd.to_numeric, errors='raise')
    except (ValueError, TypeError) as e:
        raise ValueError("Could not read haplotypes table because "
                         "it does not contain numerical data") from e
    return_type = FrequencyMatrix
    if all(df.apply(is_integer_dtype)):
        convert_options["convert_integer"] = True
        return_type = CountMatrix
    convert_options["convert_floating"] = True
    df = df.convert_dtypes(**convert_options)
    return return_type(df)


def pool_samples(haplotypes: CountMatrix | FrequencyMatrix,
                 name_mapping: dict[str, Sequence[str]]) -> None:
    logger.info('Pooling samples.')
    grouped = defaultdict(list)
    for original_sample, new_sample in name_mapping.items():
        grouped[new_sample].append(original_sample)
    to_be_pooled = {new_sample: samples_to_pool for new_sample, samples_to_pool
                    in grouped.items() if len(samples_to_pool) >= 2}
    haplotypes.pool_samples(to_be_pooled)


def create_bootstrap_replicates(haplotypes: CountMatrix | FrequencyMatrix,
                                number_of_replicates: int,
                                similarity_coefficient: HaplotypeBasedSimilairityCoefficient,
                                mask: Literal['Upper'] | Literal['Lower'],
                                matrix_format: Literal['Nexus'] | Literal['Phylip'],
                                processes: int) -> None:
    for i in range(1, number_of_replicates+1):
        logger.info('Creating bootstrap replicate %s', i)
        distance_or_similarity = 'Distances' if similarity_coefficient.calc_distance \
                                 else 'Similarities'
        file_name = Path(f"{distance_or_similarity}_"
                         f"{similarity_coefficient}_bootstrap_{i}")

        loci = haplotypes.loci_names
        bootstrap_info = pd.Series(0, index=loci)

        # Randomly sample loci in the locus_dict.
        for _ in range(len(loci)):
            locus = random.choice(loci)
            bootstrap_info.at[locus] += 1

        matrix = create_relatedness_matrix(haplotypes, similarity_coefficient,
                                           bootstrap_info, processes)
        if mask:
            matrix.mask(mask)
        matrix.write(matrix_format, file_name, mask)


def get_curve_information(haplotypes: CountMatrix | FrequencyMatrix,
                          sample_pairs: list[Tuple[str, str]],
                          similarity_coefficient: HaplotypeBasedSimilairityCoefficient,
                          processes: int = 1) -> dict[str, dict[str, pd.DataFrame]]:
    information = {sample: {} for sample, _ in sample_pairs}
    sample_combinations = [(sample1, sample2) for (_, sample1), (_, sample2) in
                           product(haplotypes.iter_samples(), repeat=2)
                           if (sample1.name, sample2.name) in sample_pairs]
    worker = partial(curve_information_worker, similarity_coefficient)
    with Pool(processes) as mp_pool:
        results = mp_pool.starmap(worker, sample_combinations)
    for (sample1_name, sample2_name), result in zip(sample_pairs, results):
        information[sample1_name][sample2_name] = result
    return information


def curve_information_worker(similarity_coefficient: HaplotypeBasedSimilairityCoefficient,
                             sample1: pd.Series, sample2: pd.Series) -> pd.DataFrame:
    similarities = similarity_coefficient.calculate(sample1, sample2, incremental=True)
    number_of_loci = sample1.smap.number_of_shared_loci(sample2, incremental=True)
    assert number_of_loci.index.equals(similarities.index)
    sample_pair_info = pd.concat([number_of_loci, similarities], axis=1)
    sample_pair_info = sample_pair_info.dropna(how="any")
    return sample_pair_info


def create_line_plots(name: str,
                      x_values: Iterable[np.ndarray],
                      y_values: Iterable[np.ndarray],
                      title: str,
                      xmin: int, xmax: int,
                      ymin: int, ymax: int,
                      xlabel: str, ylabel: str,
                      legend_labels: Iterable[str],
                      plot_type: str) -> None:
    ax = plt.figure().gca()
    plt.title(title)
    plt.xlabel(xlabel)
    plt.ylabel(ylabel)
    ax.xaxis.set_major_locator(mpl.ticker.MaxNLocator(integer=True))
    ax.set_xlim(xmin, xmax)
    ax.set_ylim(ymin, ymax)
    for xvalue, yvalue, legend_label in zip(x_values, y_values, legend_labels):
        plt.plot(xvalue, yvalue, linewidth=1.0, label=legend_label)
    plt.legend(legend_labels, bbox_to_anchor=[1, 1])
    plt.tight_layout()
    plt.savefig(name + f'.{plot_type}', dpi=300, format=plot_type)
    plt.close()


def create_line_curves(haplotypes: CountMatrix | FrequencyMatrix,
                       sample_pairs: Iterable[str, str],
                       similarity_coefficient: HaplotypeBasedSimilairityCoefficient,
                       plot_type: str,
                       processes: int = 1) -> None:
    logger.info('Creating line curves.')
    information_per_sample_pair = get_curve_information(haplotypes,
                                                        sample_pairs,
                                                        similarity_coefficient, processes)
    method = 'similarity' if not similarity_coefficient.calc_distance \
             else 'Inversed distance' \
             if similarity_coefficient.distance_method == DistanceMethod.INVERSED \
             else 'Euclidean distance'
    loci_type = 'shared ' if not similarity_coefficient.include_non_shared_loci else ''

    for sample1_name, other_samples in information_per_sample_pair.items():
        xvalues = [information['number_of_shared_loci']
                   for information in other_samples.values()]
        y_values = [information['similarities']
                    for information in other_samples.values()]
        max_y = np.nanmax([y.max() for y in y_values])
        other_sample_names = other_samples.keys()
        title = f'{similarity_coefficient} {method} - Number of {loci_type}loci\n{sample1_name}'
        filename = f'{similarity_coefficient}_{method.lower().replace(" ", "_")}_{sample1_name}'
        create_line_plots(filename,
                          x_values=xvalues,
                          y_values=y_values,
                          title=title,
                          xmin=0,
                          xmax=haplotypes.number_of_loci,
                          ymin=0,
                          ymax=1 if np.isnan(max_y) else max_y,
                          legend_labels=other_sample_names,
                          xlabel=f'Number of {loci_type}loci',
                          ylabel=f'{similarity_coefficient} {method}',
                          plot_type=plot_type)


def plot_number_of_informative_loci(number_of_loci: SimilarityMatrix,
                                    include_non_shared_loci: bool,
                                    informative_loci_as_proportion: bool,
                                    partially_informative_loci: bool,
                                    annotate: bool,
                                    plot_type: str) -> None:
    logger.info("Creating a heatmap of the number of"
                f"{' shared' if not include_non_shared_loci else ''}"
                " loci in each sample pair.")
    plot_title = (f"{'Number' if not informative_loci_as_proportion else 'Proportion'} "
                  f"of {'Partially' if partially_informative_loci else 'Completely'}"
                  f"{' Shared' if not include_non_shared_loci else ''} Loci.")
    plt.figure(figsize=(11.7, 8.3))  # A4 in inch, horizontal
    heatmap(number_of_loci.values,
            xticklabels=number_of_loci.sample_names,
            yticklabels=number_of_loci.sample_names,
            title=plot_title,
            annotate=annotate)
    plt.savefig(plot_title.replace(" ", '')[:-1] + f'_heatmap.{plot_type}',
                format=plot_type, dpi=300)
    plt.close()


def plot_number_of_loci(number_of_loci: pd.DataFrame,
                        include_non_shared_loci: bool,
                        annotate: bool,
                        plot_type: str) -> None:
    file_name = f"NumberOf{'Shared' if not include_non_shared_loci else ''}Loci_heatmap"
    plt.figure(figsize=(11.7, 8.3))  # A4 in inch, horizontal
    heatmap(number_of_loci.values,
            xticklabels=number_of_loci.columns,
            yticklabels=number_of_loci.columns,
            title=file_name,
            annotate=annotate)
    plt.savefig(f"{file_name}.{plot_type}", format=plot_type)
    plt.close()


def plot_similarity_matrix(similarity_coefficient: HaplotypeBasedSimilairityCoefficient,
                           similarity_matrix: SimilarityMatrix,
                           annotate: bool,
                           plot_type: str,) -> None:
    similarity_or_distance = 'similarity' if not similarity_coefficient.calc_distance \
                             else 'distance'
    logger.info('Creating heatmaps showing the %s %s',
                similarity_coefficient if not similarity_coefficient.calc_distance
                else similarity_coefficient.distance_method,
                similarity_or_distance)

    file_name = (f"{similarity_coefficient}_genetic_{similarity_or_distance}_heatmap")
    plt.figure(figsize=(11.7, 8.3))  # A4 in inch, horizontal
    heatmap(similarity_matrix.values,
            xticklabels=similarity_matrix.sample_names,
            yticklabels=similarity_matrix.sample_names,
            title=file_name,
            annotate=annotate)
    plt.savefig(f"{file_name}.{plot_type}", format=plot_type, dpi=300)
    plt.close()


def get_argument_parser() -> argparse.ArgumentParser:
    # Create an ArgumentParser object
    parser = argparse.ArgumentParser(description="Create a similarity/distance "
                                                 "matrix based on GBS or amplicon "
                                                 "sequencing data.")
    input_and_output = parser.add_argument_group(title="Input and output options")
    input_and_output.add_argument(
        'table',
        type=str,
        help="Name of the filtered haplotypes table retrieved "
             "from SMAP haplotype-sites or SMAP haplotype-window.")
    input_and_output.add_argument(
        "--names",
        type=Path,
        default=None,
        help="Tab-delimited text file defining the order of "
             "(new) sample names in the matrix: first column = old names, "
             "second column (optional) = new names (default = no sample "
             "list, the order of samples in the matrix equals their order "
             "in the haplotypes table).")
    input_and_output.add_argument(
        "--matrix_format",
        choices=["Phylip", "Nexus"],
        default="Phylip",
        help="Format of the distance matrix (default = Phylip).")
    input_and_output.add_argument(
        "--count_informative_regions",
        dest="print_region_information",
        action="store_true",
        help="Create a tab-delimited text file showing for each region "
             "the number of comparisons with at least one shared (Common_allele) "
             "or one unique (Discriminatory_locus) haplotype "
             "(default = output table not created).")
    input_and_output.add_argument(
        "--mask",
        choices=[None, "Upper", "Lower"],
        default=None,
        help="Mask values on the main diagonal of each matrix and "
             "above (upper) or below (lower) the main diagonal "
             "(default = None (no masking), other options are: "
             "upper (mask upper half) and lower (mask lower half).")
    filtering_options = parser.add_argument_group("Filtering options")
    filtering_options.add_argument(
        "-sc", "--sample_completeness",
        default=0,
        type=float,
        help="Minimum number of regions with haplotype data in a sample. "
             "Samples with less data are removed (default = all samples are "
             "included).")
    filtering_options.add_argument(
        "-rc", "--region_completeness",
        default=0,
        type=float,
        help="Minimum proportion of regions with haplotype data in a region. "
             "Regions with less data are removed (default = all regions are "
             "included).")
    filtering_options.add_argument(
        "--regions",
        type=Path,
        default=None,
        help="Tab-delimited text file containing a one-column "
             "list of regions formatted as in the haplotypes "
             "table of SMAP-haplotype (default = no list with "
             "regions provided).")
    distance_measure_options = parser.add_argument_group(
        "Relatedness options"
    )
    distance_measure_options.add_argument(
        "-s", "--similarity_coefficient",
        choices=Coefficient,
        type=Coefficient.__getitem__,
        default=Coefficient.JACCARD,
        help="Coefficient used to express pairwise genetic "
             "similarity between samples (default = JACCARD).")
    distance_measure_options.add_argument(
        "--distance",
        dest="calc_distance",
        action="store_true",
        help="Convert genetic similarity estimates into genetic "
             "distances (default = no conversion to distances).")
    distance_measure_options.add_argument(
        "--distance_method",
        choices=DistanceMethod,
        type=DistanceMethod.__getitem__,
        default=DistanceMethod.INVERSED,
        help="Method used for genetic distance calculations (default = INVERSED).")
    distance_measure_options.add_argument(
        "-b", "--n_bootstrap_replicates",
        default=0,
        type=int,
        help="The number of bootstrap replicates of the "
             "genetic similarity/distance matrix (default = no bootstrap replicates).")
    distance_measure_options.add_argument(
        "--include_non_shared_loci",
        dest="include_non_shared_loci",
        action="store_true",
        help="Loci with data in only one out of two samples in each comparison are "
             "included in genetic similarity and locus information calculations "
             "(default = only loci with data in both samples of each comparison are "
             "included in calculations).")
    locus_information_options = parser.add_argument_group("Locus Information Coefficient options")
    locus_information_options.add_argument(
        "--output_locus_information",
        choices=["Matrix" "List", "All"],
        default=None,
        help="Print locus information to the output directory as a matrix "
             "(Matrix option, .csv file). The locus information per locus "
             "can also be printed as a tab-delimited list (List option, .txt file) "
             "showing the number of comparisons wherein the locus was informative "
             "and the sample IDs with unique haplotypes in all comparisons for each "
             "locus (default = locus information is not printed, other options are: "
             "Matrix, List, and All (both Matrix and List).")
    locus_information_options.add_argument(
        "-lic", "--locus_information_criterion",
        choices=["Shared", "Unique"],
        default=None,
        help="Create a matrix showing the number of loci with shared or unique "
             "haplotypes in each comparison (default = Shared (matrix showing "
             "the number of loci with shared haplotypes), other option is: "
             "Unique (matrix showing the number of loci with unique haplotypes).")
    locus_information_options.add_argument(
        "--informative_loci_as_proportion",
        dest="informative_loci_as_proportion",
        action="store_true",
        help="Express the informative locus count as a proportion to the total loci "
             "(default = locus information content is expressed in absolute numbers).")
    locus_information_options.add_argument(
        "--partially_informative_loci",
        dest="partially_informative_loci",
        action="store_true",
        help="Include loci in locus information matrix with at least one "
             "shared/unique haplotype (default = only loci with only "
             "shared/unique haplotypes are included).")
    resources_group = parser.add_argument_group(title="System resources")
    resources_group.add_argument(
        "-p", "--processes",
        default=1,
        type=int,
        help="Use multiple processes to calculate similarities between samnples")
    plot_group = parser.add_argument_group("Graphical output options")
    plot_group.add_argument(
        "--plot",
        dest="plot",
        type=PlotLevel,
        default=PLOT_NOTHING,
        choices=(PLOT_ALL, PLOT_SUMMARY, PLOT_NOTHING),
        help="Select which plots are to be generated. Choosing 'nothing' "
             "disables plot generation. Passing 'summary' only generates "
             "graphs with information for all samples while 'all' will also "
             "enable generate per-sample plots [default 'summary'].")
    plot_group.add_argument(
        "-t", "--plot_type",
        dest="plot_type",
        choices=["png", "pdf"],
        default="png",
        help="Choose the file type for the plots [png].")
    plot_group.add_argument("--annotate_heatmaps",
                            dest="annotate_heatmaps",
                            default=False,
                            action="store_true",
                            help="Annotate the matrix plots with values (default no annotations).")
    plot_group.add_argument(
        "--line_curve_samples",
        default=None,
        type=Path,
        help="Tab-delimited text file containing a list of sample comparisons"
             "(first column = ID of the first sample, second column = ID of the"
             " second sample) of which line curves need to be plotted "
             "(default = no list specified, all comparisons are plotted if "
             "--plot_type PLOT_ALL.)")
    return parser


def main(args=None) -> None:
    if args is None:
        args = sys.argv

    args = parse_args(args)
    haplotypes = read_haplotype_table(args.table)

    # Rename samples
    if args.names:
        logger.info('Renaming samples.')
        with open(args.names, 'r') as open_names:
            names_mapping = parse_new_sample_names_file(open_names, haplotypes.sample_names)
        new_samples_order = list(dict.fromkeys(names_mapping.values()))
        pool_samples(haplotypes, names_mapping)
        new_sample_names = haplotypes.sample_names
        for old_name, new_name in list(names_mapping.items()):  # list() to make copy
            if new_name in new_sample_names:
                names_mapping.pop(old_name)
        haplotypes.rename_samples(names_mapping)
        # Reorder samples according to order in second column of names file
        haplotypes = haplotypes[new_samples_order]

    line_curve_samples = [(sample1, sample2) for sample1, sample2 in
                          combinations(haplotypes.sample_names, 2)]
    if args.line_curve_samples:
        line_curve_samples = []
        if args.plot < PLOT_ALL:
            raise ValueError("--line_curve_samples provided, but --plot is not set to 'all .")
        with args.line_curve_samples.open('r') as open_curves_file:
            line_curve_list = file2list(open_curves_file,
                                        correct_keys=haplotypes.sample_names,
                                        required_columns=2)
            for sample1, sample2 in line_curve_list:
                # Sample1 already checked in file2list call above.
                if sample2 not in haplotypes.sample_names:
                    raise ValueError("Requested to create line curve for sample combination "
                                     f"'{sample1}' and '{sample2}', but '{sample2}' does not "
                                     "exist in haplotypes table.")
                line_curve_samples.append((sample1, sample2))

    # Remove regions with a data completeness lower than the predefined minimum.
    logger.info('Filtering for locus and sample completeness.')
    haplotypes.filter_for_locus_completeness(args.region_completeness)
    haplotypes.filter_for_sample_completeness(args.sample_completeness)

    # Only create matrices for the selected regions
    regions_list = []
    if args.regions:
        logger.info('Filtering input table to only include regions from --regions.')
        with args.regions.open('r') as open_regions_file:
            regions_list = file2list(open_regions_file, required_columns=1)
            haplotypes.filter_loci(regions_list)

    logger.info('Randomly shuffling loci.')
    haplotypes.shuffle_loci()

    # Create a relatedness matrix based on haplotypes.
    logger.info('Creating relatedness matrix for %s regions in %s samples.',
                haplotypes.number_of_loci, haplotypes.number_of_samples)
    similairy_matrix = create_relatedness_matrix(haplotypes,
                                                 args.similarity_coefficient,
                                                 number_of_processes=args.processes)
    if args.mask:
        logger.info('Masking relatedness matrix.')
        similairy_matrix.mask(args.mask)
    file_name = Path(f"{'Distances' if args.calc_distance else 'Similarities'}_"
                     f"{args.similarity_coefficient}")
    similairy_matrix.write(args.matrix_format, file_name, args.mask)

    if args.plot >= PLOT_SUMMARY:
        plot_similarity_matrix(args.similarity_coefficient, similairy_matrix,
                               args.annotate_heatmaps, args.plot_type)

    if args.output_locus_information:
        logger.info('Gathering locus information.')
        coefficient = LocusInformationCoefficient.from_command_line_args(args)
        informative_loci, number_of_loci, counts_per_locus = create_information_matrix(
            haplotypes, similarity_coefficient=coefficient, processes=args.processes)
        shared_or_not = 'Shared' if not args.include_non_shared_loci else ''
        shared_loci_name = Path(f"NumberOf{shared_or_not}Loci.txt")
        number_of_loci.to_csv(shared_loci_name, sep="\t")
        logger.info('Writing a list with the locus information.')
        number_or_proportion = "Number"
        if args.informative_loci_as_proportion:
            number_or_proportion = "Proportion"
            informative_loci /= number_of_loci
        partial_or_complete = "Partially" if args.partially_informative_loci else "Completely"
        informative_loci_path = (f"{number_or_proportion}Of"
                                 f"{partial_or_complete}"
                                 f"{args.locus_information_criterion}Loci.txt")
        informative_loci.to_csv(informative_loci_path)
        if args.plot >= PLOT_SUMMARY:
            plot_number_of_informative_loci(informative_loci,
                                            args.include_non_shared_loci,
                                            args.informative_loci_as_proportion,
                                            args.partially_informative_loci,
                                            args.annotate_heatmaps,
                                            args.plot_type)

            plot_number_of_loci(number_of_loci,
                                args.include_non_shared_loci,
                                args.annotate_heatmaps,
                                args.plot_type)

        if args.output_locus_information in ('List', 'All'):
            column2_name = f'NumberOfComparisonsWith{args.locus_information_criterion}Locus'
            criterion_column = []
            if args.locus_information_criterion == "Unique":
                criterion_column = ["SamplesWithUniqueHaplotypes"]

            partial_representation = "Partially" if args.partially_informative_loci \
                                     else "Completely"
            counts_per_locus.columns = ['NumberOfComparisons',
                                        column2_name,
                                        ('ProportionOfComparisonsWith'
                                         f'{args.locus_information_criterion}Locus')] + \
                criterion_column
            file_name = f'{partial_representation}{args.locus_information_criterion}Loci.txt'
            counts_per_locus.to_csv(file_name, sep="\t", float_format='%.2f')

    if args.plot >= PLOT_ALL:
        create_line_curves(haplotypes, line_curve_samples,
                           args.similarity_coefficient, args.plot_type, args.processes)

    if args.n_bootstrap_replicates > 0:
        create_bootstrap_replicates(haplotypes,
                                    number_of_replicates=args.n_bootstrap_replicates,
                                    similarity_coefficient=args.similarity_coefficient,
                                    matrix_format=args.matrix_format,
                                    mask=args.mask,
                                    processes=args.processes)

    logger.info('Finished!')


if __name__ == '__main__':
    main()
