import glob
import logging
from typing import List

import yaml


class Moad:
    def __init__(self):
        logging.basicConfig(level=logging.INFO)
        pass

    def mix(self):
        moad_files = []
        files = glob.glob("*.yml")
        for filename in files:
            moad_files.append(MoadFile(filename))

        for file in moad_files:
            try:
                if file.is_openapi():
                    file.mix(moad_files)
                else:
                    print(f"No openapi: {file.name}")
            except Warning as warn:
                logging.warning(warn)


class MoadFile(object):
    def __init__(self, name: str):
        self.name = name
        with open(name, "r", encoding="utf-8") as f:
            self.yaml = yaml.safe_load(f)

    def is_openapi(self):
        return "openapi" in self.yaml

    def mix(self, files: list):
        new_content = ""
        with open(self.name, "r", encoding="utf-8") as f:
            replacing = False
            changed = False
            for line in f:
                if "moad-mixin:begin:" in line:
                    if replacing:
                        raise Warning(f"{self.name}: Found moad-mixin:begin: while already inserting content")

                    command = line.split("moad-mixin:begin:", 2)[1]
                    mixin_file, mixin_path = command.split(":", 2)
                    mixin_content = self.get_mixin_content(files, mixin_file, mixin_path)
                    new_content += line + mixin_content
                    changed = True
                    replacing = True
                elif "moad-mixin:end" in line:
                    if not replacing:
                        raise Warning(f"{self.name}: Found moad-mixin:end without matching begin")
                    new_content += line
                    replacing = False
                elif not replacing:
                    new_content += line
        if replacing:
            raise Warning(f"{self.name}: Reached end of file without finding matching moad-mixin:end")
        if changed:
            logging.info(f"Writing new version of {self.name}.")
            with open(self.name, "w", encoding="utf-8") as f:
                f.write(new_content)

    def get_mixin_content(self, files, mixin_file, mixin_path):
        mixin_path = mixin_path.strip()
        for file in files:
            if file.name == mixin_file:
                content = file.yaml
                indent = 0
                for elem in mixin_path.strip().split("/"):
                    try:
                        content = content[elem]
                    except KeyError:
                        raise Warning(f"{self.name}: Could not replace content: The path {mixin_path} was not found in file {mixin_file}")
                    indent += 1
                indent_str = "  " * indent
                info = f"# WARNING The following was automatically written into this file.\n# It will be replaced by moad with the content of {mixin_file} -> {mixin_path}\n"
                content = info + yaml.dump(content, sort_keys=False)
                content += "# End of autogenerated content\n"

                content = indent_str + content.replace("\n", f"\n{indent_str}").rstrip(" ")
                logging.info(f"Inserting content from {mixin_file}  (Path: <{mixin_path}>) into {self.name}")
                return content


class MoadPathNotFoundError(Exception):
    def __init__(self, file, path):
        self.file = file
        self.path = path


class MoadWithoutEndError(Exception):
    pass
