"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
function testStack() {
    const stack = new cdk.Stack(undefined, undefined, { env: { account: '12345', region: 'us-test-1' } });
    stack.node.setContext('availability-zones:12345:us-test-1', ['us-test-1a', 'us-test-1b']);
    return stack;
}
module.exports = {
    'creating a Cluster also creates 2 DB Instances'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Properties: {
                Engine: 'aurora',
                DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
                MasterUsername: 'admin',
                MasterUserPassword: 'tooshort',
                VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
            },
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: 'Snapshot',
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.countResources('AWS::RDS::DBInstance', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: assert_1.ABSENT,
            UpdateReplacePolicy: assert_1.ABSENT,
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'can create a cluster with a single instance'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: [{ 'Fn::GetAtt': ['DatabaseSecurityGroup5C91FDCB', 'GroupId'] }],
        }));
        test.done();
    },
    'can create a cluster with imported vpc and security group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', {
            vpcId: 'VPC12345',
        });
        const sg = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'SecurityGroupId12345');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                securityGroups: [sg],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora',
            DBSubnetGroupName: { Ref: 'DatabaseSubnets56F17B9A' },
            MasterUsername: 'admin',
            MasterUserPassword: 'tooshort',
            VpcSecurityGroupIds: ['SecurityGroupId12345'],
        }));
        test.done();
    },
    'cluster with parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const group = new lib_1.ParameterGroup(stack, 'Params', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            description: 'bye',
            parameters: {
                param: 'value',
            },
        });
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup: group,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: { Ref: 'ParamsA8366201' },
        }));
        test.done();
    },
    "sets the retention policy of the SubnetGroup to 'Retain' if the Cluster is created with 'Retain'"(test) {
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        new lib_1.DatabaseCluster(stack, 'Cluster', {
            masterUser: { username: 'admin' },
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE),
                vpc,
            },
            removalPolicy: cdk.RemovalPolicy.RETAIN,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBSubnetGroup', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain',
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'creates a secret when master credentials are not specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            MasterUsername: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:username::}}',
                    ],
                ],
            },
            MasterUserPassword: {
                'Fn::Join': [
                    '',
                    [
                        '{{resolve:secretsmanager:',
                        {
                            Ref: 'DatabaseSecret3B817195',
                        },
                        ':SecretString:password::}}',
                    ],
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"admin"}',
            },
        }));
        test.done();
    },
    'create an encrypted cluster with custom KMS key'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            storageEncryptionKey: new kms.Key(stack, 'Key'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            KmsKeyId: {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'cluster with instance parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            parameters: {
                key: 'value',
            },
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                parameterGroup,
                vpc,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB',
            },
        }));
        test.done();
    },
    'create a cluster using a specific version of MySQL'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraMysql({
                version: lib_1.AuroraMysqlEngineVersion.VER_2_04_4,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-mysql',
            EngineVersion: '5.7.mysql_aurora.2.04.4',
        }));
        test.done();
    },
    'create a cluster using a specific version of Postgresql'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraPostgres({
                version: lib_1.AuroraPostgresEngineVersion.VER_10_7,
            }),
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            Engine: 'aurora-postgresql',
            EngineVersion: '10.7',
        }));
        test.done();
    },
    'cluster exposes different read and write endpoints'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.notDeepEqual(stack.resolve(cluster.clusterEndpoint), stack.resolve(cluster.clusterReadEndpoint));
        test.done();
    },
    'imported cluster with imported security group honors allowAllOutbound'(test) {
        // GIVEN
        const stack = testStack();
        const cluster = lib_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                })],
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    'cluster with enabled monitoring'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['DatabaseMonitoringRole576991DA', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'monitoring.rds.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole',
                        ],
                    ],
                },
            ],
        }));
        test.done();
    },
    'create a cluster with imported monitoring role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const monitoringRole = new aws_iam_1.Role(stack, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['MonitoringRole90457BF9', 'Arn'],
            },
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'throws when trying to add rotation to a cluster without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple times'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const cluster = new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            masterUser: { username: 'admin' },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // WHEN
        cluster.addRotationSingleUser();
        // THEN
        test.throws(() => cluster.addRotationSingleUser(), /A single user rotation was already added to this cluster/);
        test.done();
    },
    'create a cluster with s3 import role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 export role'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const associatedRole = new aws_iam_1.Role(stack, 'AssociatedRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: associatedRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'AssociatedRole824CFCD3',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'AssociatedRole824CFCD3',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                            's3:DeleteObject*',
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'Bucket83908E77',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Bucket83908E77',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'create a cluster with s3 import and export buckets and custom parameter group'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const parameterGroup = new lib_1.ParameterGroup(stack, 'ParameterGroup', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            parameters: {
                key: 'value',
            },
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            parameterGroup,
            s3ImportBuckets: [importBucket],
            s3ExportBuckets: [exportBucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ImportRole377BC9C0',
                            'Arn',
                        ],
                    },
                },
                {
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup', {
            Family: 'aurora5.6',
            Parameters: {
                key: 'value',
                aurora_load_from_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ImportRole377BC9C0',
                        'Arn',
                    ],
                },
                aurora_select_into_s3_role: {
                    'Fn::GetAtt': [
                        'DatabaseS3ExportRole9E328562',
                        'Arn',
                    ],
                },
            },
        }));
        test.done();
    },
    'PostgreSQL cluster with s3 export buckets does not generate custom parameter group and specifies the correct port'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const bucket = new s3.Bucket(stack, 'Bucket');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.auroraPostgres({
                version: lib_1.AuroraPostgresEngineVersion.VER_11_4,
            }),
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportBuckets: [bucket],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            AssociatedRoles: [{
                    RoleArn: {
                        'Fn::GetAtt': [
                            'DatabaseS3ExportRole9E328562',
                            'Arn',
                        ],
                    },
                }],
            DBClusterParameterGroupName: 'default.aurora-postgresql11',
            Port: 5432,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup'));
        test.done();
    },
    'MySQL cluster without S3 exports or imports references the correct default ParameterGroup'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA_MYSQL,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            DBClusterParameterGroupName: 'default.aurora-mysql5.7',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::RDS::DBClusterParameterGroup'));
        test.done();
    },
    'throws when s3ExportRole and s3ExportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const exportRole = new aws_iam_1.Role(stack, 'ExportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const exportBucket = new s3.Bucket(stack, 'ExportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ExportRole: exportRole,
            s3ExportBuckets: [exportBucket],
        }));
        test.done();
    },
    'throws when s3ImportRole and s3ImportBuckets properties are both specified'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const importRole = new aws_iam_1.Role(stack, 'ImportRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
        });
        const importBucket = new s3.Bucket(stack, 'ImportBucket');
        // THEN
        test.throws(() => new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            s3ImportRole: importRole,
            s3ImportBuckets: [importBucket],
        }));
        test.done();
    },
    'can set CloudWatch log exports'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            cloudwatchLogsExports: ['error', 'general', 'slowquery', 'audit'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            EnableCloudwatchLogsExports: ['error', 'general', 'slowquery', 'audit'],
        }));
        test.done();
    },
    'can set CloudWatch log retention'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            cloudwatchLogsExports: ['error', 'general'],
            cloudwatchLogsRetention: logs.RetentionDays.THREE_MONTHS,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                    'Arn',
                ],
            },
            LogGroupName: { 'Fn::Join': ['', ['/aws/rds/cluster/', { Ref: 'DatabaseB269D8BB' }, '/error']] },
            RetentionInDays: 90,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                    'Arn',
                ],
            },
            LogGroupName: { 'Fn::Join': ['', ['/aws/rds/cluster/', { Ref: 'DatabaseB269D8BB' }, '/general']] },
            RetentionInDays: 90,
        }));
        test.done();
    },
    'throws if given unsupported CloudWatch log exports'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        test.throws(() => {
            new lib_1.DatabaseCluster(stack, 'Database', {
                engine: lib_1.DatabaseClusterEngine.AURORA,
                masterUser: {
                    username: 'admin',
                    password: cdk.SecretValue.plainText('tooshort'),
                },
                instanceProps: {
                    instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                    vpc,
                },
                cloudwatchLogsExports: ['error', 'general', 'slowquery', 'audit', 'thislogdoesnotexist', 'neitherdoesthisone'],
            });
        }, /Unsupported logs for the current engine type: thislogdoesnotexist,neitherdoesthisone/);
        test.done();
    },
    'can set deletion protection'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            masterUser: {
                username: 'admin',
                password: cdk.SecretValue.plainText('tooshort'),
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
            },
            deletionProtection: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::RDS::DBCluster', {
            DeletionProtection: true,
        }));
        test.done();
    },
    'does not throw (but adds a node error) if a (dummy) VPC does not have sufficient subnets'(test) {
        // GIVEN
        const stack = testStack();
        const vpc = ec2.Vpc.fromLookup(stack, 'VPC', { isDefault: true });
        // WHEN
        new lib_1.DatabaseCluster(stack, 'Database', {
            engine: lib_1.DatabaseClusterEngine.AURORA,
            instances: 1,
            masterUser: {
                username: 'admin',
            },
            instanceProps: {
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
                vpc,
                vpcSubnets: {
                    subnetName: 'DefinitelyDoesNotExist',
                },
            },
        });
        // THEN
        const art = assert_1.SynthUtils.synthesize(stack);
        const meta = art.findMetadataByType('aws:cdk:error');
        test.equal(meta[0].data, 'Cluster requires at least 2 subnets, got 0');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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