"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const targets = require("@aws-cdk/aws-events-targets");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
const app = new cdk.App();
class DatabaseInstanceStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const vpc = new ec2.Vpc(this, 'VPC', { maxAzs: 2 });
        /// !show
        // Set open cursors with parameter group
        const parameterGroup = new rds.ParameterGroup(this, 'ParameterGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            parameters: {
                open_cursors: '2500',
            },
        });
        /// Add XMLDB and OEM with option group
        const optionGroup = new rds.OptionGroup(this, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            configurations: [
                {
                    name: 'XMLDB',
                },
                {
                    name: 'OEM',
                    port: 1158,
                    vpc,
                },
            ],
        });
        // Allow connections to OEM
        optionGroup.optionConnections.OEM.connections.allowDefaultPortFromAnyIpv4();
        // Database instance with production values
        const instance = new rds.DatabaseInstance(this, 'Instance', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE3, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            masterUsername: 'syscdk',
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.seconds(60),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener',
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: false,
            optionGroup,
            parameterGroup,
        });
        // Allow connections on default port from any IPV4
        instance.connections.allowDefaultPortFromAnyIpv4();
        // Rotate the master user password every 30 days
        instance.addRotationSingleUser();
        // Add alarm for high CPU
        new cloudwatch.Alarm(this, 'HighCPU', {
            metric: instance.metricCPUUtilization(),
            threshold: 90,
            evaluationPeriods: 1,
        });
        // Trigger Lambda function on instance availability events
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline('exports.handler = (event) => console.log(event);'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const availabilityRule = instance.onEvent('Availability', { target: new targets.LambdaFunction(fn) });
        availabilityRule.addEventPattern({
            detail: {
                EventCategories: [
                    'availability',
                ],
            },
        });
        /// !hide
    }
}
new DatabaseInstanceStack(app, 'aws-cdk-rds-instance');
app.synth();
//# sourceMappingURL=data:application/json;base64,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