"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        let engine;
        if (this.dbCluster && this.dbInstance) {
            throw new Error('Proxy cannot target both database cluster and database instance.');
        }
        else if (this.dbCluster) {
            engine = this.dbCluster.node.defaultChild.engine;
        }
        else if (this.dbInstance) {
            engine = this.dbInstance.node.defaultChild.engine;
        }
        let engineFamily;
        switch (engine) {
            case 'aurora':
            case 'aurora-mysql':
            case 'mysql':
                engineFamily = 'MYSQL';
                break;
            case 'aurora-postgresql':
            case 'postgres':
                engineFamily = 'POSTGRESQL';
                break;
            default:
                throw new Error(`Unsupported engine type - ${engine}`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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