"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc,
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId],
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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