"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.PerformanceInsightRetention = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: core_1.Lazy.stringValue({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.anyValue({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightEncryptionKey && props.performanceInsightEncryptionKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        props.engine.bindToInstance(this, props);
        this.instanceType = (_a = props.instanceType) !== null && _a !== void 0 ? _a : ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        const instanceParameterGroupConfig = (_b = props.parameterGroup) === null || _b === void 0 ? void 0 : _b.bindToInstance({});
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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