"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10');
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10');
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11');
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11');
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
        });
        this.supportedLogTypes = ['postgresql'];
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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