# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xcsv']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.4.2,<2.0.0']

setup_kwargs = {
    'name': 'xcsv',
    'version': '0.1.0',
    'description': 'Package for working with extended CSV (XCSV) files',
    'long_description': "# xcsv\n\nxcsv is a package for reading and writing extended CSV files.\n\n## Extended CSV format\n\n* Extended header section of parseable atttributes, introduced by '#'.\n* Header row of variable and units for each column.\n* Data rows.\n\n### Example\n\n#### Extended header section\n\n* No leading/trailing whitespace.\n* Each line introduced by a comment ('#') character.\n* Each line contains a single header item.\n* Key/value separator ': '.\n* Preferably use a common vocabulary for attribute name, such as [CF conventions](http://cfconventions.org/index.html).\n* Preferably include recommended attributes from [ACDD](https://wiki.esipfed.org/Attribute_Convention_for_Data_Discovery_1-3).\n* Preferably use units from [Unified Code for Units of Measure](https://ucum.org/ucum.html) and/or [Udunits](https://www.unidata.ucar.edu/software/udunits/).\n* Units in parentheses.\n\n```\n# id: 1\n# title: The title\n# summary: This dataset...\n# authors: A B, C D\n# latitude: -73.86 (degree_north)\n# longitude: -65.86 (degree_east)\n# elevation: 1897, (m a.s.l.)\n# [a]: 2012 not a complete year\n```\n\n#### Header row\n\n* No leading/trailing whitespace.\n* Preferably use a common vocabulary for variable name, such as [CF conventions](http://cfconventions.org/index.html).\n* Units in parentheses.\n* Optional footnotes in square brackets.\n\n```\ntime (year) [a],depth (m)\n```\n\n#### Data row\n\n* No leading/trailing whitespace.\n\n```\n2012,0.575\n```\n\n## Install\n\nThe package can be installed from PyPI:\n\n```bash\n$ pip install xcsv\n```\n\n## Using the package\n\nThe package has a general `XCSV` class, that has a `metadata` attribute that holds the parsed contents of the extended file header section and the parsed column headers from the data table, and a `data` attribute that holds the data table (including the column headers as-is).\n\nThe `metadata` attribute is a `dict`, with the following general structure:\n\n```python\n{'header': {}, 'column_headers': {}}\n```\n\nand the `data` attribute is a `pandas.DataFrame`, and so has all the features of the [pandas](https://pandas.pydata.org/docs/index.html) package.\n\nThe package also has a `Reader` class for reading an extended CSV file, and similarly a `Writer` class for writing an `XCSV` object to a file in the extended CSV format.  In addition there is a `File` class that provides a convenient context manager for reading and writing these files.\n\n### Examples\n\n#### Simple read and print\n\nRead in a file and print the contents to `stdout`.  This shows how the contents of the extended CSV file are stored in the `XCSV` object.  Given the following script called, say, `simple_read.py`:\n\n```python\nimport argparse\n\nimport xcsv\n\nparser = argparse.ArgumentParser()\nparser.add_argument('filename', help='filename.csv')\nargs = parser.parse_args()\n\nwith xcsv.File(args.filename) as f:\n    content = f.read()\n    print(content.metadata)\n    print(content.data)\n```\n\nRunning it would produce:\n\n```bash\n$ python3 simple_read.py example.csv\n{'header': {'id': '1', 'title': 'The title', 'summary': 'This dataset...', 'authors': 'A B, C D', 'latitude': {'value': '-73.86', 'units': 'degree_north'}, 'longitude': {'value': '-65.86', 'units': 'degree_east'}, 'elevation': {'value': '1897,', 'units': 'm a.s.l.'}, '[a]': '2012 not a complete year'}, 'column_headers': {'time (year) [a]': {'name': 'time', 'units': 'year', 'notes': 'a'}, 'depth (m)': {'name': 'depth', 'units': 'm', 'notes': None}}}\n   time (year) [a]  depth (m)\n0             2012      0.575\n1             2011      1.125\n2             2010      2.225\n```\n\n#### Simple read and plot\n\nRead a file and plot the data:\n\n```python\nimport argparse\n\nimport matplotlib.pyplot as plt\n\nimport xcsv\n\nparser = argparse.ArgumentParser()\nparser.add_argument('filename', help='filename.csv')\nargs = parser.parse_args()\n\nwith xcsv.File(args.filename) as f:\n    content = f.read()\n    content.data.plot(x='depth (m)', y='time (year) [a]')\n    plt.show()\n```\n\n#### Simple read and write\n\nRead a file in, manipulate the data in some way, and write this modified `XCSV` object out to a new file:\n\n```python\nimport argparse\n\nimport xcsv\n\nparser = argparse.ArgumentParser()\nparser.add_argument('in_filename', help='in_filename.csv')\nparser.add_argument('out_filename', help='out_filename.csv')\nargs = parser.parse_args()\n\nwith xcsv.File(args.in_filename) as f:\n    content = f.read()\n\n# Manipulate the data...\n\nwith xcsv.File(args.out_filename, mode='w') as f:\n    f.write(xcsv=content)\n```\n\n",
    'author': 'Paul Breen',
    'author_email': 'pbree@bas.ac.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/paul-breen/xcsv',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
