# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ehandler', 'ehandler.http', 'ehandler.logging', 'ehandler.models']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.0',
 'fastapi>=0.70',
 'orjson>=3.8.0',
 'pydantic>=1.0',
 'python-dotenv>=0.10.0']

setup_kwargs = {
    'name': 'ehandler',
    'version': '0.5.1',
    'description': 'Utility to create exception handlers in FastAPI in a flexible and standard way',
    'long_description': '<img src="./assets/images/logo.png" alt="Trii Logo" title="Trii" align="right" height="32">\n\n# Exception Handler [ FastAPI Utility ]\n\nThis project works together with FastAPI and allows you to create error handlers\nin a very simple and standard way. Basically it allows you to add (or use the default\na function that will be executed when an exception occurs on the server.\n\nThis is possible using the FastAPI Exception Handler (actually Starlette) and\nusing a function as a template that receives data such as the request object\nthat was invoked on the route, the exception, the status code that handles that\nexception, and some _\'static\'_ arguments.\n\n## Table of Contents\n\n- [How to use it](#how-to-use-it)\n  - [Basic](#basic)\n  - [Explanation](#explanation)\n- [Examples](#examples)\n  - [Content callback](#content-callback)\n  - [Add additional data](#add-additional-data)\n  - [Change default status code](#change-default-statuscode)\n- [Contribute](#how-to-contribute)\n  - [Prerequisites](#prerequisites)\n  - [Install Poetry](#install-poetry)\n  - [Install Pre-Commit](#install-pre-commit)\n    - [Install hooks](#install-hooks)\n  - [Install dependencies](#install-dependencies)\n  - [Add dependencies](#add-dependency)\n  - [Add dev dependencies](#add-dev-dependency)\n  - [Run test](#run-test)\n    - [Static test](#static-test)\n      - [Single test](#single-test)\n      - [All test](#all-test)\n      - [All files](#all-files)\n    - [Static test](#unittest)\n\n## How to use it\n\n### Basic\n\nThere are several ways to use it, the simplest is to use it with the templates\n(Callbacks) that come by default. And simply create a list with the pairs of\nexceptions and status codes.\n\n<details>\n\n```python\nfrom fastapi import FastAPI\nfrom ehandler import ExceptionHandlerSetter\n\napp = FastAPI()\n\nhandlers = [(ZeroDivisionError, 500), (ValueError, 400)]\nExceptionHandlerSetter(\n    content_callback_kwargs={"show_error": True, "show_data": True}\n).add_handlers(app, handlers)\n\n\n@app.get("/value_error")\ndef raise_value_error():\n    raise ValueError("Invalid value")\n\n\n@app.get("/division")\ndef call_external():\n    return some_func()\n\n\n@app.get("/uncaught")\ndef uncaught():\n    raise Exception("This error is not handled")\n\n\ndef some_func():\n    print("Trying to do something unless that thing get fails")\n    return 1 / 0  # This raise a `ZeroDivisionError`\n```\n\n`/value_error` - **_HTTP 400 Bad Request_**\n\n```json\n{\n    "detail": "Bad Request",\n    "error": "ValueError",\n    "message": "Invalid value"\n}\n```\n\n`/division` - **_HTTP 500 Internal Server Error_**\n\n```json\n{\n    "detail": "Internal Server Error",\n    "error": "ZeroDivisionError",\n    "message": "division by zero"\n}\n```\n\n`/uncaught` - **_HTTP 500 Internal Server Error_**\n\n```json\nInternal Server Error\n```\n\n#### **Explanation**\n\nWhen an error occurs within our server that is not handled (with a `try-except`)\nthe server will respond with a _500 - Internal Server Error_\n\n`ExceptionHandlerSetter().add_handlers(app, handlers)` Basically create this to\nbe able to handle specific exceptions (or general if `Exception` is handled)\n( [original example](https://fastapi.tiangolo.com/tutorial/handling-errors/?h=#install-custom-exception-handlers) ) :\n\n```python\n@app.exception_handler(UnicornException)\nasync def unicorn_exception_handler(request: Request, exc: UnicornException):\n    return JSONResponse(\n        status_code=418,\n        content={"message": f"Oops! {exc.name} did something. There goes a rainbow..."},\n    )\n```\n\nBut with a dynamic/configurable exception and status code and using a function\n(callback) as a template, in fact 2 functions/callbacks are used:\n\n- One that parses the content, can take the exception, the request, etc, and\ncreate some content with it.\n- The other is the function that returns the response, usually just put it\ninside a `JSONResponse`.\n\n```python\n@app.exception_handler(<ExceptionClass>)\nasync def <ResponseCallback>(request: Request, content: Any = <content_callback>, status_code: int = <status_code>)\n    return <ContentCallback>,\n```\n\n</details>\n\n___\n\n## Examples\n\n### Content callback\n\n<details>\n\nUse your own callback to parse the content of the response\n\n```python\nimport traceback\n\nfrom fastapi import FastAPI\nfrom ehandler import ExceptionHandlerSetter\n\napp = FastAPI()\n\ndef parse_exception(\n    request: Request,\n    exception: Exception,\n    status_code: int,\n    debug: bool = False,\n) -> dict:\n    content = {"message": "Something went wrong"}\n    if debug:\n        content["traceback"] = traceback.format_exc()\n    return content\n\nhandlers = [(Exception, 500)]\nExceptionHandlerSetter(\n    content_callback=parse_exception\n    content_callback_kwargs={"debug": True}\n).add_handlers(app, handlers)\n```\n\n</details>\n\n___\n\n### Add additional data\n\n<details>\n\n> ⚠️ This uses the default implementation from `ehandler.parsers.parse_exception`\n\n```python\nfrom fastapi import FastAPI\nfrom ehandler import ExceptionHandlerSetter\nfrom ehandler.utils import add_data\n\napp = FastAPI()\n\nhandlers = [(Exception, 500)]\nExceptionHandlerSetter(\n    content_callback_kwargs={"show_data": True}  # If `show_data` is False it won\'t work\n).add_handlers(app, handlers)\n\n\n@app.get("/exception")\ndef raise_value_error():\n    raise add_data(ValueError("Invalid value"), {"user": "user_info"})\n```\n\n`/exception` - **_HTTP 500 Internal Server Error_**\n\n```json\n{\n    "data": {\n        "user": "user_info"\n    },\n    "detail": "Internal Server Error",\n    "message": "Invalid value"\n}\n```\n\n</details>\n\n___\n\n### Change default _status_code_\n\n<details>\n\n```python\nfrom fastapi import FastAPI\nfrom ehandler import ExceptionHandlerSetter\nfrom ehandler.utils import add_code\n\napp = FastAPI()\n\nhandlers = [(Exception, 500)]\nExceptionHandlerSetter(force_status_code=False).add_handlers(app, handlers)\n\n\n@app.get("/exception")\ndef raise_value_error():\n    raise add_code(ValueError("Invalid value"), 400)\n```\n\n`/exception` - **_HTTP 400 Bad Request_**\n\n```json\n{\n    "detail": "Bad Request",\n    "message": "Invalid value"\n}\n```\n\n</details>\n\n## How to contribute\n\n### Prerequisites\n\n> Python ^3.9\n>\n> poetry ^1.1.14\n>\n> pre-commit ^2.20.0\n\n<details>\n<summary> Install requirements </summary>\n\n### **Install Poetry**\n\nPoetry is a tool for dependency management and packaging in Python. It allows\nyou to declare the libraries your project depends on and it will manage\n(install/update) them for you.\n\nTo install poetry you can follow the official documentation on the page according\nto your operating system. [Poetry Installation](https://python-poetry.org/docs/#installation)\n\n### **Install Pre-Commit**\n\nIt is a multi-language package manager for pre-commit hooks. You specify a list\nof hooks you want and pre-commit manages the installation and execution of any\nhook written in any language before every commit.\n\nTo install poetry you can follow the official documentation on the page according\nto your operating system. [Pre-Commit Installation](https://pre-commit.com/#install)\n\n#### **Install hooks**\n\n```bash\npre-commit install\n```\n\n</details>\n\n<details>\n<summary> Dependencies </summary>\n\n### **Install dependencies**\n\n```bash\npoetry install\n```\n\nYou can also follow the documentation of poetry for a better use of this or any\nquestions. [Poetry Basic Usage](https://python-poetry.org/docs/basic-usage/)\n\n### **Add dependency**\n\nThey are the dependencies that the package needs to work.\n\n```bash\npoetry add <package>\n```\n\n```bash\npoetry add fastapi\n```\n\n### **Add dev dependency**\n\nThese are the dependencies that you need only for development,\nfor example those that are needed to test the package.\n\n```bash\npoetry add -D <package>\n```\n\n```bash\npoetry add -D requests\n```\n\n</details>\n\n<details>\n<summary> Tests </summary>\n\n### **Run test**\n\n#### **Static test**\n\nThe analysis or static test is run using pre-commit, you can run a specific\nanalysis using the id, or run all tests\n\n##### **Single test**\n\n```bash\npre-commit run pylint\n```\n\n> ⚠️ Pre-commit by default only runs on files modified in stage.\n> If you want to run on all files you can add the `--all-files` flag.\n\n##### **All test**\n\n```bash\npre-commit run\n```\n\n##### **All files**\n\n```bash\npre-commit run --all-files\npre-commit run black --all-files\n```\n\n#### **Unittest**\n\n```bash\npoetry run pytest -v\n```\n\n</details>\n',
    'author': 'Alan Vazquez',
    'author_email': 'alan@trii.co',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
