##
# File: RCI.py
# Date: 12-Jul-2021
#
# Updates:
# 13-Oct-2021  M. Yokochi - code revision according to PEP8 using Pylint (DAOTHER-7389, issue #5)
##
""" Wrapper class for Random Coil Index calculation.
    @author: Masashi Yokochi
"""
import sys
from math import pow, exp, log  # pylint: disable=redefined-builtin

import numpy as np


def lmean(list):  # pylint: disable=redefined-builtin
    """ Returns the average of the array elements.
    """

    a = np.array(list)

    return np.mean(a)


# pylint: disable=attribute-defined-outside-init
class RCI:
    """ Wrapper class for Random Coil Index calculation.
    """

    def __init__(self, verbose=False, log=sys.stderr):  # pylint: disable=redefined-builtin,redefined-outer-name
        self.__verbose = verbose
        self.__lfh = log

        self.pos_neg_value = 0  # If 1, different scaling is used for pos and neg sec chem shifts

        self.floor_value = 0.8
        self.floor_value1 = 0.5  # was 0.8
        self.early_floor_value_pos = 0.1
        self.early_floor_value_neg = -0.1

        self.incomplete_data_use = 1  # 0 - do not use data for a residue when chemical shifts are not known for all atoms, 1 - use even if data is incomplete
        self.NoNextProPro_flag = 0  # When 0, next residue correction is used for XPP situation

        self.Random_coil_flag = 4  # 1 - Wishart's random coil, 2 - Wang's random coil, 4 - Shwarzinger, 3 - Lukhin' random coil
        self.preceed_res_effect_flag = 1  # If 1, use preceeding residue effect
        self.next_res_effect_flag = 1  # If 1, use next residue effect
        self.preceed_preceed_res_effect_flag = 1  # If 1, use preceeding residue of preceeding residue effect
        self.next_next_res_effect_flag = 1  # If 1, use next residue of  next residue effect

        self.sw_neighbor_flag = 0  # If 1, Shwartz-Wang neighboring residue correction is used
        self.neighbor_flag = 1  # 0 - Wang neighbor correction, 1 - Swartzinger neighbor correction
        self.nonextpro = 0
        self.nopreceedpro = 0

        self.N_term_coef = self.C_term_coef = 0.2
        self.N_term_res_num = 2
        self.C_term_res_num = 2
        self.N_term_low = self.C_term_low = 0.06
        self.N_term_high = self.C_term_high = 0.40  # 0.22 for L30, 0.20 For Vat, Was 0.4
        self.termini_corr_flag = 3

        self.function_flag = 8
        self.no_smoothing_flag = 0
        self.three_point_smooth_flag = 0
        self.five_point_smooth_flag = 1
        self.double_three_point_smooth_flag = 0

        self.final_no_smoothing_flag = 0
        self.final_three_point_smooth_flag = 0
        self.final_five_point_smooth_flag = 1
        self.final_double_three_point_smooth_flag = 0
        self.gap_limit_smooth = 0

        self.B_Cys_switch = 2  # Correction for oxidized and reduced Cys. If 0, no correction, if 1, correction is used, if 2, CB shift from oxidized residue ignored ( Default)

        self.COp_exclude = 0  # If 1, exclude from grid search
        self.CAp_exclude = 0  # If 1, exclude from grid search
        self.CBp_exclude = 0  # If 1, exclude from grid search
        self.HAp_exclude = 0  # If 1, exclude from grid search
        self.Hp_exclude = 0  # If 1, exclude from grid search
        self.Np_exclude = 0  # If 1, exclude from grid search

        self.grid_search_flag = 0
        self.gridsearch_list = [1.0]

        self.COp_list = [0.72]
        self.HAp_list = [0.85]
        self.CAp_list = [0.72]
        self.Np_list = [0.59]
        self.CBp_list = [0.15]
        self.Hp_list = [0.13]

        self.coef_for_neg_CA = -1  # -1 for PyJ
        self.coef_for_pos_CA = 1  # 1 for PyJ
        self.coef_for_neg_CO = -1  # 0 for PyJ. -1 for Barnase
        self.coef_for_pos_CO = 1  # 1 for PyJ
        self.coef_for_neg_CB = -1  # -1 for PyJ
        self.coef_for_pos_CB = 1  # 1 for PyJ
        self.coef_for_neg_HA = -1  # -1 for PyJ
        self.coef_for_pos_HA = 1  # 0 for PyJ
        self.coef_for_neg_H = -1  # 0 for PyJ
        self.coef_for_pos_H = 1  # 1 for PyJ
        self.coef_for_neg_N = -1  # -1 for PyJ, should be removed for Frataxin
        self.coef_for_pos_N = 0  # 0 for PyJ, should be removed for Frataxin

        self.Hertz_correction = 1
        self.CA_Hertz_corr = 1.0
        self.N_Hertz_corr = 1.0
        self.H_Hertz_corr = 1.0

        self.final_smooth = 3
        self.smooth = 3

        self.gap_fill_flag = 2
        self.gap_limit = 2  # Limit for filling assignment gaps, for example, 2 means that only residues i+1 and i+2 can be used to fill missing assignment of residue i

        self.miss_ass = 1
        self.scale = 1

        self.__setup__()

    def __setup__(self):

        if self.function_flag == 1:
            self.floor_value = 0.6
        if self.function_flag == 3:
            self.floor_value = 0.1
        if self.function_flag == 4:
            self.floor_value = 0.1
        if self.function_flag == 5:
            self.floor_value = 0
        if self.function_flag == 6:
            self.floor_value = 0.6
        if self.function_flag == 7:
            self.floor_value = 0.6
        if self.function_flag == 8:
            self.floor_value1 = self.floor_value1
            self.floor_value2 = 0.6  # applied to sigma after End COrrection
            self.floor_value = 0.5  # was 1.9, applied to sigma before End Correction

        if self.Hertz_correction == 1:
            self.CA_Hertz_corr = 2.5
            self.N_Hertz_corr = 1.0
            self.H_Hertz_corr = 10.0

        self.__number_of_best_matches = 100000000000000

        self.__Hertz_correction_dic = {
            "CA": 2.5,
            "HA": 10.0,
            "C": 2.5,
            "N": 1.0,
            "CB": 2.5,
            "H": 10.0,
            "HN": 10.0,
            "NH": 10.0,
        }

        self.__atom_list = ["CA", "HA", "C", "CB", "N", "H"]

        self.__atom_place_in_simpred = {
            "CA": 2,
            "HA": 7,
            "C": 4,
            "N": 5,
            "CB": 3,
            "H": 6,
            "HN": 6,
            "NH": 6,
        }

        self.__aa_names_full_all_CAP = {
            "GLN": "Q",
            "TRP": "W",
            "GLU": "E",
            "ARG": "R",
            "THR": "T",
            "TYR": "Y",
            "ILE": "I",
            "PRO": "P",
            "ALA": "A",
            "SER": "S",
            "ASP": "D",
            "PHE": "F",
            "GLY": "G",
            "HIS": "H",
            "LYS": "K",
            "LEU": "L",
            "CYS": "C",
            "VAL": "V",
            "ASN": "N",
            "MET": "M"
        }

        # self.__aa_names_1let_2_full_all_CAP = {
        #     "Q": "GLN",
        #     "W": "TRP",
        #     "E": "GLU",
        #     "R": "ARG",
        #     "T": "THR",
        #     "Y": "TYR",
        #     "I": "ILE",
        #     "P": "PRO",
        #     "A": "ALA",
        #     "S": "SER",
        #     "D": "ASP",
        #     "F": "PHE",
        #     "G": "GLY",
        #     "H": "HIS",
        #     "K": "LYS",
        #     "L": "LEU",
        #     "C": "CYS",
        #     "V": "VAL",
        #     "N": "ASN",
        #     "M": "MET",
        #     "B": "CYS",
        # }

        # Wang's neighboring residue correction with full next residue correction when Pro is i+1
        if self.neighbor_flag == 0 and self.nonextpro == 0:

            self.preceed_res_effect = {
                "A": [
                    [-2.29, -2.21, -1.60],
                    [0.05, 0.14, 0.11],
                    [-0.03, -0.01, -0.13],
                    [0.06, -0.04, 0.01],
                    [-0.02, -0.07, -0.08],
                    [0.01, -0.01, -0.00],
                ],
                "C": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "B": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "D": [
                    [-2.06, -0.43, 0.43],
                    [0.01, 0.07, -0.21],
                    [0.22, 0.20, -0.12],
                    [-0.20, -0.07, 0.20],
                    [-0.22, -0.01, -0.05],
                    [-0.05, -0.03, 0.03],
                ],
                "E": [
                    [-0.15, -0.36, -0.13],
                    [0.01, 0.04, -0.02],
                    [0.12, 0.01, -0.23],
                    [-0.00, 0.04, 0.02],
                    [0.01, -0.01, -0.04],
                    [-0.05, -0.01, 0.03],
                ],
                "F": [
                    [-1.01, 0.20, -0.60],
                    [-0.18, -0.14, 0.05],
                    [-0.22, 0.02, 0.13],
                    [0.26, -0.06, -0.04],
                    [0.12, 0.01, 0.16],
                    [0.06, -0.01, -0.09],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [-0.34, -0.05, -0.27],
                    [0.03, -0.13, -0.40],
                    [0.14, 0.16, 0.01],
                    [0.05, 0.07, 0.22],
                    [-0.02, -0.01, 0.02],
                    [-0.05, -0.05, 0.00],
                ],
                "I": [
                    [3.16, 2.92, 0.23],
                    [0.12, 0.11, 0.12],
                    [-0.17, -0.15, 0.43],
                    [-0.04, 0.26, -0.07],
                    [0.15, 0.12, 0.02],
                    [0.05, 0.04, -0.03],
                ],
                "K": [
                    [-0.12, -0.26, -0.27],
                    [0.08, -0.09, -0.18],
                    [0.10, -0.07, -0.26],
                    [-0.08, 0.08, -0.02],
                    [-0.03, -0.02, -0.10],
                    [-0.00, -0.00, 0.03],
                ],
                "L": [
                    [0.27, -0.76, -0.74],
                    [0.07, 0.13, 0.12],
                    [-0.21, -0.07, 0.14],
                    [0.12, -0.03, -0.01],
                    [0.12, -0.06, 0.00],
                    [0.04, 0.00, -0.03],
                ],
                "M": [
                    [0.01, 0.69, -0.48],
                    [-0.19, 0.10, -0.13],
                    [-0.01, 0.09, 0.05],
                    [0.05, 0.10, -0.08],
                    [0.11, 0.05, -0.03],
                    [0.06, 0.05, -0.00],
                ],
                "N": [
                    [-1.59, -0.76, 1.09],
                    [-0.20, -0.19, -0.53],
                    [0.11, 0.18, -0.18],
                    [0.15, -0.20, 0.08],
                    [-0.24, 0.01, 0.03],
                    [-0.02, -0.03, 0.05],
                ],
                "P": [
                    [-2.05, -0.94, -0.74],
                    [0.18, 0.21, 0.04],
                    [0.14, 0.07, 0.04],
                    [-0.22, -0.16, -0.11],
                    [-0.17, 0.14, 0.08],
                    [-0.16, -0.04, 0.00],
                ],
                "Q": [
                    [-0.17, -0.09, 0.04],
                    [0.02, 0.07, -0.04],
                    [0.15, 0.13, -0.09],
                    [-0.21, 0.06, 0.15],
                    [-0.05, 0.01, -0.04],
                    [-0.01, -0.00, 0.03],
                ],
                "R": [
                    [-0.68, -0.45, -0.52],
                    [0.21, -0.07, 0.01],
                    [0.19, 0.00, -0.17],
                    [-0.12, 0.01, 0.03],
                    [-0.11, -0.03, -0.04],
                    [-0.00, 0.01, 0.06],
                ],
                "S": [
                    [-0.52, 1.16, 2.18],
                    [-0.08, -0.10, -0.34],
                    [0.18, 0.11, -0.34],
                    [0.14, -0.06, -0.01],
                    [-0.05, 0.02, 0.05],
                    [0.01, 0.01, 0.07],
                ],
                "T": [
                    [1.74, 1.23, 2.07],
                    [-0.14, -0.07, -0.20],
                    [0.06, 0.05, 0.10],
                    [-0.14, -0.11, -0.16],
                    [0.08, 0.02, 0.12],
                    [-0.01, -0.00, 0.03],
                ],
                "V": [
                    [2.80, 2.77, 0.17],
                    [0.04, -0.00, 0.13],
                    [-0.13, -0.16, 0.27],
                    [-0.17, 0.03, -0.14],
                    [0.14, 0.17, 0.01],
                    [0.02, 0.04, -0.03],
                ],
                "W": [
                    [-1.06, 0.97, -0.31],
                    [-0.02, -0.46, 0.15],
                    [-0.14, 0.00, 0.19],
                    [0.61, 0.28, -0.25],
                    [0.26, -0.08, 0.20],
                    [0.02, -0.05, -0.13],
                ],
                "Y": [
                    [-0.60, 0.46, -0.67],
                    [-0.08, -0.03, 0.05],
                    [-0.21, -0.19, 0.13],
                    [0.22, -0.10, -0.12],
                    [0.13, 0.02, 0.11],
                    [0.00, 0.01, -0.11],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.41, -0.11, 0.06],
                    [-0.12, 0.05, -0.07],
                    [0.03, 0.07, -0.04],
                    [0.05, -0.09, -0.11],
                    [-0.02, -0.01, 0.05],
                    [-0.03, -0.03, -0.01],
                ],
                "C": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "B": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "D": [
                    [-0.10, 0.23, 0.12],
                    [0.06, -0.11, -0.26],
                    [0.03, 0.28, 0.09],
                    [0.08, 0.11, -0.07],
                    [0.04, 0.04, 0.12],
                    [-0.06, -0.03, -0.05],
                ],
                "E": [
                    [-0.13, 0.26, 0.25],
                    [0.13, 0.14, 0.07],
                    [0.18, 0.25, 0.21],
                    [0.15, 0.06, -0.13],
                    [0.02, 0.07, 0.13],
                    [-0.04, -0.04, -0.05],
                ],
                "F": [
                    [-0.36, -0.35, 0.34],
                    [-0.23, -0.22, 0.04],
                    [-0.12, -0.04, 0.05],
                    [0.10, -0.09, 0.07],
                    [-0.11, -0.04, -0.10],
                    [0.08, -0.02, -0.03],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.45, -0.09, -0.28],
                    [-0.20, -0.05, -0.21],
                    [0.08, 0.22, -0.04],
                    [-0.03, -0.24, 0.09],
                    [-0.02, -0.01, -0.02],
                    [-0.04, -0.09, -0.05],
                ],
                "I": [
                    [-0.01, -0.20, 0.14],
                    [-0.04, -0.09, 0.17],
                    [0.10, 0.03, 0.20],
                    [0.05, 0.28, 0.11],
                    [0.01, -0.01, -0.21],
                    [0.06, 0.03, 0.01],
                ],
                "K": [
                    [0.12, -0.13, -0.05],
                    [0.09, -0.13, -0.16],
                    [0.22, 0.08, -0.11],
                    [-0.06, 0.01, -0.05],
                    [0.07, -0.04, 0.05],
                    [-0.04, -0.03, -0.00],
                ],
                "L": [
                    [-0.01, -0.49, 0.30],
                    [-0.27, 0.06, 0.11],
                    [0.09, 0.10, 0.01],
                    [-0.03, -0.10, 0.10],
                    [0.04, -0.07, -0.14],
                    [0.01, -0.02, 0.03],
                ],
                "M": [
                    [0.34, -0.02, 0.10],
                    [-0.05, 0.19, -0.08],
                    [0.23, 0.22, 0.14],
                    [0.18, 0.06, 0.05],
                    [0.04, 0.01, -0.06],
                    [0.00, -0.01, -0.01],
                ],
                "N": [
                    [-0.07, -0.03, -0.12],
                    [-0.01, -0.23, -0.34],
                    [0.21, 0.24, -0.14],
                    [-0.21, -0.06, 0.04],
                    [-0.02, 0.04, 0.11],
                    [-0.13, -0.04, -0.03],
                ],
                "P": [
                    [0.89, 0.92, 0.33],
                    [-0.92, -1.19, -2.49],
                    [-1.63, -2.04, 0.51],
                    [-0.94, -0.20, -1.70],
                    [-0.07, -0.17, 0.04],
                    [-0.07, 0.21, 0.18],
                ],
                "Q": [
                    [0.03, -0.31, -0.07],
                    [0.04, 0.10, 0.10],
                    [0.20, 0.37, 0.00],
                    [0.01, -0.13, -0.04],
                    [0.01, -0.13, -0.04],
                    [0.04, 0.01, 0.07],
                    [-0.05, -0.05, -0.02],
                ],
                "R": [
                    [-0.23, -0.09, -0.09],
                    [0.00, 0.16, 0.01],
                    [0.03, 0.19, 0.03],
                    [0.10, -0.12, -0.02],
                    [0.09, -0.01, 0.01],
                    [0.01, -0.02, -0.02],
                ],
                "S": [
                    [0.32, 0.30, -0.17],
                    [0.26, 0.10, 0.03],
                    [0.11, 0.10, -0.24],
                    [0.08, 0.14, -0.10],
                    [-0.03, 0.07, 0.13],
                    [0.01, 0.02, 0.04],
                ],
                "T": [
                    [0.35, 0.22, -0.54],
                    [0.38, 0.10, 0.02],
                    [0.07, 0.02, -0.03],
                    [0.14, 0.14, 0.01],
                    [0.02, 0.04, 0.09],
                    [0.05, 0.08, 0.03],
                ],
                "V": [
                    [0.16, -0.05, 0.25],
                    [0.05, -0.08, 0.26],
                    [0.12, 0.05, 0.07],
                    [-0.11, 0.18, 0.05],
                    [0.01, 0.01, -0.17],
                    [0.05, 0.05, 0.03],
                ],
                "W": [
                    [0.06, -0.59, 0.06],
                    [-0.46, -0.33, -0.00],
                    [0.02, -0.06, 0.14],
                    [0.28, -0.03, 0.07],
                    [-0.16, -0.10, -0.08],
                    [-0.05, -0.03, 0.01],
                ],
                "Y": [
                    [-0.42, -0.48, 0.03],
                    [-0.29, -0.51, -0.19],
                    [-0.12, 0.10, 0.05],
                    [0.30, 0.00, 0.07],
                    [-0.09, -0.07, -0.03],
                    [0.09, -0.06, -0.03],
                ],
            }

        # Wang's neighboring residue correction, no next residue correction when Pro is i+1
        elif self.neighbor_flag == 0 and self.nonextpro == 1:

            self.preceed_res_effect = {
                "A": [
                    [-2.29, -2.21, -1.60],
                    [0.05, 0.14, 0.11],
                    [-0.03, -0.01, -0.13],
                    [0.06, -0.04, 0.01],
                    [-0.02, -0.07, -0.08],
                    [0.01, -0.01, -0.00],
                ],
                "C": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "B": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "D": [
                    [-2.06, -0.43, 0.43],
                    [0.01, 0.07, -0.21],
                    [0.22, 0.20, -0.12],
                    [-0.20, -0.07, 0.20],
                    [-0.22, -0.01, -0.05],
                    [-0.05, -0.03, 0.03],
                ],
                "E": [
                    [-0.15, -0.36, -0.13],
                    [0.01, 0.04, -0.02],
                    [0.12, 0.01, -0.23],
                    [-0.00, 0.04, 0.02],
                    [0.01, -0.01, -0.04],
                    [-0.05, -0.01, 0.03],
                ],
                "F": [
                    [-1.01, 0.20, -0.60],
                    [-0.18, -0.14, 0.05],
                    [-0.22, 0.02, 0.13],
                    [0.26, -0.06, -0.04],
                    [0.12, 0.01, 0.16],
                    [0.06, -0.01, -0.09],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [-0.34, -0.05, -0.27],
                    [0.03, -0.13, -0.40],
                    [0.14, 0.16, 0.01],
                    [0.05, 0.07, 0.22],
                    [-0.02, -0.01, 0.02],
                    [-0.05, -0.05, 0.00],
                ],
                "I": [
                    [3.16, 2.92, 0.23],
                    [0.12, 0.11, 0.12],
                    [-0.17, -0.15, 0.43],
                    [-0.04, 0.26, -0.07],
                    [0.15, 0.12, 0.02],
                    [0.05, 0.04, -0.03],
                ],
                "K": [
                    [-0.12, -0.26, -0.27],
                    [0.08, -0.09, -0.18],
                    [0.10, -0.07, -0.26],
                    [-0.08, 0.08, -0.02],
                    [-0.03, -0.02, -0.10],
                    [-0.00, -0.00, 0.03],
                ],
                "L": [
                    [0.27, -0.76, -0.74],
                    [0.07, 0.13, 0.12],
                    [-0.21, -0.07, 0.14],
                    [0.12, -0.03, -0.01],
                    [0.12, -0.06, 0.00],
                    [0.04, 0.00, -0.03],
                ],
                "M": [
                    [0.01, 0.69, -0.48],
                    [-0.19, 0.10, -0.13],
                    [-0.01, 0.09, 0.05],
                    [0.05, 0.10, -0.08],
                    [0.11, 0.05, -0.03],
                    [0.06, 0.05, -0.00],
                ],
                "N": [
                    [-1.59, -0.76, 1.09],
                    [-0.20, -0.19, -0.53],
                    [0.11, 0.18, -0.18],
                    [0.15, -0.20, 0.08],
                    [-0.24, 0.01, 0.03],
                    [-0.02, -0.03, 0.05],
                ],
                "P": [
                    [-2.05, -0.94, -0.74],
                    [0.18, 0.21, 0.04],
                    [0.14, 0.07, 0.04],
                    [-0.22, -0.16, -0.11],
                    [-0.17, 0.14, 0.08],
                    [-0.16, -0.04, 0.00],
                ],
                "Q": [
                    [-0.17, -0.09, 0.04],
                    [0.02, 0.07, -0.04],
                    [0.15, 0.13, -0.09],
                    [-0.21, 0.06, 0.15],
                    [-0.05, 0.01, -0.04],
                    [-0.01, -0.00, 0.03],
                ],
                "R": [
                    [-0.68, -0.45, -0.52],
                    [0.21, -0.07, 0.01],
                    [0.19, 0.00, -0.17],
                    [-0.12, 0.01, 0.03],
                    [-0.11, -0.03, -0.04],
                    [-0.00, 0.01, 0.06],
                ],
                "S": [
                    [-0.52, 1.16, 2.18],
                    [-0.08, -0.10, -0.34],
                    [0.18, 0.11, -0.34],
                    [0.14, -0.06, -0.01],
                    [-0.05, 0.02, 0.05],
                    [0.01, 0.01, 0.07],
                ],
                "T": [
                    [1.74, 1.23, 2.07],
                    [-0.14, -0.07, -0.20],
                    [0.06, 0.05, 0.10],
                    [-0.14, -0.11, -0.16],
                    [0.08, 0.02, 0.12],
                    [-0.01, -0.00, 0.03],
                ],
                "V": [
                    [2.80, 2.77, 0.17],
                    [0.04, -0.00, 0.13],
                    [-0.13, -0.16, 0.27],
                    [-0.17, 0.03, -0.14],
                    [0.14, 0.17, 0.01],
                    [0.02, 0.04, -0.03],
                ],
                "W": [
                    [-1.06, 0.97, -0.31],
                    [-0.02, -0.46, 0.15],
                    [-0.14, 0.00, 0.19],
                    [0.61, 0.28, -0.25],
                    [0.26, -0.08, 0.20],
                    [0.02, -0.05, -0.13],
                ],
                "Y": [
                    [-0.60, 0.46, -0.67],
                    [-0.08, -0.03, 0.05],
                    [-0.21, -0.19, 0.13],
                    [0.22, -0.10, -0.12],
                    [0.13, 0.02, 0.11],
                    [0.00, 0.01, -0.11],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.41, -0.11, 0.06],
                    [-0.12, 0.05, -0.07],
                    [0.03, 0.07, -0.04],
                    [0.05, -0.09, -0.11],
                    [-0.02, -0.01, 0.05],
                    [-0.03, -0.03, -0.01],
                ],
                "C": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "B": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "D": [
                    [-0.10, 0.23, 0.12],
                    [0.06, -0.11, -0.26],
                    [0.03, 0.28, 0.09],
                    [0.08, 0.11, -0.07],
                    [0.04, 0.04, 0.12],
                    [-0.06, -0.03, -0.05],
                ],
                "E": [
                    [-0.13, 0.26, 0.25],
                    [0.13, 0.14, 0.07],
                    [0.18, 0.25, 0.21],
                    [0.15, 0.06, -0.13],
                    [0.02, 0.07, 0.13],
                    [-0.04, -0.04, -0.05],
                ],
                "F": [
                    [-0.36, -0.35, 0.34],
                    [-0.23, -0.22, 0.04],
                    [-0.12, -0.04, 0.05],
                    [0.10, -0.09, 0.07],
                    [-0.11, -0.04, -0.10],
                    [0.08, -0.02, -0.03],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.45, -0.09, -0.28],
                    [-0.20, -0.05, -0.21],
                    [0.08, 0.22, -0.04],
                    [-0.03, -0.24, 0.09],
                    [-0.02, -0.01, -0.02],
                    [-0.04, -0.09, -0.05],
                ],
                "I": [
                    [-0.01, -0.20, 0.14],
                    [-0.04, -0.09, 0.17],
                    [0.10, 0.03, 0.20],
                    [0.05, 0.28, 0.11],
                    [0.01, -0.01, -0.21],
                    [0.06, 0.03, 0.01],
                ],
                "K": [
                    [0.12, -0.13, -0.05],
                    [0.09, -0.13, -0.16],
                    [0.22, 0.08, -0.11],
                    [-0.06, 0.01, -0.05],
                    [0.07, -0.04, 0.05],
                    [-0.04, -0.03, -0.00],
                ],
                "L": [
                    [-0.01, -0.49, 0.30],
                    [-0.27, 0.06, 0.11],
                    [0.09, 0.10, 0.01],
                    [-0.03, -0.10, 0.10],
                    [0.04, -0.07, -0.14],
                    [0.01, -0.02, 0.03],
                ],
                "M": [
                    [0.34, -0.02, 0.10],
                    [-0.05, 0.19, -0.08],
                    [0.23, 0.22, 0.14],
                    [0.18, 0.06, 0.05],
                    [0.04, 0.01, -0.06],
                    [0.00, -0.01, -0.01],
                ],
                "N": [
                    [-0.07, -0.03, -0.12],
                    [-0.01, -0.23, -0.34],
                    [0.21, 0.24, -0.14],
                    [-0.21, -0.06, 0.04],
                    [-0.02, 0.04, 0.11],
                    [-0.13, -0.04, -0.03],
                ],
                "P": [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]],
                "Q": [
                    [0.03, -0.31, -0.07],
                    [0.04, 0.10, 0.10],
                    [0.20, 0.37, 0.00],
                    [0.01, -0.13, -0.04],
                    [0.01, -0.13, -0.04],
                    [0.04, 0.01, 0.07],
                    [-0.05, -0.05, -0.02],
                ],
                "R": [
                    [-0.23, -0.09, -0.09],
                    [0.00, 0.16, 0.01],
                    [0.03, 0.19, 0.03],
                    [0.10, -0.12, -0.02],
                    [0.09, -0.01, 0.01],
                    [0.01, -0.02, -0.02],
                ],
                "S": [
                    [0.32, 0.30, -0.17],
                    [0.26, 0.10, 0.03],
                    [0.11, 0.10, -0.24],
                    [0.08, 0.14, -0.10],
                    [-0.03, 0.07, 0.13],
                    [0.01, 0.02, 0.04],
                ],
                "T": [
                    [0.35, 0.22, -0.54],
                    [0.38, 0.10, 0.02],
                    [0.07, 0.02, -0.03],
                    [0.14, 0.14, 0.01],
                    [0.02, 0.04, 0.09],
                    [0.05, 0.08, 0.03],
                ],
                "V": [
                    [0.16, -0.05, 0.25],
                    [0.05, -0.08, 0.26],
                    [0.12, 0.05, 0.07],
                    [-0.11, 0.18, 0.05],
                    [0.01, 0.01, -0.17],
                    [0.05, 0.05, 0.03],
                ],
                "W": [
                    [0.06, -0.59, 0.06],
                    [-0.46, -0.33, -0.00],
                    [0.02, -0.06, 0.14],
                    [0.28, -0.03, 0.07],
                    [-0.16, -0.10, -0.08],
                    [-0.05, -0.03, 0.01],
                ],
                "Y": [
                    [-0.42, -0.48, 0.03],
                    [-0.29, -0.51, -0.19],
                    [-0.12, 0.10, 0.05],
                    [0.30, 0.00, 0.07],
                    [-0.09, -0.07, -0.03],
                    [0.09, -0.06, -0.03],
                ],
            }

        # Wang's neighboring residue correction, no next residue correction for CA and HA when Pro is i+1
        elif self.neighbor_flag == 0 and self.nonextpro == 2:

            self.preceed_res_effect = {
                "A": [
                    [-2.29, -2.21, -1.60],
                    [0.05, 0.14, 0.11],
                    [-0.03, -0.01, -0.13],
                    [0.06, -0.04, 0.01],
                    [-0.02, -0.07, -0.08],
                    [0.01, -0.01, -0.00],
                ],
                "C": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "B": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "D": [
                    [-2.06, -0.43, 0.43],
                    [0.01, 0.07, -0.21],
                    [0.22, 0.20, -0.12],
                    [-0.20, -0.07, 0.20],
                    [-0.22, -0.01, -0.05],
                    [-0.05, -0.03, 0.03],
                ],
                "E": [
                    [-0.15, -0.36, -0.13],
                    [0.01, 0.04, -0.02],
                    [0.12, 0.01, -0.23],
                    [-0.00, 0.04, 0.02],
                    [0.01, -0.01, -0.04],
                    [-0.05, -0.01, 0.03],
                ],
                "F": [
                    [-1.01, 0.20, -0.60],
                    [-0.18, -0.14, 0.05],
                    [-0.22, 0.02, 0.13],
                    [0.26, -0.06, -0.04],
                    [0.12, 0.01, 0.16],
                    [0.06, -0.01, -0.09],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [-0.34, -0.05, -0.27],
                    [0.03, -0.13, -0.40],
                    [0.14, 0.16, 0.01],
                    [0.05, 0.07, 0.22],
                    [-0.02, -0.01, 0.02],
                    [-0.05, -0.05, 0.00],
                ],
                "I": [
                    [3.16, 2.92, 0.23],
                    [0.12, 0.11, 0.12],
                    [-0.17, -0.15, 0.43],
                    [-0.04, 0.26, -0.07],
                    [0.15, 0.12, 0.02],
                    [0.05, 0.04, -0.03],
                ],
                "K": [
                    [-0.12, -0.26, -0.27],
                    [0.08, -0.09, -0.18],
                    [0.10, -0.07, -0.26],
                    [-0.08, 0.08, -0.02],
                    [-0.03, -0.02, -0.10],
                    [-0.00, -0.00, 0.03],
                ],
                "L": [
                    [0.27, -0.76, -0.74],
                    [0.07, 0.13, 0.12],
                    [-0.21, -0.07, 0.14],
                    [0.12, -0.03, -0.01],
                    [0.12, -0.06, 0.00],
                    [0.04, 0.00, -0.03],
                ],
                "M": [
                    [0.01, 0.69, -0.48],
                    [-0.19, 0.10, -0.13],
                    [-0.01, 0.09, 0.05],
                    [0.05, 0.10, -0.08],
                    [0.11, 0.05, -0.03],
                    [0.06, 0.05, -0.00],
                ],
                "N": [
                    [-1.59, -0.76, 1.09],
                    [-0.20, -0.19, -0.53],
                    [0.11, 0.18, -0.18],
                    [0.15, -0.20, 0.08],
                    [-0.24, 0.01, 0.03],
                    [-0.02, -0.03, 0.05],
                ],
                "P": [
                    [-2.05, -0.94, -0.74],
                    [0.18, 0.21, 0.04],
                    [0.14, 0.07, 0.04],
                    [-0.22, -0.16, -0.11],
                    [-0.17, 0.14, 0.08],
                    [-0.16, -0.04, 0.00],
                ],
                "Q": [
                    [-0.17, -0.09, 0.04],
                    [0.02, 0.07, -0.04],
                    [0.15, 0.13, -0.09],
                    [-0.21, 0.06, 0.15],
                    [-0.05, 0.01, -0.04],
                    [-0.01, -0.00, 0.03],
                ],
                "R": [
                    [-0.68, -0.45, -0.52],
                    [0.21, -0.07, 0.01],
                    [0.19, 0.00, -0.17],
                    [-0.12, 0.01, 0.03],
                    [-0.11, -0.03, -0.04],
                    [-0.00, 0.01, 0.06],
                ],
                "S": [
                    [-0.52, 1.16, 2.18],
                    [-0.08, -0.10, -0.34],
                    [0.18, 0.11, -0.34],
                    [0.14, -0.06, -0.01],
                    [-0.05, 0.02, 0.05],
                    [0.01, 0.01, 0.07],
                ],
                "T": [
                    [1.74, 1.23, 2.07],
                    [-0.14, -0.07, -0.20],
                    [0.06, 0.05, 0.10],
                    [-0.14, -0.11, -0.16],
                    [0.08, 0.02, 0.12],
                    [-0.01, -0.00, 0.03],
                ],
                "V": [
                    [2.80, 2.77, 0.17],
                    [0.04, -0.00, 0.13],
                    [-0.13, -0.16, 0.27],
                    [-0.17, 0.03, -0.14],
                    [0.14, 0.17, 0.01],
                    [0.02, 0.04, -0.03],
                ],
                "W": [
                    [-1.06, 0.97, -0.31],
                    [-0.02, -0.46, 0.15],
                    [-0.14, 0.00, 0.19],
                    [0.61, 0.28, -0.25],
                    [0.26, -0.08, 0.20],
                    [0.02, -0.05, -0.13],
                ],
                "Y": [
                    [-0.60, 0.46, -0.67],
                    [-0.08, -0.03, 0.05],
                    [-0.21, -0.19, 0.13],
                    [0.22, -0.10, -0.12],
                    [0.13, 0.02, 0.11],
                    [0.00, 0.01, -0.11],
                ],
            }

            self.nAext_res_effect = {
                "A": [
                    [-0.41, -0.11, 0.06],
                    [-0.12, 0.05, -0.07],
                    [0.03, 0.07, -0.04],
                    [0.05, -0.09, -0.11],
                    [-0.02, -0.01, 0.05],
                    [-0.03, -0.03, -0.01],
                ],
                "C": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "B": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "D": [
                    [-0.10, 0.23, 0.12],
                    [0.06, -0.11, -0.26],
                    [0.03, 0.28, 0.09],
                    [0.08, 0.11, -0.07],
                    [0.04, 0.04, 0.12],
                    [-0.06, -0.03, -0.05],
                ],
                "E": [
                    [-0.13, 0.26, 0.25],
                    [0.13, 0.14, 0.07],
                    [0.18, 0.25, 0.21],
                    [0.15, 0.06, -0.13],
                    [0.02, 0.07, 0.13],
                    [-0.04, -0.04, -0.05],
                ],
                "F": [
                    [-0.36, -0.35, 0.34],
                    [-0.23, -0.22, 0.04],
                    [-0.12, -0.04, 0.05],
                    [0.10, -0.09, 0.07],
                    [-0.11, -0.04, -0.10],
                    [0.08, -0.02, -0.03],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.45, -0.09, -0.28],
                    [-0.20, -0.05, -0.21],
                    [0.08, 0.22, -0.04],
                    [-0.03, -0.24, 0.09],
                    [-0.02, -0.01, -0.02],
                    [-0.04, -0.09, -0.05],
                ],
                "I": [
                    [-0.01, -0.20, 0.14],
                    [-0.04, -0.09, 0.17],
                    [0.10, 0.03, 0.20],
                    [0.05, 0.28, 0.11],
                    [0.01, -0.01, -0.21],
                    [0.06, 0.03, 0.01],
                ],
                "K": [
                    [0.12, -0.13, -0.05],
                    [0.09, -0.13, -0.16],
                    [0.22, 0.08, -0.11],
                    [-0.06, 0.01, -0.05],
                    [0.07, -0.04, 0.05],
                    [-0.04, -0.03, -0.00],
                ],
                "L": [
                    [-0.01, -0.49, 0.30],
                    [-0.27, 0.06, 0.11],
                    [0.09, 0.10, 0.01],
                    [-0.03, -0.10, 0.10],
                    [0.04, -0.07, -0.14],
                    [0.01, -0.02, 0.03],
                ],
                "M": [
                    [0.34, -0.02, 0.10],
                    [-0.05, 0.19, -0.08],
                    [0.23, 0.22, 0.14],
                    [0.18, 0.06, 0.05],
                    [0.04, 0.01, -0.06],
                    [0.00, -0.01, -0.01],
                ],
                "N": [
                    [-0.07, -0.03, -0.12],
                    [-0.01, -0.23, -0.34],
                    [0.21, 0.24, -0.14],
                    [-0.21, -0.06, 0.04],
                    [-0.02, 0.04, 0.11],
                    [-0.13, -0.04, -0.03],
                ],
                "P": [
                    [0.89, 0.92, 0.33],
                    [-0.92, -1.19, -2.49],
                    [0, 0, 0],
                    [-0.94, -0.20, -1.70],
                    [-0.07, -0.17, 0.04],
                    [0, 0, 0],
                ],
                "Q": [
                    [0.03, -0.31, -0.07],
                    [0.04, 0.10, 0.10],
                    [0.20, 0.37, 0.00],
                    [0.01, -0.13, -0.04],
                    [0.01, -0.13, -0.04],
                    [0.04, 0.01, 0.07],
                    [-0.05, -0.05, -0.02],
                ],
                "R": [
                    [-0.23, -0.09, -0.09],
                    [0.00, 0.16, 0.01],
                    [0.03, 0.19, 0.03],
                    [0.10, -0.12, -0.02],
                    [0.09, -0.01, 0.01],
                    [0.01, -0.02, -0.02],
                ],
                "S": [
                    [0.32, 0.30, -0.17],
                    [0.26, 0.10, 0.03],
                    [0.11, 0.10, -0.24],
                    [0.08, 0.14, -0.10],
                    [-0.03, 0.07, 0.13],
                    [0.01, 0.02, 0.04],
                ],
                "T": [
                    [0.35, 0.22, -0.54],
                    [0.38, 0.10, 0.02],
                    [0.07, 0.02, -0.03],
                    [0.14, 0.14, 0.01],
                    [0.02, 0.04, 0.09],
                    [0.05, 0.08, 0.03],
                ],
                "V": [
                    [0.16, -0.05, 0.25],
                    [0.05, -0.08, 0.26],
                    [0.12, 0.05, 0.07],
                    [-0.11, 0.18, 0.05],
                    [0.01, 0.01, -0.17],
                    [0.05, 0.05, 0.03],
                ],
                "W": [
                    [0.06, -0.59, 0.06],
                    [-0.46, -0.33, -0.00],
                    [0.02, -0.06, 0.14],
                    [0.28, -0.03, 0.07],
                    [-0.16, -0.10, -0.08],
                    [-0.05, -0.03, 0.01],
                ],
                "Y": [
                    [-0.42, -0.48, 0.03],
                    [-0.29, -0.51, -0.19],
                    [-0.12, 0.10, 0.05],
                    [0.30, 0.00, 0.07],
                    [-0.09, -0.07, -0.03],
                    [0.09, -0.06, -0.03],
                ],
            }

        # Wang's neighboring residue correction, -0.77 next residue correction for CA when Pro is i+1
        elif self.neighbor_flag == 0 and self.nonextpro == 3:

            self.preceed_res_effect = {
                "A": [
                    [-2.29, -2.21, -1.60],
                    [0.05, 0.14, 0.11],
                    [-0.03, -0.01, -0.13],
                    [0.06, -0.04, 0.01],
                    [-0.02, -0.07, -0.08],
                    [0.01, -0.01, -0.00],
                ],
                "C": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "B": [
                    [0.80, 1.36, 0.67],
                    [0.08, -0.00, -0.05],
                    [-0.03, 0.44, 0.40],
                    [0.28, -0.18, 0.19],
                    [0.22, 0.03, 0.12],
                    [0.06, -0.00, -0.04],
                ],
                "D": [
                    [-2.06, -0.43, 0.43],
                    [0.01, 0.07, -0.21],
                    [0.22, 0.20, -0.12],
                    [-0.20, -0.07, 0.20],
                    [-0.22, -0.01, -0.05],
                    [-0.05, -0.03, 0.03],
                ],
                "E": [
                    [-0.15, -0.36, -0.13],
                    [0.01, 0.04, -0.02],
                    [0.12, 0.01, -0.23],
                    [-0.00, 0.04, 0.02],
                    [0.01, -0.01, -0.04],
                    [-0.05, -0.01, 0.03],
                ],
                "F": [
                    [-1.01, 0.20, -0.60],
                    [-0.18, -0.14, 0.05],
                    [-0.22, 0.02, 0.13],
                    [0.26, -0.06, -0.04],
                    [0.12, 0.01, 0.16],
                    [0.06, -0.01, -0.09],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [-0.34, -0.05, -0.27],
                    [0.03, -0.13, -0.40],
                    [0.14, 0.16, 0.01],
                    [0.05, 0.07, 0.22],
                    [-0.02, -0.01, 0.02],
                    [-0.05, -0.05, 0.00],
                ],
                "I": [
                    [3.16, 2.92, 0.23],
                    [0.12, 0.11, 0.12],
                    [-0.17, -0.15, 0.43],
                    [-0.04, 0.26, -0.07],
                    [0.15, 0.12, 0.02],
                    [0.05, 0.04, -0.03],
                ],
                "K": [
                    [-0.12, -0.26, -0.27],
                    [0.08, -0.09, -0.18],
                    [0.10, -0.07, -0.26],
                    [-0.08, 0.08, -0.02],
                    [-0.03, -0.02, -0.10],
                    [-0.00, -0.00, 0.03],
                ],
                "L": [
                    [0.27, -0.76, -0.74],
                    [0.07, 0.13, 0.12],
                    [-0.21, -0.07, 0.14],
                    [0.12, -0.03, -0.01],
                    [0.12, -0.06, 0.00],
                    [0.04, 0.00, -0.03],
                ],
                "M": [
                    [0.01, 0.69, -0.48],
                    [-0.19, 0.10, -0.13],
                    [-0.01, 0.09, 0.05],
                    [0.05, 0.10, -0.08],
                    [0.11, 0.05, -0.03],
                    [0.06, 0.05, -0.00],
                ],
                "N": [
                    [-1.59, -0.76, 1.09],
                    [-0.20, -0.19, -0.53],
                    [0.11, 0.18, -0.18],
                    [0.15, -0.20, 0.08],
                    [-0.24, 0.01, 0.03],
                    [-0.02, -0.03, 0.05],
                ],
                "P": [
                    [-2.05, -0.94, -0.74],
                    [0.18, 0.21, 0.04],
                    [0.14, 0.07, 0.04],
                    [-0.22, -0.16, -0.11],
                    [-0.17, 0.14, 0.08],
                    [-0.16, -0.04, 0.00],
                ],
                "Q": [
                    [-0.17, -0.09, 0.04],
                    [0.02, 0.07, -0.04],
                    [0.15, 0.13, -0.09],
                    [-0.21, 0.06, 0.15],
                    [-0.05, 0.01, -0.04],
                    [-0.01, -0.00, 0.03],
                ],
                "R": [
                    [-0.68, -0.45, -0.52],
                    [0.21, -0.07, 0.01],
                    [0.19, 0.00, -0.17],
                    [-0.12, 0.01, 0.03],
                    [-0.11, -0.03, -0.04],
                    [-0.00, 0.01, 0.06],
                ],
                "S": [
                    [-0.52, 1.16, 2.18],
                    [-0.08, -0.10, -0.34],
                    [0.18, 0.11, -0.34],
                    [0.14, -0.06, -0.01],
                    [-0.05, 0.02, 0.05],
                    [0.01, 0.01, 0.07],
                ],
                "T": [
                    [1.74, 1.23, 2.07],
                    [-0.14, -0.07, -0.20],
                    [0.06, 0.05, 0.10],
                    [-0.14, -0.11, -0.16],
                    [0.08, 0.02, 0.12],
                    [-0.01, -0.00, 0.03],
                ],
                "V": [
                    [2.80, 2.77, 0.17],
                    [0.04, -0.00, 0.13],
                    [-0.13, -0.16, 0.27],
                    [-0.17, 0.03, -0.14],
                    [0.14, 0.17, 0.01],
                    [0.02, 0.04, -0.03],
                ],
                "W": [
                    [-1.06, 0.97, -0.31],
                    [-0.02, -0.46, 0.15],
                    [-0.14, 0.00, 0.19],
                    [0.61, 0.28, -0.25],
                    [0.26, -0.08, 0.20],
                    [0.02, -0.05, -0.13],
                ],
                "Y": [
                    [-0.60, 0.46, -0.67],
                    [-0.08, -0.03, 0.05],
                    [-0.21, -0.19, 0.13],
                    [0.22, -0.10, -0.12],
                    [0.13, 0.02, 0.11],
                    [0.00, 0.01, -0.11],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.41, -0.11, 0.06],
                    [-0.12, 0.05, -0.07],
                    [0.03, 0.07, -0.04],
                    [0.05, -0.09, -0.11],
                    [-0.02, -0.01, 0.05],
                    [-0.03, -0.03, -0.01],
                ],
                "C": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "B": [
                    [-0.85, -1.17, -0.16],
                    [-0.18, 0.10, -0.07],
                    [-0.10, 0.17, 0.09],
                    [0.25, 0.21, 0.37],
                    [-0.05, 0.01, 0.04],
                    [0.06, 0.03, 0.04],
                ],
                "D": [
                    [-0.10, 0.23, 0.12],
                    [0.06, -0.11, -0.26],
                    [0.03, 0.28, 0.09],
                    [0.08, 0.11, -0.07],
                    [0.04, 0.04, 0.12],
                    [-0.06, -0.03, -0.05],
                ],
                "E": [
                    [-0.13, 0.26, 0.25],
                    [0.13, 0.14, 0.07],
                    [0.18, 0.25, 0.21],
                    [0.15, 0.06, -0.13],
                    [0.02, 0.07, 0.13],
                    [-0.04, -0.04, -0.05],
                ],
                "F": [
                    [-0.36, -0.35, 0.34],
                    [-0.23, -0.22, 0.04],
                    [-0.12, -0.04, 0.05],
                    [0.10, -0.09, 0.07],
                    [-0.11, -0.04, -0.10],
                    [0.08, -0.02, -0.03],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.45, -0.09, -0.28],
                    [-0.20, -0.05, -0.21],
                    [0.08, 0.22, -0.04],
                    [-0.03, -0.24, 0.09],
                    [-0.02, -0.01, -0.02],
                    [-0.04, -0.09, -0.05],
                ],
                "I": [
                    [-0.01, -0.20, 0.14],
                    [-0.04, -0.09, 0.17],
                    [0.10, 0.03, 0.20],
                    [0.05, 0.28, 0.11],
                    [0.01, -0.01, -0.21],
                    [0.06, 0.03, 0.01],
                ],
                "K": [
                    [0.12, -0.13, -0.05],
                    [0.09, -0.13, -0.16],
                    [0.22, 0.08, -0.11],
                    [-0.06, 0.01, -0.05],
                    [0.07, -0.04, 0.05],
                    [-0.04, -0.03, -0.00],
                ],
                "L": [
                    [-0.01, -0.49, 0.30],
                    [-0.27, 0.06, 0.11],
                    [0.09, 0.10, 0.01],
                    [-0.03, -0.10, 0.10],
                    [0.04, -0.07, -0.14],
                    [0.01, -0.02, 0.03],
                ],
                "M": [
                    [0.34, -0.02, 0.10],
                    [-0.05, 0.19, -0.08],
                    [0.23, 0.22, 0.14],
                    [0.18, 0.06, 0.05],
                    [0.04, 0.01, -0.06],
                    [0.00, -0.01, -0.01],
                ],
                "N": [
                    [-0.07, -0.03, -0.12],
                    [-0.01, -0.23, -0.34],
                    [0.21, 0.24, -0.14],
                    [-0.21, -0.06, 0.04],
                    [-0.02, 0.04, 0.11],
                    [-0.13, -0.04, -0.03],
                ],
                "P": [
                    [0.89, 0.92, 0.33],
                    [-0.92, -1.19, -2.49],
                    [-0.77, -0.77, -0.77],
                    [-0.94, -0.20, -1.70],
                    [-0.07, -0.17, 0.04],
                    [-0.07, 0.21, 0.18],
                ],
                "Q": [
                    [0.03, -0.31, -0.07],
                    [0.04, 0.10, 0.10],
                    [0.20, 0.37, 0.00],
                    [0.01, -0.13, -0.04],
                    [0.01, -0.13, -0.04],
                    [0.04, 0.01, 0.07],
                    [-0.05, -0.05, -0.02],
                ],
                "R": [
                    [-0.23, -0.09, -0.09],
                    [0.00, 0.16, 0.01],
                    [0.03, 0.19, 0.03],
                    [0.10, -0.12, -0.02],
                    [0.09, -0.01, 0.01],
                    [0.01, -0.02, -0.02],
                ],
                "S": [
                    [0.32, 0.30, -0.17],
                    [0.26, 0.10, 0.03],
                    [0.11, 0.10, -0.24],
                    [0.08, 0.14, -0.10],
                    [-0.03, 0.07, 0.13],
                    [0.01, 0.02, 0.04],
                ],
                "T": [
                    [0.35, 0.22, -0.54],
                    [0.38, 0.10, 0.02],
                    [0.07, 0.02, -0.03],
                    [0.14, 0.14, 0.01],
                    [0.02, 0.04, 0.09],
                    [0.05, 0.08, 0.03],
                ],
                "V": [
                    [0.16, -0.05, 0.25],
                    [0.05, -0.08, 0.26],
                    [0.12, 0.05, 0.07],
                    [-0.11, 0.18, 0.05],
                    [0.01, 0.01, -0.17],
                    [0.05, 0.05, 0.03],
                ],
                "W": [
                    [0.06, -0.59, 0.06],
                    [-0.46, -0.33, -0.00],
                    [0.02, -0.06, 0.14],
                    [0.28, -0.03, 0.07],
                    [-0.16, -0.10, -0.08],
                    [-0.05, -0.03, 0.01],
                ],
                "Y": [
                    [-0.42, -0.48, 0.03],
                    [-0.29, -0.51, -0.19],
                    [-0.12, 0.10, 0.05],
                    [0.30, 0.00, 0.07],
                    [-0.09, -0.07, -0.03],
                    [0.09, -0.06, -0.03],
                ],
            }

        elif self.neighbor_flag == 1 and self.nonextpro == 0:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.00, 0.00, 0.00],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.00, 0.00, 0.00],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [-2.00, -2.00, -2.00],
                    [0.00, 0.00, 0.00],
                    [-0.18, -0.18, -0.18],
                    [0.11, 0.11, 0.11],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        elif self.neighbor_flag == 1 and self.nonextpro == 1:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.00, 0.00, 0.00],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.00, 0.00, 0.00],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        elif self.neighbor_flag == 1 and self.nonextpro == 2:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.00, 0.00, 0.00],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.00, 0.00, 0.00],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [0, 0, 0],
                    [0.00, 0.00, 0.00],
                    [-0.18, -0.18, -0.18],
                    [0, 0, 0],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        # Original unusued value of -0.77 for Pro for next residue correction is used
        elif self.neighbor_flag == 1 and self.nonextpro == 3:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.00, 0.00, 0.00],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.00, 0.00, 0.00],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [-0.77, -0.77, -0.77],
                    [0.00, 0.00, 0.00],
                    [-0.18, -0.18, -0.18],
                    [0.11, 0.11, 0.11],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.sw_neighbor_flag == 1 and self.nonextpro == 0:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.06, -0.04, 0.01],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [-0.12, 0.01, 0.03],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.15, -0.20, 0.08],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [-0.20, -0.07, 0.20],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [-0.21, 0.06, 0.15],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [-0.00, 0.04, 0.02],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.05, 0.07, 0.22],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [-0.04, 0.26, -0.07],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.12, -0.03, -0.01],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.08, 0.08, -0.02],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.05, 0.10, -0.08],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.26, -0.06, -0.04],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [-0.22, -0.16, -0.11],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.14, -0.06, -0.01],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [-0.14, -0.11, -0.16],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.61, 0.28, -0.25],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.22, -0.10, -0.12],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.17, 0.03, -0.14],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.05, -0.09, -0.11],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.10, -0.12, -0.02],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [-0.21, -0.06, 0.04],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.08, 0.11, -0.07],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.01, -0.13, -0.04],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.15, 0.06, -0.13],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [-0.03, -0.24, 0.09],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.05, 0.28, 0.11],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [-0.03, -0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [-0.06, 0.01, -0.05],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.18, 0.06, 0.05],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.10, -0.09, 0.07],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [-2.00, -2.00, -2.00],
                    [-0.94, -0.20, -1.70],
                    [-0.18, -0.18, -0.18],
                    [0.11, 0.11, 0.11],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.08, 0.14, -0.10],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.14, 0.14, 0.01],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.28, -0.03, 0.07],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.30, 0.00, 0.07],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [-0.11, 0.18, 0.05],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.sw_neighbor_flag == 1 and self.nonextpro == 1:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.06, -0.04, 0.01],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [-0.12, 0.01, 0.03],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.15, -0.20, 0.08],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [-0.20, -0.07, 0.20],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [-0.21, 0.06, 0.15],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [-0.00, 0.04, 0.02],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.05, 0.07, 0.22],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [-0.04, 0.26, -0.07],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.12, -0.03, -0.01],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.08, 0.08, -0.02],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.05, 0.10, -0.08],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.26, -0.06, -0.04],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [-0.22, -0.16, -0.11],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.14, -0.06, -0.01],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [-0.14, -0.11, -0.16],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.61, 0.28, -0.25],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.22, -0.10, -0.12],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.17, 0.03, -0.14],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.05, -0.09, -0.11],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.10, -0.12, -0.02],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [-0.21, -0.06, 0.04],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.08, 0.11, -0.07],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.01, -0.13, -0.04],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.15, 0.06, -0.13],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [-0.03, -0.24, 0.09],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.05, 0.28, 0.11],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [-0.03, -0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [-0.06, 0.01, -0.05],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.18, 0.06, 0.05],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.10, -0.09, 0.07],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.08, 0.14, -0.10],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.14, 0.14, 0.01],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.28, -0.03, 0.07],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.30, 0.00, 0.07],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [-0.11, 0.18, 0.05],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.sw_neighbor_flag == 1 and self.nonextpro == 2:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.06, -0.04, 0.01],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [-0.12, 0.01, 0.03],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.15, -0.20, 0.08],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [-0.20, -0.07, 0.20],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [-0.21, 0.06, 0.15],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [-0.00, 0.04, 0.02],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.05, 0.07, 0.22],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [-0.04, 0.26, -0.07],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.12, -0.03, -0.01],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.08, 0.08, -0.02],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.05, 0.10, -0.08],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.26, -0.06, -0.04],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [-0.22, -0.16, -0.11],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.14, -0.06, -0.01],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [-0.14, -0.11, -0.16],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.61, 0.28, -0.25],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.22, -0.10, -0.12],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.17, 0.03, -0.14],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.05, -0.09, -0.11],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.10, -0.12, -0.02],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [-0.21, -0.06, 0.04],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.08, 0.11, -0.07],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.01, -0.13, -0.04],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.15, 0.06, -0.13],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [-0.03, -0.24, 0.09],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.05, 0.28, 0.11],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [-0.03, -0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [-0.06, 0.01, -0.05],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.18, 0.06, 0.05],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.10, -0.09, 0.07],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [0, 0, 0],
                    [-0.94, -0.20, -1.70],
                    [-0.18, -0.18, -0.18],
                    [0, 0, 0],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.08, 0.14, -0.10],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.14, 0.14, 0.01],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.28, -0.03, 0.07],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.30, 0.00, 0.07],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [-0.11, 0.18, 0.05],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.sw_neighbor_flag == 1 and self.nonextpro == 3:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.06, -0.04, 0.01],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [-0.12, 0.01, 0.03],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.15, -0.20, 0.08],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [-0.20, -0.07, 0.20],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.28, -0.18, 0.19],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [-0.21, 0.06, 0.15],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [-0.00, 0.04, 0.02],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-2.45, -0.43, 1.86],
                    [-0.07, -0.09, -0.25],
                    [0.13, -0.26, -0.32],
                    [-0.16, 0.17, 0.14],
                    [-0.41, -0.10, 0.01],
                    [-0.04, 0.04, 0.06],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.05, 0.07, 0.22],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [-0.04, 0.26, -0.07],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.12, -0.03, -0.01],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.08, 0.08, -0.02],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.05, 0.10, -0.08],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.26, -0.06, -0.04],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [
                    [0.87, 0.87, 0.87],
                    [-0.09, -0.09, -0.09],
                    [0.02, 0.02, 0.02],
                    [-0.22, -0.16, -0.11],
                    [0.10, 0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                ],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.14, -0.06, -0.01],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [-0.14, -0.11, -0.16],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.61, 0.28, -0.25],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.22, -0.10, -0.12],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [-0.17, 0.03, -0.14],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.05, -0.09, -0.11],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.10, -0.12, -0.02],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [-0.21, -0.06, 0.04],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.08, 0.11, -0.07],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.25, 0.21, 0.37],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.01, -0.13, -0.04],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.15, 0.06, -0.13],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [-0.13, 0.13, -1.20],
                    [0.61, 0.47, -0.41],
                    [0.09, 0.12, -1.09],
                    [-0.20, -0.07, 0.14],
                    [-0.02, 0.06, -0.05],
                    [-0.08, -0.03, 0.12],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [-0.03, -0.24, 0.09],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.05, 0.28, 0.11],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [-0.03, -0.10, 0.10],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [-0.06, 0.01, -0.05],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.18, 0.06, 0.05],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.10, -0.09, 0.07],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [-0.77, -0.77, -0.77],
                    [-0.94, -0.20, -1.70],
                    [-0.18, -0.18, -0.18],
                    [0.11, 0.11, 0.11],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.08, 0.14, -0.10],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.14, 0.14, 0.01],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.28, -0.03, 0.07],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.30, 0.00, 0.07],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [-0.11, 0.18, 0.05],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        elif self.neighbor_flag == 1 and self.nopreceedpro == 1:

            self.preceed_res_effect = {
                "A": [
                    [-0.57, -0.57, -0.57],
                    [-0.07, -0.07, -0.07],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.07, 0.07, 0.07],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [1.62, 1.62, 1.62],
                    [-0.19, -0.19, -0.19],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [0.87, 0.87, 0.87],
                    [-0.10, -0.10, -0.10],
                    [0.23, 0.23, 0.23],
                    [0.00, 0.00, 0.00],
                    [0.13, 0.13, 0.13],
                    [-0.02, -0.02, -0.02],
                ],
                "D": [
                    [0.86, 0.86, 0.86],
                    [-0.13, -0.13, -0.13],
                    [0.25, 0.25, 0.25],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.02, -0.02, 0.02],
                ],
                "C": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [3.07, 3.07, 3.07],
                    [-0.28, -0.28, -0.28],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [1.62, 1.62, 1.62],
                    [-0.18, -0.18, -0.18],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [1.51, 1.51, 1.51],
                    [-0.20, -0.20, -0.20],
                    [0.05, 0.05, 0.05],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [1.68, 1.68, 1.68],
                    [-0.22, -0.22, -0.22],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [0.20, 0.20, 0.20],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [4.87, 4.87, 4.87],
                    [-0.18, -0.18, -0.18],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [1.05, 1.05, 1.05],
                    [-0.13, -0.13, -0.13],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.05, -0.05, -0.05],
                ],
                "K": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, -0.01],
                ],
                "M": [
                    [1.57, 1.57, 1.57],
                    [-0.18, -0.18, -0.18],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [-0.01, -0.01, 0.01],
                ],
                "F": [
                    [2.78, 2.78, 2.78],
                    [-0.25, -0.25, -0.25],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.15, 0.15, 0.15],
                    [-0.08, -0.08, -0.08],
                ],
                "P": [[0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0], [0, 0, 0]],
                "S": [
                    [2.55, 2.55, 2.55],
                    [-0.15, -0.15, -0.15],
                    [0.13, 0.13, 0.13],
                    [0.00, 0.00, 0.00],
                    [0.19, 0.19, 0.19],
                    [0.00, 0.00, 0.00],
                ],
                "T": [
                    [2.78, 2.78, 2.78],
                    [-0.13, -0.13, -0.13],
                    [0.12, 0.12, 0.12],
                    [0.00, 0.00, 0.00],
                    [0.14, 0.14, 0.14],
                    [0.00, 0.00, 0.00],
                ],
                "W": [
                    [3.19, 3.19, 3.19],
                    [-0.30, -0.30, -0.30],
                    [0.03, 0.03, 0.03],
                    [0.00, 0.00, 0.00],
                    [0.04, 0.04, 0.04],
                    [-0.15, -0.15, -0.15],
                ],
                "Y": [
                    [3.01, 3.01, 3.01],
                    [-0.24, -0.24, -0.24],
                    [0.06, 0.06, 0.06],
                    [0.00, 0.00, 0.00],
                    [0.09, 0.09, 0.09],
                    [-0.08, -0.08, -0.08],
                ],
                "V": [
                    [4.34, 4.34, 4.34],
                    [-0.18, -0.18, -0.18],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [0.17, 0.17, 0.17],
                    [-0.02, -0.02, -0.02],
                ],
            }

            self.next_res_effect = {
                "A": [
                    [-0.33, -0.33, -0.33],
                    [-0.77, -0.77, -0.77],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.03, -0.03, -0.03],
                ],
                "R": [
                    [-0.14, -0.14, -0.14],
                    [-0.49, -0.49, -0.49],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.26, -0.26, -0.26],
                    [-0.66, -0.66, -0.66],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.20, -0.20, -0.20],
                    [-0.58, -0.58, -0.58],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "B": [
                    [-0.26, -0.26, -0.26],
                    [-0.51, -0.51, -0.51],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                ],
                "Q": [
                    [-0.14, -0.14, -0.14],
                    [-0.48, -0.48, -0.48],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "E": [
                    [-0.20, -0.20, -0.20],
                    [-0.48, -0.48, -0.48],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.55, -0.55, -0.55],
                    [-0.65, -0.65, -0.65],
                    [-0.09, -0.09, -0.09],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.06, -0.06, -0.06],
                ],
                "I": [
                    [-0.14, -0.14, -0.14],
                    [-0.58, -0.58, -0.58],
                    [0.20, 0.20, 0.20],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                ],
                "L": [
                    [-0.14, -0.14, -0.14],
                    [-0.50, -0.50, -0.50],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.03, -0.03, -0.03],
                ],
                "K": [
                    [-0.20, -0.20, -0.20],
                    [-0.50, -0.50, -0.50],
                    [-0.11, -0.11, -0.11],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.20, -0.20, -0.20],
                    [-0.41, -0.41, -0.41],
                    [0.10, 0.10, 0.10],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.01, -0.01, -0.01],
                ],
                "F": [
                    [-0.49, -0.49, -0.49],
                    [-0.83, -0.83, -0.83],
                    [-0.23, -0.23, -0.23],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.09, -0.09, -0.09],
                ],
                "P": [
                    [-0.32, -0.32, -0.32],
                    [-2.84, -2.84, -2.84],
                    [-2.00, -2.00, -2.00],
                    [0.00, 0.00, 0.00],
                    [-0.18, -0.18, -0.18],
                    [0.11, 0.11, 0.11],
                ],
                "S": [
                    [-0.03, -0.03, -0.03],
                    [-0.40, -0.40, -0.40],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                ],
                "T": [
                    [-0.03, -0.03, -0.03],
                    [-0.19, -0.19, -0.19],
                    [-0.04, -0.04, -0.04],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.05, 0.05, 0.05],
                ],
                "W": [
                    [-0.26, -0.26, -0.26],
                    [-0.85, -0.85, -0.85],
                    [-0.17, -0.17, -0.17],
                    [0.00, 0.00, 0.00],
                    [-0.13, -0.13, -0.13],
                    [-0.10, -0.10, -0.10],
                ],
                "Y": [
                    [-0.43, -0.43, -0.43],
                    [-0.85, -0.85, -0.85],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.10, -0.10, -0.10],
                ],
                "V": [
                    [-0.14, -0.14, -0.14],
                    [-0.57, -0.57, -0.57],
                    [-0.21, -0.21, -0.21],
                    [0.00, 0.00, 0.00],
                    [-0.05, -0.05, -0.05],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.preceed_preceed_res_effect_flag == 1:

            self.preceed_preceed_res_effect = {
                "A": [
                    [-0.15, -0.15, -0.15],
                    [-0.02, -0.02, -0.02],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.10, -0.10, -0.10],
                    [0.00, 0.00, 0.00],
                ],
                "R": [
                    [-0.06, -0.06, -0.06],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                ],
                "N": [
                    [-0.17, -0.17, -0.17],
                    [-0.03, -0.03, -0.03],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.29, -0.29, -0.29],
                    [-0.04, -0.04, -0.04],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [-0.11, -0.11, -0.11],
                    [-0.01, -0.01, -0.01],
                ],
                "C": [
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                ],
                "B": [
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                ],
                "Q": [
                    [-0.06, -0.06, -0.06],
                    [-0.03, -0.03, -0.03],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                ],
                "E": [
                    [-0.12, -0.12, -0.12],
                    [-0.03, -0.03, -0.03],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [0.17, 0.17, 0.17],
                    [-0.07, -0.07, -0.07],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.01, 0.01, 0.01],
                ],
                "I": [
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [-0.09, -0.09, -0.09],
                    [-0.01, -0.01, -0.01],
                ],
                "L": [
                    [-0.06, -0.06, -0.06],
                    [-0.01, -0.01, -0.01],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [-0.08, -0.08, -0.08],
                    [-0.01, -0.01, -0.01],
                ],
                "K": [
                    [-0.06, -0.06, -0.06],
                    [-0.03, -0.03, -0.03],
                    [0.02, 0.02, 0.02],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                ],
                "M": [
                    [-0.06, -0.06, -0.06],
                    [-0.02, -0.02, -0.02],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                ],
                "F": [
                    [-0.46, -0.46, -0.46],
                    [-0.10, -0.10, -0.10],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.37, -0.37, -0.37],
                    [-0.04, -0.04, -0.04],
                ],
                "P": [
                    [-0.17, -0.17, -0.17],
                    [-0.02, -0.02, -0.02],
                    [0.04, 0.04, 0.04],
                    [0.00, 0.00, 0.00],
                    [-0.12, -0.12, -0.12],
                    [-0.01, -0.01, -0.01],
                ],
                "S": [
                    [-0.17, -0.17, -0.17],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.08, -0.08, -0.08],
                    [-0.01, -0.01, -0.01],
                ],
                "T": [
                    [-0.12, -0.12, -0.12],
                    [-0.05, -0.05, -0.05],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.06, -0.06, -0.06],
                    [-0.01, -0.01, -0.01],
                ],
                "W": [
                    [-0.64, -0.64, -0.64],
                    [-0.17, -0.17, -0.17],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [-0.62, -0.62, -0.62],
                    [-0.16, -0.16, -0.16],
                ],
                "Y": [
                    [-0.52, -0.52, -0.52],
                    [-0.13, -0.13, -0.13],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [-0.42, -0.42, -0.42],
                    [-0.04, -0.04, -0.04],
                ],
                "V": [
                    [-0.06, -0.06, -0.06],
                    [-0.03, -0.03, -0.03],
                    [0.01, 0.01, 0.01],
                    [0.00, 0.00, 0.00],
                    [-0.08, -0.08, -0.08],
                    [-0.01, -0.01, -0.01],
                ],
            }

        if self.next_next_res_effect_flag == 1:

            self.next_next_res_effect = {
                "A": [
                    [-0.12, -0.12, -0.12],
                    [-0.11, -0.11, -0.11],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.02, -0.02, -0.02],
                ],
                "R": [
                    [-0.06, -0.06, -0.06],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                ],
                "N": [
                    [-0.18, -0.18, -0.18],
                    [-0.09, -0.09, -0.09],
                    [-0.06, -0.06, -0.06],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.01, -0.01, -0.01],
                ],
                "D": [
                    [-0.12, -0.12, -0.12],
                    [-0.08, -0.08, -0.08],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                    [-0.02, -0.02, -0.02],
                ],
                "C": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                ],
                "B": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [-0.03, -0.03, -0.03],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                ],
                "Q": [
                    [-0.06, -0.06, -0.06],
                    [-0.05, -0.05, -0.05],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.01, -0.01, -0.01],
                ],
                "E": [
                    [-0.06, -0.06, -0.06],
                    [-0.09, -0.09, -0.09],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.02, -0.02, -0.02],
                ],
                "G": [
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                ],
                "H": [
                    [-0.12, -0.12, -0.12],
                    [-0.10, -0.10, -0.10],
                    [-0.05, -0.05, -0.05],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.03, -0.03, -0.03],
                ],
                "I": [
                    [-0.18, -0.18, -0.18],
                    [-0.20, -0.20, -0.20],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.03, -0.03, -0.03],
                ],
                "L": [
                    [-0.06, -0.06, -0.06],
                    [-0.13, -0.13, -0.13],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                ],
                "K": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                ],
                "M": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.02, -0.02, -0.02],
                ],
                "F": [
                    [-0.18, -0.18, -0.18],
                    [-0.27, -0.27, -0.27],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.03, -0.03, -0.03],
                    [-0.06, -0.06, -0.06],
                ],
                "P": [
                    [-0.18, -0.18, -0.18],
                    [-0.47, -0.47, -0.47],
                    [-0.22, -0.22, -0.22],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.01, -0.01, -0.01],
                ],
                "S": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                ],
                "T": [
                    [-0.06, -0.06, -0.06],
                    [-0.08, -0.08, -0.08],
                    [-0.01, -0.01, -0.01],
                    [0.00, 0.00, 0.00],
                    [0.01, 0.01, 0.01],
                    [-0.01, -0.01, -0.01],
                ],
                "W": [
                    [0.00, 0.00, 0.00],
                    [-0.26, -0.26, -0.26],
                    [-0.02, -0.02, -0.02],
                    [0.00, 0.00, 0.00],
                    [-0.08, -0.08, -0.08],
                    [-0.08, -0.08, -0.08],
                ],
                "Y": [
                    [-0.24, -0.24, -0.24],
                    [-0.28, -0.28, -0.28],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.04, -0.04, -0.04],
                    [-0.05, -0.05, -0.05],
                ],
                "V": [
                    [-0.24, -0.24, -0.24],
                    [-0.20, -0.20, -0.20],
                    [-0.07, -0.07, -0.07],
                    [0.00, 0.00, 0.00],
                    [-0.01, -0.01, -0.01],
                    [-0.02, -0.02, -0.02],
                ],
            }

        if self.Random_coil_flag == 1:  # 1 - Wishart's random coil, 2 - Wang's random coil, 3 - Lukhin' random coil,  4 - Schwarzinger's, 5 - average of 1,2 & 3
            # Wishart random coil values
            self.random_coil_dic = {
                "I": [121.7, 176.4, 61.1, 38.8, 8.00, 4.17],
                "V": [120.5, 176.3, 62.2, 32.9, 8.03, 4.12],
                "D": [121.4, 176.3, 54.2, 41.1, 8.34, 4.64],
                "N": [119.0, 175.2, 53.1, 38.9, 8.40, 4.74],
                "F": [120.9, 175.8, 57.7, 39.6, 8.30, 4.62],
                "H": [118.2, 174.1, 55.0, 29.0, 8.42, 4.73],
                "W": [122.2, 176.1, 57.5, 29.60, 8.25, 4.66],
                "Y": [120.8, 175.9, 57.9, 38.8, 8.12, 4.55],
                "K": [121.6, 176.6, 56.2, 33.1, 8.29, 4.32],
                "L": [122.6, 177.6, 55.1, 42.4, 8.16, 4.34],
                "M": [120.7, 176.3, 55.4, 32.9, 8.28, 4.48],
                "Q": [120.6, 176.0, 55.7, 29.4, 8.32, 4.34],
                "R": [121.3, 176.3, 56.0, 30.9, 8.23, 4.34],
                "E": [121.7, 176.6, 56.6, 29.9, 8.42, 4.35],
                "T": [116.0, 174.7, 61.8, 69.8, 8.15, 4.35],
                "C": [118.9, 174.6, 58.2, 28.0, 8.43, 4.71],
                "S": [116.6, 174.6, 58.3, 63.8, 8.31, 4.47],
                "A": [125.0, 177.8, 52.5, 19.1, 8.24, 4.32],
                "G": [109.1, 174.9, 45.1, None, 8.33, 3.96],
                "P": [None, 177.3, 63.3, 32.1, None, 4.42],
                "B": [118.6, 174.6, 55.4, 41.1, 8.43, 4.71],
            }

        elif self.Random_coil_flag == 2:  # 1 - Wishart's random coil, 2 - Wang's random coil, 3 - Lukhin' random coil,  4 - Schwarzinger's, 5 - average of 1,2 & 3
            # Wang's random coil values
            self.random_coil_dic = {
                "I": [120.58, 175.52, 60.79, 38.43, 7.94, 4.18],
                "V": [119.91, 175.66, 62.00, 32.35, 7.98, 4.13],
                "D": [120.37, 176.00, 54.00, 40.78, 8.31, 4.62],
                "N": [118.50, 174.84, 53.00, 38.43, 8.35, 4.66],
                "F": [119.72, 175.46, 57.46, 39.41, 8.09, 4.59],
                "H": [118.92, 174.78, 55.74, 29.50, 8.18, 4.60],
                "W": [120.99, 175.87, 57.54, 29.60, 7.97, 4.60],
                "Y": [119.37, 175.29, 57.64, 38.78, 7.99, 4.56],
                "K": [121.10, 176.15, 56.29, 32.53, 8.17, 4.28],
                "L": [121.57, 176.70, 54.77, 42.14, 8.06, 4.36],
                "M": [120.14, 175.94, 55.43, 32.92, 8.22, 4.47],
                "Q": [119.82, 175.75, 55.89, 29.01, 8.20, 4.29],
                "R": [120.75, 176.01, 56.18, 30.36, 8.21, 4.26],
                "E": [120.62, 176.32, 56.66, 29.87, 8.36, 4.28],
                "T": [113.88, 174.78, 61.30, 68.92, 8.16, 4.44],
                "C": [118.10, 175.11, 58.24, 29.54, 8.10, 4.59],
                "S": [116.00, 174.41, 58.20, 63.75, 8.22, 4.45],
                "A": [123.82, 177.28, 52.46, 18.98, 8.09, 4.31],
                "G": [109.48, 174.01, 45.28, None, 8.37, 3.97],
                "P": [None, 176.62, 63.24, 31.81, None, 4.41],
                "B": [118.7, 175.5, 55.6, 41.2, 8.54, 4.76],
            }

        elif self.Random_coil_flag == 3:  # 1 - Wishart's random coil, 2 - Wang's random coil, 3 - Lukhin' random coil, 4 - Schwarzinger's, 5 - average of 1, 2 & 3
            # Lukhin heteronuclear + Wang proton chifts
            self.random_coil_dic = {
                "G": [110.03, 173.96, 45.41, None, 8.37, 3.97],
                "B": [118.7, 175.5, 55.6, 41.2, 8.54, 4.76],
                "A": [125.10, 177.30, 52.42, 19.03, 8.09, 4.31],
                "S": [116.43, 174.41, 58.27, 64.14, 8.22, 4.45],
                "C": [119.19, 174.84, 58.01, 28.20, 8.43, 4.71],
                "M": [120.39, 175.45, 55.34, 33.00, 8.22, 4.47],
                "K": [121.82, 176.39, 56.59, 32.62, 8.21, 4.26],
                "V": [120.93, 175.79, 62.13, 32.65, 7.97, 4.60],
                "T": [114.17, 174.75, 61.62, 69.83, 8.16, 4.44],
                "I": [121.19, 175.69, 60.98, 38.87, 7.94, 4.18],
                "L": [122.22, 177.15, 54.82, 42.82, 8.06, 4.36],
                "D": [120.51, 176.45, 54.12, 40.83, 8.31, 4.62],
                "N": [119.17, 174.65, 53.22, 38.74, 8.35, 4.66],
                "E": [121.44, 176.27, 56.66, 30.13, 8.36, 4.28],
                "Q": [120.34, 175.54, 55.78, 29.34, 8.20, 4.29],
                "R": [122.42, 176.05, 56.25, 30.56, 8.21, 4.26],
                "H": [120.21, 174.54, 55.78, 29.78, 8.18, 4.60],
                "F": [119.84, 174.79, 57.91, 39.34, 8.09, 4.59],
                "Y": [120.41, 175.80, 57.77, 38.88, 7.99, 4.56],
                "W": [120.19, 175.85, 57.50, 29.09, 7.97, 4.60],
                "P": [136.73, 176.60, 63.27, 32.09, None, 4.41],
            }

        elif self.Random_coil_flag == 4:  # Random coil values:1 - Wishart's, 2 - Wang's, 3 - Lukhin's, 4 - Schwarzinger's, 5 - average of 1, 2 & 3

            self.random_coil_dic = {
                "A": [125.0, 178.5, 52.8, 19.3, 8.35, 4.35],
                "B": [118.7, 175.5, 55.6, 41.2, 8.54, 4.76],
                "C": [118.8, 175.3, 58.6, 28.3, 8.44, 4.59],
                "D": [119.1, 175.9, 53.0, 38.3, 8.56, 4.82],
                "E": [120.2, 176.8, 56.1, 29.9, 8.40, 4.42],
                "F": [120.7, 176.6, 58.1, 39.8, 8.31, 4.65],
                "G": [107.5, 174.9, 45.4, None, 8.41, 4.02],
                "H": [118.1, 175.1, 55.4, 29.1, 8.56, 4.79],
                "I": [120.4, 177.1, 61.6, 38.9, 8.17, 4.21],
                "K": [121.6, 177.4, 56.7, 33.2, 8.36, 4.36],
                "L": [122.4, 178.2, 55.5, 42.5, 8.28, 4.38],
                "M": [120.3, 177.1, 55.8, 32.9, 8.42, 4.52],
                "N": [119.0, 176.1, 53.3, 39.1, 8.51, 4.79],
                "P": [None, 177.8, 63.7, 32.2, None, 4.45],
                "Z": [None, None, 63.0, 34.8, None, 4.60],
                "Q": [120.5, 176.8, 56.2, 29.5, 8.44, 4.38],
                "R": [121.2, 177.1, 56.5, 30.9, 8.39, 4.38],
                "S": [115.5, 175.4, 58.7, 64.1, 8.43, 4.51],
                "T": [112.0, 175.6, 62.0, 70.0, 8.25, 4.43],
                "V": [119.3, 177.0, 62.6, 31.8, 8.16, 4.16],
                "W": [122.1, 177.1, 57.6, 29.8, 8.22, 4.70],
                "Y": [120.9, 176.7, 58.3, 38.9, 8.26, 4.58],
            }

        elif self.Random_coil_flag == 5:  # Random coil values:1 - Wishart's, 2 - Wang's, 3 - Lukhin's, 4 - Schwarzinger's, 5 - mean of 1 and 4; 6 - average of 1, 2 & 3
            # Average of Wishart and Shwartz RC
            self.random_coil_dic = {
                "A": [125.00, 178.15, 52.65, 19.20, 8.29, 4.33],
                "C": [118.85, 174.95, 58.40, 28.15, 8.43, 4.65],
                "E": [120.95, 176.70, 56.35, 29.90, 8.41, 4.38],
                "D": [120.25, 176.10, 53.60, 39.70, 8.45, 4.73],
                "G": [108.30, 174.90, 45.25, None, 8.37, 3.99],
                "F": [120.80, 176.20, 57.90, 39.70, 8.30, 4.63],
                "I": [121.05, 176.75, 61.35, 38.85, 8.09, 4.19],
                "H": [118.15, 174.60, 55.20, 29.05, 8.49, 4.76],
                "K": [121.60, 177.00, 56.45, 33.15, 8.32, 4.34],
                "M": [120.50, 176.70, 55.60, 32.90, 8.35, 4.50],
                "L": [122.50, 177.90, 55.30, 42.45, 8.22, 4.36],
                "N": [119.00, 175.65, 53.20, 39.00, 8.46, 4.77],
                "Q": [120.55, 176.40, 55.95, 29.45, 8.38, 4.36],
                "P": [None, 177.55, 63.50, 32.15, None, 4.44],
                "S": [116.05, 175.00, 58.50, 63.95, 8.37, 4.49],
                "R": [121.25, 176.70, 56.25, 30.90, 8.31, 4.36],
                "T": [114.00, 175.15, 61.90, 69.90, 8.20, 4.39],
                "W": [122.15, 176.60, 57.55, 29.70, 8.23, 4.68],
                "V": [119.90, 176.65, 62.40, 32.35, 8.09, 4.14],
                "Y": [120.85, 176.30, 58.10, 38.85, 8.19, 4.56],
                "B": [118.65, 174.95, 55.5, 41.15, 8.515, 4.725],
            }

        elif self.Random_coil_flag == 6:  # Random coil values:1 - Wishart's, 2 - Wang's, 3 - Lukhin's, 4 - Schwarzinger's, 6 - average of 1, 2 & 3

            self.random_coil_dic = {
                "A": [124.64, 177.46, 52.46, 19.04, 8.14, 4.31],
                "C": [118.73, 174.85, 58.15, 28.58, 8.32, 4.67],
                "E": [121.25, 176.40, 56.64, 29.97, 8.38, 4.30],
                "D": [120.76, 176.25, 54.11, 40.90, 8.32, 4.63],
                "G": [109.54, 174.29, 45.26, None, 8.36, 3.97],
                "F": [120.15, 175.35, 57.69, 39.45, 8.16, 4.60],
                "I": [121.16, 175.87, 60.96, 38.70, 7.96, 4.18],
                "H": [119.11, 174.47, 55.51, 29.43, 8.26, 4.64],
                "K": [121.51, 176.38, 56.36, 32.75, 8.22, 4.29],
                "M": [120.41, 175.90, 55.39, 32.94, 8.24, 4.47],
                "L": [122.13, 177.15, 54.90, 42.45, 8.09, 4.35],
                "N": [118.89, 174.90, 53.11, 38.69, 8.37, 4.69],
                "Q": [120.25, 175.76, 55.79, 29.25, 8.24, 4.31],
                "P": [None, 176.84, 63.27, 32.00, None, 4.41],
                "S": [116.34, 174.47, 58.26, 63.90, 8.25, 4.46],
                "R": [121.49, 176.12, 56.14, 30.61, 8.22, 4.29],
                "T": [114.68, 174.74, 61.57, 69.52, 8.16, 4.41],
                "W": [121.13, 175.94, 57.51, 29.43, 8.06, 4.62],
                "V": [120.45, 175.92, 62.11, 32.63, 7.99, 4.28],
                "Y": [120.19, 175.66, 57.77, 38.82, 8.03, 4.56],
            }

        elif self.Random_coil_flag == 7:
            # Average of Wang and Shwartz RC
            self.random_coil_dic = {
                "A": [124.41, 177.89, 52.63, 19.14, 8.22, 4.33],
                "C": [118.45, 175.21, 58.42, 28.92, 8.27, 4.59],
                "E": [120.41, 176.56, 56.38, 29.88, 8.38, 4.35],
                "D": [119.73, 175.95, 53.50, 39.54, 8.44, 4.72],
                "G": [108.49, 174.45, 45.34, None, 8.39, 4.00],
                "F": [120.21, 176.03, 57.78, 39.60, 8.20, 4.62],
                "I": [120.49, 176.31, 61.20, 38.66, 8.05, 4.20],
                "H": [118.51, 174.94, 55.57, 29.30, 8.37, 4.70],
                "K": [121.35, 176.78, 56.50, 32.87, 8.27, 4.32],
                "M": [120.22, 176.52, 55.61, 32.91, 8.32, 4.49],
                "L": [121.98, 177.45, 55.14, 42.32, 8.17, 4.37],
                "N": [118.75, 175.47, 53.15, 38.77, 8.43, 4.72],
                "Q": [120.16, 176.28, 56.05, 29.26, 8.32, 4.33],
                "P": [None, 177.21, 63.47, 32.01, None, 4.43],
                "S": [115.75, 174.91, 58.45, 63.92, 8.32, 4.48],
                "R": [120.97, 176.56, 56.34, 30.63, 8.30, 4.32],
                "T": [112.94, 175.19, 61.65, 69.46, 8.21, 4.44],
                "W": [121.54, 176.49, 57.57, 29.70, 8.10, 4.65],
                "V": [119.60, 176.33, 62.30, 32.08, 8.07, 4.14],
                "Y": [120.14, 176.00, 57.97, 38.84, 8.12, 4.57],
            }

        elif self.Random_coil_flag == 8:
            # Average of Lukin and Shwartz RC
            self.random_coil_dic = {
                "A": [125.05, 177.90, 52.61, 19.16, 8.22, 4.33],
                "C": [119.00, 175.07, 58.30, 28.25, 8.43, 4.65],
                "E": [120.82, 176.54, 56.38, 30.02, 8.38, 4.35],
                "D": [119.81, 176.18, 53.56, 39.56, 8.44, 4.72],
                "G": [108.77, 174.43, 45.41, None, 8.39, 4.00],
                "F": [120.27, 175.69, 58.00, 39.57, 8.20, 4.62],
                "I": [120.80, 176.39, 61.29, 38.88, 8.05, 4.20],
                "H": [119.16, 174.82, 55.59, 29.44, 8.37, 4.70],
                "K": [121.71, 176.89, 56.65, 32.91, 8.29, 4.31],
                "M": [120.34, 176.27, 55.57, 32.95, 8.32, 4.49],
                "L": [122.31, 177.68, 55.16, 42.66, 8.17, 4.37],
                "N": [119.09, 175.38, 53.26, 38.92, 8.43, 4.72],
                "Q": [120.42, 176.17, 55.99, 29.42, 8.32, 4.33],
                "P": [None, 177.20, 63.48, 32.15, None, 4.43],
                "S": [115.97, 174.91, 58.48, 64.12, 8.32, 4.48],
                "R": [121.81, 176.57, 56.38, 30.73, 8.30, 4.32],
                "T": [113.09, 175.18, 61.81, 69.91, 8.21, 4.44],
                "W": [121.14, 176.47, 57.55, 29.45, 8.10, 4.65],
                "V": [120.12, 176.39, 62.37, 32.23, 8.06, 4.38],
                "Y": [120.66, 176.25, 58.03, 38.89, 8.12, 4.57],
            }

        elif self.Random_coil_flag == 9:
            # Average of Mean RC  and Shwartz RC
            self.random_coil_dic = {
                "A": [124.82, 177.98, 52.63, 19.17, 8.25, 4.33],
                "C": [118.77, 175.07, 58.38, 28.44, 8.38, 4.63],
                "E": [120.72, 176.60, 56.37, 29.93, 8.39, 4.36],
                "D": [119.93, 176.07, 53.55, 39.60, 8.44, 4.72],
                "G": [108.52, 174.59, 45.33, None, 8.38, 4.00],
                "F": [120.43, 175.97, 57.89, 39.62, 8.23, 4.62],
                "I": [120.78, 176.49, 61.28, 38.80, 8.06, 4.20],
                "H": [118.60, 174.78, 55.45, 29.27, 8.41, 4.71],
                "K": [121.56, 176.89, 56.53, 32.98, 8.29, 4.33],
                "M": [120.35, 176.50, 55.59, 32.92, 8.33, 4.49],
                "L": [122.27, 177.68, 55.20, 42.48, 8.18, 4.37],
                "N": [118.94, 175.50, 53.20, 38.89, 8.44, 4.74],
                "Q": [120.38, 176.28, 56.00, 29.38, 8.34, 4.34],
                "P": [None, 177.32, 63.48, 32.10, None, 4.43],
                "S": [115.92, 174.94, 58.48, 64.00, 8.34, 4.48],
                "R": [121.34, 176.61, 56.32, 30.75, 8.30, 4.33],
                "T": [113.34, 175.17, 61.78, 69.76, 8.21, 4.42],
                "W": [121.61, 176.52, 57.55, 29.62, 8.14, 4.66],
                "V": [119.88, 176.46, 62.36, 32.22, 8.07, 4.22],
                "Y": [120.55, 176.18, 58.03, 38.86, 8.14, 4.57],
            }

        self.__N_place = 0
        self.__CO_place = 1
        self.__CA_place = 2
        self.__CB_place = 3
        self.__NH_place = 4
        self.__HA_place = 5

        __coil_place = 1
        __beta_place = 0
        __helix_place = 2

        # self.__simpred_res_num = 0
        # self.__simpred_res_name = 1
        # self.__simpred_res_CA = 2
        # self.__simpred_res_CB = 3
        # self.__simpred_res_CO = 4
        # self.__simpred_res_N = 5
        # self.__simpred_res_NH = 6
        # self.__simpred_res_HA = 7

        if self.smooth == 0:
            self.no_smoothing_flag = 1
            self.three_point_smooth_flag = 0
            self.five_point_smooth_flag = 0
        elif self.smooth == 3:
            self.three_point_smooth_flag = 1
            self.no_smoothing_flag = 0
            self.five_point_smooth_flag = 0
        elif self.smooth == 5:
            self.five_point_smooth_flag = 1
            self.three_point_smooth_flag = 0
            self.no_smoothing_flag = 0

        if self.final_smooth == 0:
            self.final_no_smoothing_flag = 1
            self.final_three_point_smooth_flag = 0
            self.final_five_point_smooth_flag = 0
        elif self.final_smooth == 3:
            self.final_three_point_smooth_flag = 1
            self.final_no_smoothing_flag = 0
            self.final_five_point_smooth_flag = 0
        elif self.final_smooth == 5:
            self.final_five_point_smooth_flag = 1
            self.final_three_point_smooth_flag = 0
            self.final_no_smoothing_flag = 0

        self.__sec_str_place = {"H": __helix_place, "C": __coil_place, "B": __beta_place}

        self.__coeff_dict = {
            "100000": {
                "CA": 0.6,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001000": {
                "CA": 0.0,
                "CO": 0.3,
                "C": 0.3,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "000010": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.6,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010000": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.6,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "000100": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.6,
                "H": 0.6,
                "HN": 0.6,
            },
            "000001": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.6,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "000110": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.71,
                "HA": 0.0,
                "NH": 0.51,
                "H": 0.51,
                "HN": 0.51,
            },
            "000011": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.27,
                "HA": 0.77,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001010": {
                "CA": 0.0,
                "CO": 0.77,
                "C": 0.77,
                "CB": 0.0,
                "N": 0.27,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "000101": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.67,
                "NH": 0.03,
                "H": 0.03,
                "HN": 0.03,
            },
            "001001": {
                "CA": 0.0,
                "CO": 0.55,
                "C": 0.55,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.69,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001100": {
                "CA": 0.0,
                "CO": 0.85,
                "C": 0.85,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.25,
                "H": 0.25,
                "HN": 0.25,
            },
            "010010": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.71,
                "N": 0.65,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010001": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.06,
                "N": 0.0,
                "HA": 0.69,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010100": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.8,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.4,
                "H": 0.4,
                "HN": 0.4,
            },
            "011000": {
                "CA": 0.0,
                "CO": 0.77,
                "C": 0.77,
                "CB": 0.27,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "100010": {
                "CA": 0.77,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.27,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "100001": {
                "CA": 0.6,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.69,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "100100": {
                "CA": 0.85,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.25,
                "H": 0.25,
                "HN": 0.25,
            },
            "101000": {
                "CA": 0.71,
                "CO": 0.65,
                "C": 0.65,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "110000": {
                "CA": 0.75,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.38,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001011": {
                "CA": 0.0,
                "CO": 0.72,
                "C": 0.72,
                "CB": 0.0,
                "N": 0.34,
                "HA": 0.83,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001110": {
                "CA": 0.0,
                "CO": 0.93,
                "C": 0.93,
                "CB": 0.0,
                "N": 0.27,
                "HA": 0.0,
                "NH": 0.2,
                "H": 0.2,
                "HN": 0.2,
            },
            "000111": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.27,
                "HA": 0.77,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010011": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.07,
                "N": 0.27,
                "HA": 0.82,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010110": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.81,
                "N": 0.7,
                "HA": 0.0,
                "NH": 0.26,
                "H": 0.26,
                "HN": 0.26,
            },
            "001101": {
                "CA": 0.0,
                "CO": 0.76,
                "C": 0.76,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.82,
                "NH": 0.18,
                "H": 0.18,
                "HN": 0.18,
            },
            "011010": {
                "CA": 0.0,
                "CO": 0.82,
                "C": 0.82,
                "CB": 0.36,
                "N": 0.31,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "010101": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.07,
                "N": 0.0,
                "HA": 0.76,
                "NH": 0.04,
                "H": 0.04,
                "HN": 0.04,
            },
            "011001": {
                "CA": 0.0,
                "CO": 0.61,
                "C": 0.61,
                "CB": 0.06,
                "N": 0.0,
                "HA": 0.76,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "011100": {
                "CA": 0.0,
                "CO": 0.87,
                "C": 0.87,
                "CB": 0.3,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.23,
                "H": 0.23,
                "HN": 0.23,
            },
            "100011": {
                "CA": 0.75,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.37,
                "HA": 0.82,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "100110": {
                "CA": 0.83,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.2,
                "HA": 0.0,
                "NH": 0.23,
                "H": 0.23,
                "HN": 0.23,
            },
            "101010": {
                "CA": 0.79,
                "CO": 0.74,
                "C": 0.74,
                "CB": 0.0,
                "N": 0.32,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "100101": {
                "CA": 0.77,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.82,
                "NH": 0.2,
                "H": 0.2,
                "HN": 0.2,
            },
            "101001": {
                "CA": 0.64,
                "CO": 0.59,
                "C": 0.59,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.78,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "101100": {
                "CA": 0.76,
                "CO": 0.76,
                "C": 0.76,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.26,
                "H": 0.26,
                "HN": 0.26,
            },
            "110010": {
                "CA": 0.8,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.43,
                "N": 0.32,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "110001": {
                "CA": 0.7,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.17,
                "N": 0.0,
                "HA": 0.74,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "110100": {
                "CA": 0.84,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.36,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.2,
                "H": 0.2,
                "HN": 0.2,
            },
            "111000": {
                "CA": 0.81,
                "CO": 0.73,
                "C": 0.73,
                "CB": 0.31,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "111001": {
                "CA": 0.7,
                "CO": 0.63,
                "C": 0.63,
                "CB": 0.14,
                "N": 0.0,
                "HA": 0.82,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "111100": {
                "CA": 0.83,
                "CO": 0.8,
                "C": 0.8,
                "CB": 0.34,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.28,
                "H": 0.28,
                "HN": 0.28,
            },
            "111010": {
                "CA": 0.83,
                "CO": 0.75,
                "C": 0.75,
                "CB": 0.36,
                "N": 0.34,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "110101": {
                "CA": 0.81,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.17,
                "N": 0.0,
                "HA": 0.83,
                "NH": 0.18,
                "H": 0.18,
                "HN": 0.18,
            },
            "110011": {
                "CA": 0.78,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.18,
                "N": 0.36,
                "HA": 0.82,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "110110": {
                "CA": 0.92,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.4,
                "N": 0.22,
                "HA": 0.0,
                "NH": 0.2,
                "H": 0.2,
                "HN": 0.2,
            },
            "101101": {
                "CA": 0.73,
                "CO": 0.72,
                "C": 0.72,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.85,
                "NH": 0.25,
                "H": 0.25,
                "HN": 0.25,
            },
            "101011": {
                "CA": 0.68,
                "CO": 0.67,
                "C": 0.67,
                "CB": 0.0,
                "N": 0.42,
                "HA": 0.85,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "101110": {
                "CA": 0.85,
                "CO": 0.81,
                "C": 0.81,
                "CB": 0.0,
                "N": 0.23,
                "HA": 0.0,
                "NH": 0.25,
                "H": 0.25,
                "HN": 0.25,
            },
            "100111": {
                "CA": 0.77,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.34,
                "HA": 0.84,
                "NH": 0.1,
                "H": 0.1,
                "HN": 0.1,
            },
            "011101": {
                "CA": 0.0,
                "CO": 0.78,
                "C": 0.78,
                "CB": 0.05,
                "N": 0.0,
                "HA": 0.85,
                "NH": 0.18,
                "H": 0.18,
                "HN": 0.18,
            },
            "011011": {
                "CA": 0.0,
                "CO": 0.74,
                "C": 0.74,
                "CB": 0.05,
                "N": 0.37,
                "HA": 0.86,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "011110": {
                "CA": 0.0,
                "CO": 0.89,
                "C": 0.89,
                "CB": 0.29,
                "N": 0.26,
                "HA": 0.0,
                "NH": 0.2,
                "H": 0.2,
                "HN": 0.2,
            },
            "010111": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.07,
                "N": 0.27,
                "HA": 0.82,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "001111": {
                "CA": 0.0,
                "CO": 0.77,
                "C": 0.77,
                "CB": 0.0,
                "N": 0.34,
                "HA": 0.86,
                "NH": 0.07,
                "H": 0.07,
                "HN": 0.07,
            },
            "111101": {
                "CA": 0.79,
                "CO": 0.73,
                "C": 0.73,
                "CB": 0.13,
                "N": 0.0,
                "HA": 0.86,
                "NH": 0.25,
                "H": 0.25,
                "HN": 0.25,
            },
            "110111": {
                "CA": 0.83,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.19,
                "N": 0.37,
                "HA": 0.84,
                "NH": 0.1,
                "H": 0.1,
                "HN": 0.1,
            },
            "101111": {
                "CA": 0.76,
                "CO": 0.76,
                "C": 0.76,
                "CB": 0.0,
                "N": 0.41,
                "HA": 0.89,
                "NH": 0.18,
                "H": 0.18,
                "HN": 0.18,
            },
            "011111": {
                "CA": 0.0,
                "CO": 0.8,
                "C": 0.8,
                "CB": 0.06,
                "N": 0.36,
                "HA": 0.89,
                "NH": 0.08,
                "H": 0.08,
                "HN": 0.08,
            },
            "111011": {
                "CA": 0.74,
                "CO": 0.7,
                "C": 0.7,
                "CB": 0.15,
                "N": 0.47,
                "HA": 0.87,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            },
            "111110": {
                "CA": 0.87,
                "CO": 0.8,
                "C": 0.8,
                "CB": 0.39,
                "N": 0.24,
                "HA": 0.0,
                "NH": 0.26,
                "H": 0.26,
                "HN": 0.26,
            },
            "111111": {
                "CA": 0.81,
                "CO": 0.78,
                "C": 0.78,
                "CB": 0.15,
                "N": 0.43,
                "HA": 0.9,
                "NH": 0.18,
                "H": 0.18,
                "HN": 0.18,
            },
            "000000": {
                "CA": 0.0,
                "CO": 0.0,
                "C": 0.0,
                "CB": 0.0,
                "N": 0.0,
                "HA": 0.0,
                "NH": 0.0,
                "H": 0.0,
                "HN": 0.0,
            }
        }

        self.__coef_mean = self.__mean_coef()

    def __mean_coef(self):
        """ Function to calculate mean weighting coefficient.
        """

        all_atoms_trigger = "111111"

        l_atom_list = ["CA", "HA", "CO", "CB", "N", "H"]
        l_coef_list = [self.__coeff_dict[all_atoms_trigger][atom] for atom in l_atom_list]

        return lmean(l_coef_list)

    def __read_csi_pseudo(self):
        """ Create pseudo-CSI list to run the program without CSI/Dynamr.
        """

        sec_str_list = []

        for l_bmrb_to_aa_entry in self.__bmrb_to_aa_list:
            if l_bmrb_to_aa_entry[0] in self.__aa_names_full_all_CAP:
                residue_num = l_bmrb_to_aa_entry[1]
                residue_name = self.__aa_names_full_all_CAP[l_bmrb_to_aa_entry[0][0:3]]
                sec_str_type = "C"
                sec_str_list.append([residue_num, residue_name, sec_str_type])

        return sec_str_list

    def __write_atom_list(self):
        """ Function to create an atom list.
        """

        l_atom_list = []
        if self.CAp_list[0] != 0.0 and self.CAp_exclude != 1:
            l_atom_list.append("CA")
        if self.CBp_list[0] != 0.0 and self.CBp_exclude != 1:
            l_atom_list.append("CB")
        if self.COp_list[0] != 0.0 and self.COp_exclude != 1:
            l_atom_list.append("C")
        if self.Np_list[0] != 0.0 and self.Np_exclude != 1:
            l_atom_list.append("N")
        if self.Hp_list[0] != 0.0 and self.Hp_exclude != 1:
            l_atom_list.append("H")
        if self.HAp_list[0] != 0.0 and self.HAp_exclude != 1:
            l_atom_list.append("HA")

        return l_atom_list

    def __gap_fill(self, L_list, l_atom):
        """ Gap filling function when smooth=0, either  i+n or i-n
        """

        L_all = []
        if len(L_list) > 0:
            done = 0
            while not done:
                for L_aa in self.__bmrb_to_aa_list:
                    L_residue_number_found = 0
                    L_residue_number = L_aa[1]
                    L_residue_name = self.__aa_names_full_all_CAP[L_aa[0][0:3]]
                    while not L_residue_number_found:
                        for L_item in L_list:
                            if L_residue_number == L_item[0] and L_residue_name == L_item[1] and l_atom == L_item[6]:
                                L_residue_number_found = 1
                                L_bmrb_shift = L_item[2]
                                L_simpred_shift = L_item[3]
                                L_chem_shift_diff = L_item[4]
                                L_chem_shift_diff_abs = L_item[5]
                                L_atom_type = L_item[6]
                                L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift, L_chem_shift_diff, L_chem_shift_diff_abs, L_atom_type])

                        if L_residue_number_found == 0 and L_residue_number not in self.__excluded_residues:

                            for i in range(1, self.gap_limit + 1):
                                if L_residue_number_found == 0:
                                    positive_found = 0
                                    # negative_found = 0
                                    for L_item in L_list:
                                        if L_residue_number + i == L_item[0] and l_atom == L_item[6]:
                                            L_residue_number_found = 1
                                            positive_found = 1
                                            L_bmrb_shift = L_item[2]
                                            L_simpred_shift = L_item[3]
                                            L_chem_shift_diff = L_item[4]
                                            L_chem_shift_diff_abs = L_item[5]
                                            L_atom_type = L_item[6]
                                            L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift, L_chem_shift_diff, L_chem_shift_diff_abs, L_atom_type])

                                    if positive_found == 0:
                                        for L_item in L_list:
                                            if L_residue_number - i == L_item[0] and l_atom == L_item[6]:
                                                L_residue_number_found = 1
                                                # negative_found = 1
                                                L_bmrb_shift = L_item[2]
                                                L_simpred_shift = L_item[3]
                                                L_chem_shift_diff = L_item[4]
                                                L_chem_shift_diff_abs = L_item[5]
                                                L_atom_type = L_item[6]
                                                L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift, L_chem_shift_diff, L_chem_shift_diff_abs, L_atom_type])  # noqa: E501

                                    else:
                                        L_residue_number_found = 1
                            else:  # pylint: disable=useless-else-on-loop
                                L_residue_number_found = 1
                else:  # pylint: disable=useless-else-on-loop
                    done = 1

        return L_all

    def __gap_fill2(self, L_list, l_atom):
        """ Gap filling function when smooth=0, uses averaging of i+n and i-n
        """

        L_all = []
        if len(L_list) > 0:
            done = 0
            while not done:
                for L_aa in self.__bmrb_to_aa_list:
                    L_residue_number_found = 0
                    L_residue_number = L_aa[1]
                    L_residue_name = self.__aa_names_full_all_CAP[L_aa[0][0:3]]
                    while not L_residue_number_found:
                        for L_item in L_list:
                            if L_residue_number == L_item[0] and L_residue_name == L_item[1] and l_atom == L_item[6]:
                                L_residue_number_found = 1
                                L_bmrb_shift = L_item[2]
                                L_simpred_shift = L_item[3]
                                L_chem_shift_diff = L_item[4]
                                L_chem_shift_diff_abs = L_item[5]
                                L_atom_type = L_item[6]
                                L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift, L_chem_shift_diff, L_chem_shift_diff_abs, L_atom_type])

                        if L_residue_number_found == 0 and self.__real_BMRB_first_res <= L_residue_number <= self.__real_BMRB_last_res and L_residue_number not in self.__excluded_residues:  # noqa: E501
                            positive_found = 0
                            negative_found = 0
                            pos_neg_list_true = []
                            pos_neg_list_abs = []
                            pos_neg_list_true_ave = None
                            pos_neg_list_abs_ave = None
                            for i in range(1, self.gap_limit + 1):
                                if L_residue_number_found == 0:
                                    if positive_found != 1:
                                        for L_item in L_list:
                                            if L_residue_number + i == L_item[0] and l_atom == L_item[6]:
                                                positive_found = 1
                                                L_chem_shift_diff = L_item[4]
                                                L_chem_shift_diff_abs = L_item[5]
                                                pos_neg_list_abs.append(L_chem_shift_diff_abs)
                                                pos_neg_list_true.append(L_chem_shift_diff)
                                    if negative_found != 1:
                                        for L_item in L_list:
                                            if L_residue_number - i == L_item[0] and l_atom == L_item[6]:
                                                negative_found = 1
                                                L_chem_shift_diff = L_item[4]
                                                L_chem_shift_diff_abs = L_item[5]
                                                pos_neg_list_abs.append(L_chem_shift_diff_abs)
                                                pos_neg_list_true.append(L_chem_shift_diff)

                            if negative_found == 1 or positive_found == 1:
                                if len(pos_neg_list_abs) > 0:
                                    pos_neg_list_abs_ave = lmean(pos_neg_list_abs)
                                if len(pos_neg_list_true) > 0:
                                    pos_neg_list_true_ave = lmean(pos_neg_list_true)
                                L_residue_number_found = 1
                                L_bmrb_shift = L_item[2]
                                L_simpred_shift = L_item[3]
                                L_chem_shift_diff = pos_neg_list_true_ave
                                L_chem_shift_diff_abs = pos_neg_list_abs_ave
                                L_atom_type = L_item[6]
                                L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift, L_chem_shift_diff, L_chem_shift_diff_abs, L_atom_type])

                            else:
                                L_residue_number_found = 1
                        else:
                            L_residue_number_found = 1
                else:  # pylint: disable=useless-else-on-loop
                    done = 1
        else:
            done = 1

        return L_all

    def __smoothing(self, L_smooth, L_list, l_atom_type):
        """ Smoothing function.
        """
        # Building residue number list
        first_residue = 0
        last_residue = 0
        first_residue_flag = 0
        smooth_factor = L_smooth[2]
        central_residue = (smooth_factor // 2) + 1
        N_terminus_smooth_factor = central_residue
        # C_terminus_smooth_factor = smooth_factor // 2
        L_all = []
        ############################################
        # To find first and last residues
        ############################################
        for L_item in L_list:
            if first_residue_flag == 0:
                first_residue = L_item[0]
                first_residue_flag = 1
            else:
                last_residue = L_item[0]

        #################################################
        # Find residue number offset
        #################################################
        offset = first_residue - 1

        ###################################################
        # To build a list with residue numbers
        ###################################################
        residue_number_list = list(range(first_residue, last_residue + 1))

        # counter = 0
        # list_renew_flag = 0
        L_start = 0
        L_end = L_smooth[2]
        # L_length = len(L_list)
        if len(L_list) > 1:
            ####################################################
            # Smoothing N_terminus
            ####################################################
            Real_first_residue = L_list[0][0]
            Real_last_residue = L_list[-1][0]
            L_chem_shift_diff_list = []
            L_chem_shift_diff_abs_list = []
            Old_first_residue = Real_first_residue
            negative_switch = 0
            for L_number in range(0, (smooth_factor - N_terminus_smooth_factor)):
                L_chem_shift_diff_list = []
                L_chem_shift_diff_abs_list = []
                L_residue_number = L_list[L_number][0]
                L_residue_name = L_list[L_number][1]
                L_bmrb_shift = L_list[L_number][2]
                L_simpred_shift = L_list[L_number][3]
                L_chem_shift_diff = L_list[L_number][4]
                L_chem_shift_diff_abs = L_list[L_number][5]
                L_chem_shift_diff_list.append(L_chem_shift_diff)
                L_chem_shift_diff_abs_list.append(L_chem_shift_diff_abs)
                Old_first_residue = L_residue_number
                D_residue_number = L_residue_number
                plus_done = 0
                D_counter = 1
                while not plus_done:
                    D_residue_number += 1
                    if abs(D_residue_number - Old_first_residue) <= central_residue:
                        for L_entry in L_list:
                            N_residue_number = L_entry[0]
                            if N_residue_number == D_residue_number:
                                N_chem_shift_diff = L_entry[4]
                                N_chem_shift_diff_abs = L_entry[5]
                                L_chem_shift_diff_list.append(N_chem_shift_diff)
                                L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                Old_first_residue = N_residue_number
                                D_counter += 1
                                if D_counter == central_residue:
                                    plus_done = 1
                    else:
                        plus_done = 1
                if negative_switch == 1:
                    Old_first_residue = L_residue_number
                    D_residue_number = L_residue_number
                    minus_done = 0
                    D_counter = 1
                    while not minus_done:
                        D_residue_number -= 1
                        if abs(D_residue_number - Old_first_residue) <= central_residue and D_residue_number > 0:
                            for L_entry in L_list:
                                N_residue_number = L_entry[0]
                                if N_residue_number == D_residue_number:
                                    N_chem_shift_diff = L_entry[4]
                                    N_chem_shift_diff_abs = L_entry[5]
                                    L_chem_shift_diff_list.append(N_chem_shift_diff)
                                    L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                    Old_first_residue = N_residue_number
                                    D_counter += 1
                                    if D_counter == central_residue - 1:
                                        plus_done = 1
                        else:
                            minus_done = 1
                N_chem_shift_diff_mean = None
                if len(L_chem_shift_diff_list) > 0:
                    N_chem_shift_diff_mean = lmean(L_chem_shift_diff_list)
                N_chem_shift_diff_abs_mean = None
                if len(L_chem_shift_diff_abs_list) > 0:
                    N_chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_list)
                N_chem_shift_diff_len = len(L_chem_shift_diff_list)

                L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift,
                              N_chem_shift_diff_mean, N_chem_shift_diff_abs_mean, l_atom_type, N_chem_shift_diff_len, smooth_factor])
                negative_switch = 1

            #####################################################
            # Smoothing main
            #####################################################
            done = 0
            while not done:
                if L_end <= last_residue - offset:  # For Ubiquitin
                    L_smooth_list = []
                    missing_chemshift = []
                    bigger_missing = 0
                    smaller_missing = 0
                    new_end = 0
                    new_start = 0

                    L_residue_number = None
                    L_residue_name = None
                    L_bmrb_shift = None
                    L_simpred_shift = None

                    for L_number in residue_number_list[L_start:L_end]:
                        found_flag = 0
                        for L_item in L_list:
                            if L_number == L_item[0]:
                                L_smooth_list.append(L_item)
                                found_flag = 1
                        if found_flag == 0:
                            missing_chemshift.append(L_number)
                    more_smaller = 0
                    more_bigger = 0

                    if len(L_smooth_list) == smooth_factor and smooth_factor != 0:
                        L_chem_shift_diff_tmp_list = []
                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = L_smooth_list[central_residue - 1][0]
                        L_residue_name = L_smooth_list[central_residue - 1][1]
                        L_bmrb_shift = L_smooth_list[central_residue - 1][2]
                        L_simpred_shift = L_smooth_list[central_residue - 1][3]
                        for entry in L_smooth_list:
                            L_chem_shift_diff = entry[4]
                            L_chem_shift_diff_abs = entry[5]
                            L_chem_shift_diff_tmp_list.append(L_chem_shift_diff)
                            L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)
                        chem_shift_diff_mean = lmean(L_chem_shift_diff_tmp_list)
                        chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                        chem_shift_len = len(L_chem_shift_diff_tmp_list)
                        L_start += 1
                        L_end += 1
                        L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift,
                                      chem_shift_diff_mean, chem_shift_diff_abs_mean, l_atom_type, chem_shift_len, smooth_factor])
                    elif len(L_smooth_list) < smooth_factor:

                        L_chem_shift_diff_tmp_list = []
                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = residue_number_list[L_start:L_end][central_residue - 1]
                        for L_item in L_list:
                            if L_residue_number == L_item[0]:
                                L_residue_name = L_item[1]
                                L_bmrb_shift = L_item[2]
                                L_simpred_shift = L_item[3]
                        for miss_number in missing_chemshift:
                            if miss_number > L_residue_number:
                                bigger_missing += 1
                            elif miss_number < L_residue_number:
                                smaller_missing += 1
                            elif miss_number == L_residue_number:
                                bigger_missing += 1
                        if bigger_missing > 0:
                            bigger_counter = 0
                            bigger_flag = 0
                            new_end = residue_number_list[L_end - 1]
                            while not bigger_flag:
                                new_end += 1
                                if new_end < last_residue:
                                    if new_end < residue_number_list[L_end] + self.gap_limit_smooth:
                                        for L_item in L_list:
                                            if new_end == L_item[0]:
                                                L_smooth_list.append(L_item)
                                                bigger_counter += 1
                                                if bigger_counter == bigger_missing:
                                                    bigger_flag = 1

                                    else:
                                        bigger_flag = 1
                                else:
                                    more_smaller = bigger_missing - bigger_counter
                                    bigger_flag = 1
                        if smaller_missing > 0:
                            smaller_counter = 0
                            smaller_flag = 0
                            new_start = residue_number_list[L_start]
                            while not smaller_flag:
                                new_start -= 1
                                if new_start >= first_residue:
                                    if new_start >= residue_number_list[L_start] - self.gap_limit_smooth:
                                        for L_item in L_list:
                                            if new_start == L_item[0]:
                                                L_smooth_list.append(L_item)
                                                smaller_counter += 1
                                                if smaller_counter == smaller_missing:
                                                    smaller_flag = 1
                                    else:
                                        smaller_flag = 1
                                else:
                                    more_bigger = smaller_missing - smaller_counter
                                    smaller_flag = 1

                        if more_bigger > 0:
                            more_bigger_counter = 0
                            more_bigger_flag = 0
                            new_end = residue_number_list[L_end - 1]
                            while not more_bigger_flag:
                                new_end += 1
                                if new_end <= last_residue and new_end < residue_number_list[L_end - 1] + self.gap_limit_smooth:
                                    for L_item in L_list:
                                        if new_end == L_item[0]:
                                            L_smooth_list.append(L_item)
                                            more_bigger_counter += 1
                                            if more_bigger_counter == more_bigger:
                                                more_bigger_flag = 1
                                else:
                                    more_bigger_flag = 1

                        if more_smaller > 0:
                            more_smaller_counter = 0
                            more_smaller_flag = 0
                            new_start = residue_number_list[L_start]
                            while not more_smaller_flag:
                                new_start -= 1

                                if new_start >= first_residue and new_start >= residue_number_list[L_start] - self.gap_limit_smooth:
                                    for L_item in L_list:
                                        if new_start == L_item[0]:
                                            L_smooth_list.append(L_item)
                                            more_smaller_counter += 1
                                            if more_smaller_counter == more_smaller:
                                                more_smaller_flag = 1
                                else:
                                    more_smaller_flag = 1

                        for entry in L_smooth_list:
                            L_chem_shift_diff = entry[4]
                            L_chem_shift_diff_abs = entry[5]
                            L_chem_shift_diff_tmp_list.append(L_chem_shift_diff)
                            L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)
                        chem_shift_diff_mean = None
                        if len(L_chem_shift_diff_tmp_list) > 0:
                            chem_shift_diff_mean = lmean(L_chem_shift_diff_tmp_list)
                        chem_shift_diff_abs_mean = None
                        if len(L_chem_shift_diff_abs_tmp_list) > 0:
                            chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                        chem_shift_diff_len = len(L_chem_shift_diff_tmp_list)

                        L_start += 1
                        L_end += 1
                        if L_residue_number not in (Real_first_residue, Real_last_residue):
                            L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift,
                                          chem_shift_diff_mean, chem_shift_diff_abs_mean, l_atom_type, chem_shift_diff_len, smooth_factor])

                    else:
                        L_chem_shift_diff_tmp_list = []
                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = residue_number_list[L_start:L_end][central_residue - 1]
                        for L_item in L_list:
                            if L_residue_number == L_item[0]:
                                L_residue_name = L_item[1]
                                L_bmrb_shift = L_item[2]
                                L_simpred_shift = L_item[3]
                        for entry in L_smooth_list:
                            L_chem_shift_diff = entry[4]
                            L_chem_shift_diff_abs = entry[5]
                            L_chem_shift_diff_tmp_list.append(L_chem_shift_diff)
                            L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)
                        chem_shift_diff_mean = None
                        if len(L_chem_shift_diff_tmp_list) > 0:
                            chem_shift_diff_mean = lmean(L_chem_shift_diff_tmp_list)
                        chem_shift_diff_abs_mean = None
                        if len(L_chem_shift_diff_abs_tmp_list) > 0:
                            chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                        chem_shift_diff_len = len(L_chem_shift_diff_tmp_list)
                        L_start += 1
                        L_end += 1
                        if L_residue_number not in (Real_first_residue, Real_last_residue):
                            L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift,
                                          chem_shift_diff_mean, chem_shift_diff_abs_mean, l_atom_type, chem_shift_diff_len, smooth_factor])

                else:
                    done = 1

            ####################################################
            # Smoothing C_terminus
            ####################################################
            if smooth_factor != 0:
                negative_switch = 0
                Real_first_residue = L_list[0][0]
                Real_last_residue = L_list[1][0]
                Old_first_residue = Real_first_residue
                negative_switch = 0
                for L_number in range(1, (smooth_factor - N_terminus_smooth_factor + 1)):
                    L_chem_shift_diff_list = []
                    L_chem_shift_diff_abs_list = []

                    L_residue_number = L_list[-L_number][0]
                    L_residue_name = L_list[-L_number][1]
                    L_bmrb_shift = L_list[-L_number][2]
                    L_simpred_shift = L_list[-L_number][3]
                    L_chem_shift_diff = L_list[-L_number][4]
                    L_chem_shift_diff_abs = L_list[-L_number][5]
                    L_chem_shift_diff_list.append(L_chem_shift_diff)
                    L_chem_shift_diff_abs_list.append(L_chem_shift_diff_abs)
                    Old_first_residue = L_residue_number
                    D_residue_number = L_residue_number
                    plus_done = 0
                    D_counter = 1
                    while not plus_done:
                        D_residue_number -= 1
                        if abs(D_residue_number - Old_first_residue) <= central_residue:
                            for L_entry in L_list:
                                N_residue_number = L_entry[0]
                                if N_residue_number == D_residue_number:
                                    N_chem_shift_diff = L_entry[4]
                                    N_chem_shift_diff_abs = L_entry[5]
                                    L_chem_shift_diff_list.append(N_chem_shift_diff)
                                    L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                    Old_first_residue = N_residue_number
                                    D_counter += 1
                                    if D_counter == central_residue:
                                        plus_done = 1
                        else:
                            plus_done = 1
                    if negative_switch == 1:
                        Old_first_residue = L_residue_number
                        D_residue_number = L_residue_number
                        minus_done = 0
                        D_counter = 1
                        while not minus_done:
                            D_residue_number += 1
                            if abs(D_residue_number - Old_first_residue) <= central_residue and D_residue_number < Real_last_residue:
                                for L_entry in L_list:
                                    N_residue_number = L_entry[0]
                                    if N_residue_number == D_residue_number:
                                        N_chem_shift_diff = L_entry[4]
                                        N_chem_shift_diff_abs = L_entry[5]
                                        L_chem_shift_diff_list.append(N_chem_shift_diff)
                                        L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                        Old_first_residue = N_residue_number
                                        D_counter += 1
                                        if D_counter == central_residue - 1:
                                            plus_done = 1
                            else:
                                minus_done = 1
                    N_chem_shift_diff_mean = None
                    if len(L_chem_shift_diff_list) > 0:
                        N_chem_shift_diff_mean = lmean(L_chem_shift_diff_list)
                    N_chem_shift_diff_abs_mean = None
                    if len(L_chem_shift_diff_abs_list) > 0:
                        N_chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_list)
                    N_chem_shift_diff_len = len(L_chem_shift_diff_list)
                    L_all.append([L_residue_number, L_residue_name, L_bmrb_shift, L_simpred_shift,
                                  N_chem_shift_diff_mean, N_chem_shift_diff_abs_mean,
                                  l_atom_type, N_chem_shift_diff_len, smooth_factor])
                    negative_switch = 1

        return L_all

    def __grid_search(self, l_all_atoms_smooth, smooth_type, lCOp_list, lCAp_list, lCBp_list, lHAp_list, lHp_list, lNp_list, s_final_smooth):
        """ Function to do a grid search
        """

        list_of_files = []

        CA_CB_CO_HA_all_residues_abs_corr = []

        if self.grid_search_flag == 1:
            lCOp_list = lCAp_list = lCBp_list = lHAp_list = lHp_list = lNp_list = self.gridsearch_list

        zero_list = [0.000001]
        if self.function_flag in (0, 1, 2, 3, 4, 7, 8):
            zero_list = [0]
        if self.function_flag in (5, 6):
            zero_list = [1]

        if self.COp_exclude == 1:  # If 1, exclude from grid search
            lCOp_list = zero_list
        if self.CAp_exclude == 1:  # If 1, exclude from grid search
            lCAp_list = zero_list
        if self.CBp_exclude == 1:  # If 1, exclude from grid search
            lCBp_list = zero_list
        if self.HAp_exclude == 1:  # If 1, exclude from grid search
            lHAp_list = zero_list
        if self.Hp_exclude == 1:  # If 1, exclude from grid search
            lHp_list = zero_list  # If 1, exclude from grid search
        if self.Np_exclude == 1:  # If 1, exclude from grid search
            lNp_list = zero_list

        all_the_same_switch = 0
        for coefCApos in lCAp_list:
            for coefCAneg in [1]:  # pylint: disable=unused-variable
                for coefCOpos in lCOp_list:
                    for coefCOneg in [1]:  # pylint: disable=unused-variable
                        for coefHApos in lHAp_list:
                            for coefHAneg in [1]:  # pylint: disable=unused-variable
                                for coefCBpos in lCBp_list:
                                    for coefCBneg in [1]:  # pylint: disable=unused-variable
                                        for coefNpos in lNp_list:
                                            for coefNneg in [1]:  # pylint: disable=unused-variable
                                                for coefHpos in lHp_list:
                                                    for coefHneg in [1]:  # pylint: disable=unused-variable

                                                        CA_CB_CO_HA_all_residues_abs = []
                                                        list_of_coefficients = []

                                                        if self.CAp_exclude != 1:
                                                            list_of_coefficients.append(coefCApos)
                                                        if self.COp_exclude != 1:
                                                            list_of_coefficients.append(coefCOpos)
                                                        if self.HAp_exclude != 1:
                                                            list_of_coefficients.append(coefHApos)
                                                        if self.CBp_exclude != 1:
                                                            list_of_coefficients.append(coefCBpos)
                                                        if self.Np_exclude != 1:
                                                            list_of_coefficients.append(coefNpos)
                                                        if self.Hp_exclude != 1:
                                                            list_of_coefficients.append(coefHpos)

                                                        all_zero = 1
                                                        skip_switch = 0
                                                        all_the_same = 1
                                                        for coef in list_of_coefficients:
                                                            if coef > 0.0001:
                                                                all_zero = 0
                                                            if abs(coef - list_of_coefficients[0]) > 0.000001:
                                                                all_the_same = 0
                                                        if all_the_same == 1 and all_the_same_switch == 1 and all_zero != 1:
                                                            skip_switch = 1
                                                        if all_the_same == 1 and all_zero != 1:
                                                            all_the_same_switch = 1

                                                        coeff_list = {
                                                            "CAp": coefCApos,
                                                            "COp": coefCOpos,
                                                            "HAp": coefHApos,
                                                            "CBp": coefCBpos,
                                                            "Np": coefNpos,
                                                            "Hp": coefHpos,
                                                        }

                                                        if all_zero == 0 and skip_switch == 0:
                                                            for residue_number in range(self.__firstresidue, self.__lastresidue + 1):
                                                                residue_data_abs = []
                                                                value_abs = 0
                                                                CA_found = CB_found = CO_found = NH_found = N_found = HA_found = 0
                                                                for entry in l_all_atoms_smooth:

                                                                    resnumber = entry[0]
                                                                    true_diff_mean = entry[4]
                                                                    abs_diff_mean = entry[5]

                                                                    atom_type = entry[6]
                                                                    if resnumber == residue_number and abs_diff_mean is not None:
                                                                        resname = entry[1]

                                                                        if atom_type == "CA":
                                                                            CA_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefCApos, atom_type, self.CAp_exclude, self.CA_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_pos_CA * coefCApos, atom_type,
                                                                                                             self.CAp_exclude, self.CA_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_neg_CA * coefCApos, atom_type,
                                                                                                             self.CAp_exclude, self.CA_Hertz_corr])

                                                                        elif atom_type == "HA":
                                                                            HA_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefHApos, atom_type, self.HAp_exclude, self.H_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_pos_HA * coefHApos,
                                                                                                             atom_type, self.HAp_exclude, self.H_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_neg_HA * coefHApos,
                                                                                                             atom_type, self.HAp_exclude, self.H_Hertz_corr])

                                                                        elif atom_type == "C":
                                                                            CO_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefCOpos, atom_type, self.COp_exclude, self.CA_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_pos_CO * coefCOpos,
                                                                                                             atom_type, self.COp_exclude, self.CA_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_neg_CO * coefCOpos,
                                                                                                             atom_type, self.COp_exclude, self.CA_Hertz_corr])

                                                                        elif atom_type == "CB":
                                                                            CB_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefCBpos, atom_type, self.CBp_exclude, self.CA_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_pos_CB * coefCBpos,
                                                                                                             atom_type, self.CBp_exclude, self.CA_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_neg_CB * coefCBpos,
                                                                                                             atom_type, self.CBp_exclude, self.CA_Hertz_corr])

                                                                        elif atom_type == "N":
                                                                            N_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean), coefNpos, atom_type,
                                                                                                         self.Np_exclude, self.N_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_pos_N * coefNpos,
                                                                                                             atom_type, self.Np_exclude, self.N_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_neg_N * coefNpos,
                                                                                                             atom_type, self.Np_exclude, self.N_Hertz_corr])

                                                                        elif atom_type == "H":
                                                                            NH_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean), coefHpos,
                                                                                                         atom_type, self.Hp_exclude, self.H_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_pos_H * coefHpos,
                                                                                                             atom_type, self.Hp_exclude, self.H_Hertz_corr])
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean, self.coef_for_neg_H * coefHpos,
                                                                                                             atom_type, self.Hp_exclude, self.H_Hertz_corr])

                                                                        elif atom_type == "HN":
                                                                            NH_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_pos_H * coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])  # noqa: E501
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_neg_H * coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])  # noqa: E501

                                                                        elif atom_type == "NH":
                                                                            NH_found = 1
                                                                            if self.pos_neg_value == 0:
                                                                                residue_data_abs.append([resnumber, resname, abs(abs_diff_mean),
                                                                                                         coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])
                                                                            else:
                                                                                if true_diff_mean > 0:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_pos_H * coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])  # noqa: E501
                                                                                else:
                                                                                    residue_data_abs.append([resnumber, resname, true_diff_mean,
                                                                                                             self.coef_for_neg_H * coefHpos, atom_type, self.Hp_exclude, self.H_Hertz_corr])  # noqa: E501

                                                                atoms_abs = []

                                                                resid_name_abs = None

                                                                if self.function_flag == 0:
                                                                    value_abs = 0
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs += coeffabs * pow(exp(-valueabs), 2)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 1:
                                                                    value_abs = 1
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs *= pow(valueabs, -coeffabs)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 2:
                                                                    value_abs = 1
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs *= coeffabs * (valueabs ** 0.2) / pow(exp(valueabs), 0.6)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 3:
                                                                    value_abs = 0
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]

                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs += pow(valueabs, -coeffabs)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 4:
                                                                    value_abs = 0
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs += coeffabs / valueabs
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 5:
                                                                    value_abs = 0
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs += coeffabs / log(1 + valueabs)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 6:
                                                                    value_abs = 1
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs *= coeffabs / log(1 + valueabs)
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 7:
                                                                    value_abs = 1
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if valueabs < self.floor_value:
                                                                            valueabs = self.floor_value
                                                                        if l_exclude != 1 and coeffabs > 0.01:
                                                                            value_abs *= exp(pow(valueabs, -coeffabs))
                                                                        atoms_abs.append(atomabs)

                                                                elif self.function_flag == 8:
                                                                    valueabs_list = []
                                                                    valueabs_origin = None
                                                                    valueabs = None
                                                                    coef_trigger = "%s%s%s%s%s%s" % (CA_found, CB_found, CO_found, NH_found, N_found, HA_found)
                                                                    coef_list = []
                                                                    for item in residue_data_abs:
                                                                        resid_name_abs = item[1]
                                                                        valueabs_origin = item[2]
                                                                        coeffabs = item[3]
                                                                        atomabs = item[4]
                                                                        l_exclude = item[5]
                                                                        if self.miss_ass == 1:
                                                                            coeffabs = self.__coeff_dict[coef_trigger][atomabs]
                                                                        coef_list.append(coeffabs)

                                                                        if valueabs_origin is not None and l_exclude != 1 and abs(coeffabs) > 0.01:
                                                                            valueabs = valueabs_origin * item[6]
                                                                            if l_exclude != 1 and abs(coeffabs) > 0.01 and valueabs is not None:
                                                                                if abs(valueabs) < self.floor_value1:
                                                                                    valueabs = self.floor_value1 if valueabs >= 0 else -self.floor_value1
                                                                            valueabs_list.append(valueabs * coeffabs * 5)

                                                                        atoms_abs.append(atomabs)
                                                                    if len(valueabs_list) > 0:
                                                                        coef_local_mean = lmean(coef_list)
                                                                        coef_offset = self.__coef_mean / coef_local_mean
                                                                        valueabs_mean = lmean(valueabs_list)
                                                                        if self.scale == 1:
                                                                            valueabs_mean *= coef_offset
                                                                        if valueabs_mean != 0:
                                                                            value_abs += 1 / (abs(valueabs_mean) ** 1.5)
                                                                        if value_abs > self.floor_value:
                                                                            value_abs = self.floor_value
                                                                    else:
                                                                        value_abs = None

                                                                l_incomplete_switch = 1
                                                                if len(atoms_abs) < len(self.__atom_list):
                                                                    l_incomplete_switch = self.incomplete_data_use
                                                                if value_abs is not None and l_incomplete_switch == 1:
                                                                    CA_CB_CO_HA_all_residues_abs.append([residue_number, resid_name_abs, value_abs, atoms_abs,
                                                                                                         self.__atom_list, "abs", smooth_type])

                                                            s_final_again = 0
                                                            if self.termini_corr_flag == 1:
                                                                CA_CB_CO_HA_all_residues_abs_new = self.__end_effect(CA_CB_CO_HA_all_residues_abs)
                                                            elif self.termini_corr_flag == 2:
                                                                CA_CB_CO_HA_all_residues_abs_new = self.__end_effect2(CA_CB_CO_HA_all_residues_abs)
                                                            elif self.termini_corr_flag == 3:
                                                                CA_CB_CO_HA_all_residues_abs_new = self.__end_effect3(CA_CB_CO_HA_all_residues_abs)
                                                            elif self.termini_corr_flag == 4:
                                                                CA_CB_CO_HA_all_residues_abs_new = self.__end_effect4(CA_CB_CO_HA_all_residues_abs)
                                                            elif self.termini_corr_flag == 5:
                                                                CA_CB_CO_HA_all_residues_abs_new = self.__end_effect5(CA_CB_CO_HA_all_residues_abs)
                                                            else:
                                                                CA_CB_CO_HA_all_residues_abs_new = CA_CB_CO_HA_all_residues_abs

                                                            if s_final_smooth in [3, 5, 9]:
                                                                if s_final_smooth == 9:
                                                                    s_final_smooth = 3
                                                                    s_final_again = 1
                                                                CA_CB_CO_HA_all_residues_abs_fsmooth = self.__final_smoothing(s_final_smooth, CA_CB_CO_HA_all_residues_abs_new)
                                                                if s_final_smooth != 0 and s_final_smooth != 9 and self.termini_corr_flag == 6:
                                                                    CA_CB_CO_HA_all_residues_abs_new2 = self.__end_effect3(CA_CB_CO_HA_all_residues_abs_fsmooth)
                                                                    CA_CB_CO_HA_all_residues_abs_fsmooth = CA_CB_CO_HA_all_residues_abs_new2
                                                                if s_final_again == 1:
                                                                    s_final_smooth = 3
                                                                    CA_CB_CO_HA_all_residues_abs_fsmooth = self.__final_smoothing(s_final_smooth, CA_CB_CO_HA_all_residues_abs_fsmooth)  # noqa: E501
                                                                CA_CB_CO_HA_all_residues_abs_corr, list_of_files, _d_Pearson_coeff, _d_Spearman_coeff = self.__combo(CA_CB_CO_HA_all_residues_abs_fsmooth,  # noqa: E501
                                                                                                                                                                     CA_CB_CO_HA_all_residues_abs_corr,  # noqa: E501
                                                                                                                                                                     coeff_list)  # noqa: E501
                                                            else:
                                                                CA_CB_CO_HA_all_residues_abs_corr, list_of_files, _d_Pearson_coeff, _d_Spearman_coeff = self.__combo(CA_CB_CO_HA_all_residues_abs_new,  # noqa: E501
                                                                                                                                                                     CA_CB_CO_HA_all_residues_abs_corr,  # noqa: E501
                                                                                                                                                                     coeff_list)  # noqa: E501

        all_lists = [CA_CB_CO_HA_all_residues_abs_corr]

        len_aa_list = len(self.__bmrb_to_aa_list)
        ret = {'seq_id': [aa[1] for aa in self.__bmrb_to_aa_list], 'rci': [None] * len_aa_list, 'nmr_rmsd': [None] * len_aa_list, 's2': [None] * len_aa_list}

        for file in list_of_files:

            correlation_sorted = []

            for item in all_lists:
                for each in item:
                    if each[0] is file:
                        smoothing = each[4]
                        treatmrent = each[3]
                        atoms_for_plot = ''.join(self.__atom_list)
                        for entry in each[1]:
                            comparison_paramter = entry[0]
                            comparison_method = entry[2]
                            parameter = entry[3]
                            coefficients = ""
                            coefficients_for_plot = ""
                            coef_dictionary = entry[4]
                            md_plot_data = entry[5]
                            dynamr_plot_data = entry[6]
                            dynamr_2_md = entry[7]
                            for coef_name in entry[4]:
                                coef_value = "%s" % coef_dictionary[coef_name]
                                coefficients += coef_name + " " + coef_value + " "
                                coefficients_for_plot += coef_name + coef_value
                            out_title = "%.2fvs %s %s%s CA%.2fCB%.2fCO%.2fN%.2fH%.2fHA%.2f" % (float(parameter), self.function_flag,
                                                                                               treatmrent, smoothing, coefCApos,  # pylint: disable=undefined-loop-variable
                                                                                               coefCBpos, coefCOpos, coefNpos, coefHpos,  # pylint: disable=undefined-loop-variable
                                                                                               coefHApos)  # pylint: disable=undefined-loop-variable
                            plot_file_name = "%s_%s_%s_%s_%s_%s" % (file, atoms_for_plot, comparison_method, smoothing, treatmrent, coefficients_for_plot)
                            if comparison_method == "correlation":
                                correlation_sorted.append([comparison_paramter, out_title, md_plot_data, dynamr_plot_data, plot_file_name, dynamr_2_md, coefficients])
            correlation_sorted.sort()

            if len(correlation_sorted) > 10:
                for corr_element in correlation_sorted[-11:]:
                    for dynadatacorr in corr_element[3]:
                        dyna_corr_residue_number = dynadatacorr[0]
                        dyna_corr_value = dynadatacorr[2]
                        #
                        # dyna_corr_residue_name = dynadatacorr[1]
                        # rci_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value, dyna_corr_residue_name)
                        # md_rmsd_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value * 29.55, dyna_corr_residue_name)
                        # nmr_rmsd_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value * 16.44, dyna_corr_residue_name)
                        # s2_out = "%s %s %s" % (dyna_corr_residue_number, 1 - (0.4 * log(1 + (dyna_corr_value * 17.7))), dyna_corr_residue_name)
                        #
                        if dyna_corr_residue_number in ret['seq_id']:
                            row_idx = ret['seq_id'].index(dyna_corr_residue_number)
                            ret['rci'][row_idx] = float(f'{dyna_corr_value:.3f}')
                            ret['nmr_rmsd'][row_idx] = float(f'{dyna_corr_value * 16.44:.3f}')
                            ret['s2'][row_idx] = float(f'{1 - (0.4 * log(1 + (dyna_corr_value * 17.7))):.3f}')

            else:
                for corr_element in correlation_sorted:
                    for dynadatacorr in corr_element[3]:
                        dyna_corr_residue_number = dynadatacorr[0]
                        dyna_corr_value = dynadatacorr[2]
                        #
                        # dyna_corr_residue_name = dynadatacorr[1]
                        # rci_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value, dyna_corr_residue_name)
                        # md_rmsd_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value * 29.55, dyna_corr_residue_name)
                        # nmr_rmsd_out = "%s %s %s" % (dyna_corr_residue_number, dyna_corr_value * 16.44, dyna_corr_residue_name)
                        # s2_out = "%s %s %s" % (dyna_corr_residue_number, 1 - (0.4 * log(1 + (dyna_corr_value * 17.7))), dyna_corr_residue_name)
                        #
                        if dyna_corr_residue_number in ret['seq_id']:
                            row_idx = ret['seq_id'].index(dyna_corr_residue_number)
                            ret['rci'][row_idx] = float(f'{dyna_corr_value:.3f}')
                            ret['nmr_rmsd'][row_idx] = float(f'{dyna_corr_value * 16.44:.3f}')
                            ret['s2'][row_idx] = float(f'{1 - (0.4 * log(1 + (dyna_corr_value * 17.7))):.3f}')

        return ret

    def __end_effect(self, l_list):
        """ Function1 to correct sigma for end effects.
        """

        result_list = []
        l_N_switch = l_C_switch = 0
        l_N_end_list = l_C_end_list = []
        l_N_mean = l_C_mean = 9999
        for l_item in l_list:
            l_res_num, l_sigma = l_item[0], l_item[2]
            if abs(l_res_num - self.__firstresidue) <= 4:  # 4 for DnaB
                l_N_end_list.append(l_sigma)
            elif abs(self.__lastresidue - l_res_num) <= 4:  # 4 for DnaB
                l_C_end_list.append(l_sigma)
        if len(l_N_end_list) > 0:
            l_N_mean = lmean(l_N_end_list)
        if len(l_C_end_list) > 0:
            l_C_mean = lmean(l_C_end_list)
        if self.N_term_low < l_N_mean < self.N_term_high:
            l_N_switch = 1
        if self.C_term_low < l_C_mean < self.C_term_high:
            l_C_switch = 1
        for l_item in l_list:
            (
                l_res_num,
                # l_res_name,
                l_sigma,
                # l_atomlist1,
                # l_atomlist2,
                # l_abs_or_true,
                # l_smooth,
            ) = (
                l_item[0],
                # l_item[1],
                l_item[2],
                # l_item[3],
                # l_item[4],
                # l_item[5],
                # l_item[6],
            )
            if abs(l_res_num - self.__firstresidue) <= self.N_term_res_num and l_N_switch == 1:
                l_sigma = l_sigma * self.N_term_coef * 10
                l_item[2] = l_sigma
                result_list.append(l_item)
            elif abs(self.__lastresidue - l_res_num) <= self.C_term_res_num and l_C_switch == 1:
                l_sigma = l_sigma * self.C_term_coef * 10
                l_item[2] = l_sigma
                result_list.append(l_item)
            else:
                result_list.append(l_item)

        return result_list

    def __end_effect2(self, l_list):
        """ Function2 to correct sigma for end effects.
        """

        result_list = []
        l_N_switch = l_C_switch = 0
        l_N_end_list = []
        l_C_end_list = []
        l_N_mean = l_C_mean = 9999
        for l_item in l_list:
            l_res_num, l_sigma = l_item[0], l_item[2]
            if abs(l_res_num - self.__firstresidue) <= 4:  # 4 for DnaB
                l_N_end_list.append(l_sigma)
            elif abs(self.__lastresidue - l_res_num) <= 4:  # 4 for DnaB
                l_C_end_list.append(l_sigma)
        if len(l_N_end_list) > 0:
            l_N_mean = lmean(l_N_end_list)
        if len(l_C_end_list) > 0:
            l_C_mean = lmean(l_C_end_list)
        if self.N_term_low < l_N_mean < self.N_term_high:
            l_N_switch = 1
        if self.C_term_low < l_C_mean < self.C_term_high:
            l_C_switch = 1
        for l_item in l_list:
            (
                l_res_num,
                # l_res_name,
                l_sigma,
                # l_atomlist1,
                # l_atomlist2,
                # l_abs_or_true,
                # l_smooth,
            ) = (
                l_item[0],
                # l_item[1],
                l_item[2],
                # l_item[3],
                # l_item[4],
                # l_item[5],
                # l_item[6],
            )
            if abs(l_res_num - self.__firstresidue) <= self.N_term_res_num and l_N_switch == 1:
                l_sigma = l_sigma + self.N_term_coef
                l_item[2] = l_sigma
                result_list.append(l_item)
            elif abs(self.__lastresidue - l_res_num) <= self.C_term_res_num and l_C_switch == 1:
                l_sigma = l_sigma + self.C_term_coef
                l_item[2] = l_sigma
                result_list.append(l_item)
            else:
                result_list.append(l_item)

        return result_list

    def __end_effect3(self, l_list):
        """ Function3 to correct sigma for end effects.
        """

        result_list = []
        # l_N_switch = l_C_switch = 0
        l_N_end_list = []
        l_C_end_list = []
        l_N_max = l_C_max = None
        for l_item in l_list:
            l_res_num, l_sigma = l_item[0], l_item[2]
            if abs(l_res_num - self.__firstresidue) <= 4:  # 4 for DnaB
                l_N_end_list.append([l_sigma, l_res_num])
            elif abs(self.__lastresidue - l_res_num) <= 4:  # 4 for DnaB
                l_C_end_list.append([l_sigma, l_res_num])
        if len(l_N_end_list) > 0:
            l_N_max = l_N_end_list[0][0]
            l_N_max_place = l_N_end_list[0][1]
            for l_N_end_list_item in l_N_end_list:
                if l_N_end_list_item[0] > l_N_max:
                    l_N_max = l_N_end_list_item[0]
                    l_N_max_place = l_N_end_list_item[1]
        if len(l_C_end_list) > 0:
            l_C_max = l_C_end_list[0][0]
            l_C_max_place = l_C_end_list[0][1]
            for l_C_end_list_item in l_C_end_list:
                if l_C_end_list_item[0] > l_C_max:
                    l_C_max = l_C_end_list_item[0]
                    l_C_max_place = l_C_end_list_item[1]
        for l_item in l_list:
            (
                l_res_num,
                # l_res_name,
                l_sigma,
                # l_atomlist1,
                # l_atomlist2,
                # l_abs_or_true,
                # l_smooth,
            ) = (
                l_item[0],
                # l_item[1],
                l_item[2],
                # l_item[3],
                # l_item[4],
                # l_item[5],
                # l_item[6],
            )
            if abs(l_res_num - self.__firstresidue) <= 3:  # 4 for DnaB
                if l_N_max is not None and l_sigma < l_N_max and l_res_num < l_N_max_place:
                    l_N_diff = l_sigma - l_N_max
                    l_sigma += abs(l_N_diff) * 2
                    if l_sigma > self.floor_value2:
                        l_sigma = self.floor_value2
                    l_item[2] = l_sigma
                    result_list.append(l_item)
                else:
                    result_list.append(l_item)
            elif abs(self.__lastresidue - l_res_num) <= 3:  # 4 for DnaB
                if l_C_max is not None and l_sigma < l_C_max and l_res_num > l_C_max_place:
                    l_C_diff = l_sigma - l_C_max
                    l_sigma += abs(l_C_diff) * 2
                    if l_sigma > self.floor_value2:
                        l_sigma = self.floor_value2
                    l_item[2] = l_sigma
                    result_list.append(l_item)
                else:
                    result_list.append(l_item)
            else:
                result_list.append(l_item)

        return result_list

    def __end_effect4(self, l_list):
        """ Function4 to correct sigma for end effects.
        """

        result_list = []
        l_N_end_list = []
        l_C_end_list = []
        l_N_max = None
        # l_N_min = None
        for l_item in l_list:
            l_res_num, l_sigma = l_item[0], l_item[2]
            if abs(l_res_num - self.__firstresidue) <= 4:  # 4 for DnaB
                l_N_end_list.append(l_sigma)
            elif abs(self.__lastresidue - l_res_num) <= 4:  # 4 for DnaB
                l_C_end_list.append(l_sigma)
        if len(l_N_end_list) > 0:
            l_N_max = max(l_N_end_list)
        if len(l_C_end_list) > 0:
            l_C_max = max(l_C_end_list)
        for l_item in l_list:
            (
                l_res_num,
                # l_res_name,
                l_sigma,
                # l_atomlist1,
                # l_atomlist2,
                # l_abs_or_true,
                # l_smooth,
            ) = (
                l_item[0],
                # l_item[1],
                l_item[2],
                # l_item[3],
                # l_item[4],
                # l_item[5],
                # l_item[6],
            )
            if abs(l_res_num - self.__firstresidue) <= 3:  # 4 for DnaB
                if l_N_max is not None and l_sigma < l_N_max:
                    l_item[2] = l_N_max
                    result_list.append(l_item)
                else:
                    result_list.append(l_item)
            elif abs(self.__lastresidue - l_res_num) <= 3:  # 4 for DnaB
                if l_C_max is not None and l_sigma < l_C_max:
                    l_item[2] = l_C_max
                    result_list.append(l_item)
                else:
                    result_list.append(l_item)
            else:
                result_list.append(l_item)

        return result_list

    def __end_effect5(self, l_list):
        """ Function5 to correct sigma for end effects.
        """

        result_list = []
        l_N_end_list = []
        l_C_end_list = []
        l_N_max = l_C_max = None
        for l_item in l_list:
            l_res_num, l_sigma = l_item[0], l_item[2]
            if abs(l_res_num - self.__firstresidue) <= 4:  # 4 for DnaB
                l_N_end_list.append(l_sigma)
            elif abs(self.__lastresidue - l_res_num) <= 4:  # 4 for DnaB
                l_C_end_list.append(l_sigma)
        if len(l_N_end_list) > 0:
            l_N_max = max(l_N_end_list)
        if len(l_C_end_list) > 0:
            l_C_max = max(l_C_end_list)
        for l_item in l_list:
            (
                l_res_num,
                # l_res_name,
                l_sigma,
                # l_atomlist1,
                # l_atomlist2,
                # l_abs_or_true,
                # l_smooth,
            ) = (
                l_item[0],
                # l_item[1],
                l_item[2],
                # l_item[3],
                # l_item[4],
                # l_item[5],
                # l_item[6],
            )
            if abs(l_res_num - self.__firstresidue) <= 3:  # 4 for DnaB
                if l_N_max is not None and l_sigma < l_N_max:
                    continue
                result_list.append(l_item)
            elif abs(self.__lastresidue - l_res_num) <= 3:  # 4 for DnaB
                if l_C_max is not None and l_sigma < l_C_max:
                    continue
                result_list.append(l_item)

            else:
                result_list.append(l_item)

        return result_list

    def __final_smoothing(self, L_smooth, L_list):
        """ Smoothing function for final result.
        """
        # Building residue number list
        first_residue = 0
        last_residue = 0
        first_residue_flag = 0
        smooth_factor = L_smooth
        central_residue = (smooth_factor // 2) + 1
        N_terminus_smooth_factor = central_residue
        # C_terminus_smooth_factor = smooth_factor // 2
        N_chem_shift_diff_abs_mean = None
        L_all = []
        ############################################
        # To find first and last residues
        ############################################
        for L_item in L_list:
            if first_residue_flag == 0:
                first_residue = L_item[0]
                first_residue_flag = 1
            else:
                last_residue = L_item[0]

        #################################################
        # Find residue number offset
        #################################################
        offset = first_residue - 1

        ###################################################
        # To build a list with residue numbers
        ###################################################
        residue_number_list = []
        for L_number in range(first_residue, (last_residue + 1)):
            residue_number_list.append(L_number)

        L_start = 0
        L_end = L_smooth
        if len(L_list) > 1:
            ####################################################
            # Smoothing N_terminus
            ####################################################
            Real_first_residue = L_list[0][0]
            Real_last_residue = L_list[-1][0]
            Old_first_residue = Real_first_residue
            negative_switch = 0
            L_abs_or_true = None
            L_first_smooth = None
            for L_number in range(0, (smooth_factor - N_terminus_smooth_factor)):
                L_chem_shift_diff_abs_list = []
                L_residue_number = L_list[L_number][0]
                L_residue_name = L_list[L_number][1]
                L_used_atoms = L_list[L_number][1]
                L_chem_shift_diff_abs = L_list[L_number][2]
                L_used_atoms = L_list[L_number][3]
                L_complete_atoms = L_list[L_number][4]
                L_abs_or_true = L_list[L_number][5]
                L_first_smooth = L_list[L_number][6]
                if L_chem_shift_diff_abs is not None:
                    L_chem_shift_diff_abs_list.append(L_chem_shift_diff_abs)
                Old_first_residue = L_residue_number
                D_residue_number = L_residue_number
                plus_done = 0
                D_counter = 1
                while not plus_done:
                    D_residue_number += 1
                    if abs(D_residue_number - Old_first_residue) <= central_residue:
                        for L_entry in L_list:
                            N_residue_number = L_entry[0]
                            if N_residue_number == D_residue_number:
                                N_chem_shift_diff_abs = L_entry[2]
                                if N_chem_shift_diff_abs is not None:
                                    L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                Old_first_residue = N_residue_number
                                D_counter += 1
                                if D_counter == central_residue:
                                    plus_done = 1
                    else:
                        plus_done = 1
                if negative_switch == 1:
                    Old_first_residue = L_residue_number
                    D_residue_number = L_residue_number
                    minus_done = 0
                    D_counter = 1
                    while not minus_done:
                        D_residue_number -= 1
                        if abs(D_residue_number - Old_first_residue) <= central_residue and D_residue_number > 0:
                            for L_entry in L_list:
                                N_residue_number = L_entry[0]
                                if N_residue_number == D_residue_number:
                                    N_chem_shift_diff_abs = L_entry[2]
                                    if N_chem_shift_diff_abs is not None:
                                        L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                    Old_first_residue = N_residue_number
                                    D_counter += 1
                                    if D_counter == (central_residue - 1):
                                        plus_done = 1
                        else:
                            minus_done = 1
                if len(L_chem_shift_diff_abs_list) > 0:
                    N_chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_list)
                    L_all.append([L_residue_number, L_residue_name, N_chem_shift_diff_abs_mean, L_used_atoms, L_complete_atoms, L_abs_or_true, L_first_smooth])
                negative_switch = 1

            #####################################################
            # Smoothing main
            #####################################################
            done = 0
            while not done:
                chem_shift_diff_abs_mean = None
                if L_end <= (last_residue - offset):  # For Ubiquitin

                    L_smooth_list = []
                    missing_chemshift = []
                    bigger_missing = 0
                    smaller_missing = 0
                    new_end = 0
                    new_start = 0
                    L_residue_number = None
                    L_residue_name = None
                    # chem_shift_diff_mean = None
                    chem_shift_diff_abs_mean = None
                    for L_number in residue_number_list[L_start:L_end]:
                        found_flag = 0
                        for L_item in L_list:
                            if (L_number == L_item[0]) and (L_item[2] is not None):
                                L_smooth_list.append(L_item)
                                found_flag = 1
                        if found_flag == 0:
                            missing_chemshift.append(L_number)
                    more_smaller = 0
                    more_bigger = 0

                    if len(L_smooth_list) == smooth_factor:

                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = L_smooth_list[central_residue - 1][0]
                        L_residue_name = L_smooth_list[central_residue - 1][1]
                        L_used_atoms = L_smooth_list[central_residue - 1][3]
                        L_complete_atoms = L_smooth_list[central_residue - 1][4]
                        L_abs_or_true = L_smooth_list[central_residue - 1][5]
                        L_first_smooth = L_smooth_list[central_residue - 1][6]

                        for entry in L_smooth_list:
                            L_chem_shift_diff_abs = entry[2]

                            if L_chem_shift_diff_abs is not None:
                                L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)

                        if len(L_chem_shift_diff_abs_tmp_list) > 0:
                            chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                            L_all.append([L_residue_number, L_residue_name, chem_shift_diff_abs_mean, L_used_atoms, L_complete_atoms, L_abs_or_true, L_first_smooth])

                        L_start += 1
                        L_end += 1

                    elif len(L_smooth_list) < smooth_factor:
                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = residue_number_list[(L_start):(L_end)][central_residue - 1]

                        for L_item in L_list:
                            if L_residue_number == L_item[0]:
                                L_residue_name = L_item[1]
                                L_used_atoms = L_item[3]
                                L_complete_atoms = L_item[4]
                                L_abs_or_true = L_item[5]
                                L_first_smooth = L_item[6]
                        for miss_number in missing_chemshift:
                            if miss_number > L_residue_number:
                                bigger_missing += 1
                            elif miss_number < L_residue_number:
                                smaller_missing += 1
                            elif miss_number == L_residue_number:
                                bigger_missing += 1
                        if bigger_missing > 0:
                            bigger_counter = 0
                            bigger_flag = 0
                            new_end = residue_number_list[L_end - 1]
                            while not bigger_flag:
                                new_end += 1
                                if new_end <= last_residue:
                                    if new_end < residue_number_list[L_end] + self.gap_limit_smooth:
                                        for L_item in L_list:
                                            if new_end == L_item[0] and L_item[2] is not None:
                                                L_smooth_list.append(L_item)
                                                bigger_counter += 1
                                                if bigger_counter == bigger_missing:
                                                    bigger_flag = 1

                                    else:
                                        bigger_flag = 1

                                else:
                                    more_smaller = bigger_missing - bigger_counter
                                    bigger_flag = 1

                        if smaller_missing > 0:
                            smaller_counter = 0
                            smaller_flag = 0
                            new_start = residue_number_list[L_start]
                            while not smaller_flag:
                                new_start -= 1
                                if new_start >= first_residue:
                                    if new_start >= residue_number_list[L_start] - self.gap_limit_smooth:
                                        for L_item in L_list:
                                            if new_start == L_item[0] and L_item[2] is not None:
                                                L_smooth_list.append(L_item)
                                                smaller_counter += 1
                                                if smaller_counter == smaller_missing:
                                                    smaller_flag = 1
                                    else:
                                        smaller_flag = 1

                                else:
                                    more_bigger = smaller_missing - smaller_counter
                                    smaller_flag = 1

                        if more_bigger > 0:
                            more_bigger_counter = 0
                            more_bigger_flag = 0
                            new_end = residue_number_list[L_end]
                            while not more_bigger_flag:
                                new_end += 1
                                if new_end <= last_residue and new_end < (residue_number_list[L_end] + self.gap_limit_smooth):
                                    for L_item in L_list:
                                        if new_end == L_item[0] and L_item[2] is not None:
                                            L_smooth_list.append(L_item)
                                            more_bigger_counter += 1
                                            if more_bigger_counter == more_bigger:
                                                more_bigger_flag = 1
                                else:
                                    more_bigger_flag = 1

                        if more_bigger > 0:
                            more_bigger_counter = 0
                            more_bigger_flag = 0
                            new_end = residue_number_list[L_end]
                            while not more_bigger_flag:
                                new_end += 1
                                if new_end <= last_residue and new_end < residue_number_list[L_end] + self.gap_limit_smooth:
                                    for L_item in L_list:
                                        if new_end == L_item[0]:
                                            L_smooth_list.append(L_item)
                                            more_bigger_counter += 1
                                            if more_bigger_counter == more_bigger:
                                                more_bigger_flag = 1
                                else:
                                    more_bigger_flag = 1

                        if more_smaller > 0:
                            more_smaller_counter = 0
                            more_smaller_flag = 0
                            new_start = residue_number_list[L_start]
                            while not more_smaller_flag:
                                new_start -= 1
                                if new_start >= first_residue and new_start >= residue_number_list[L_start] - self.gap_limit_smooth:
                                    for L_item in L_list:
                                        if new_start == L_item[0] and L_item[2] is not None:
                                            L_smooth_list.append(L_item)
                                            more_smaller_counter += 1
                                            if more_smaller_counter == more_smaller:
                                                more_smaller_flag = 1
                                else:
                                    more_smaller_flag = 1

                        for entry in L_smooth_list:
                            L_chem_shift_diff_abs = entry[2]
                            if L_chem_shift_diff_abs is not None:
                                L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)

                        if len(L_chem_shift_diff_abs_tmp_list) > 0 and L_residue_number != Real_first_residue and L_residue_number != Real_last_residue:
                            chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                            L_all.append([L_residue_number, L_residue_name, chem_shift_diff_abs_mean, L_used_atoms, L_complete_atoms, L_abs_or_true, L_first_smooth])

                        L_start += 1
                        L_end += 1

                    else:
                        L_chem_shift_diff_abs_tmp_list = []
                        L_residue_number = residue_number_list[L_start:L_end][central_residue - 1]
                        for L_item in L_list:
                            if L_residue_number == L_item[0]:
                                L_residue_name = L_item[1]
                                L_used_atoms = L_item[3]
                                L_complete_atoms = L_item[4]
                                L_abs_or_true = L_item[5]
                                L_first_smooth = L_item[6]
                        for entry in L_smooth_list:
                            L_chem_shift_diff_abs = entry[2]
                            if L_chem_shift_diff_abs is not None:
                                L_chem_shift_diff_abs_tmp_list.append(L_chem_shift_diff_abs)

                        if len(L_chem_shift_diff_abs_tmp_list) > 0 and L_residue_number != Real_first_residue and L_residue_number != Real_last_residue:
                            chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_tmp_list)
                            L_all.append([L_residue_number, L_residue_name, chem_shift_diff_abs_mean, L_used_atoms, L_complete_atoms, L_abs_or_true, L_first_smooth])

                        L_start += 1
                        L_end += 1

                else:
                    done = 1

            ####################################################
            # Smoothing C_terminus
            ####################################################
            negative_switch = 0
            Real_first_residue = L_list[0][0]
            Real_last_residue = L_list[1][0]
            Old_first_residue = Real_first_residue
            negative_switch = 0
            for L_number in range(1, (smooth_factor - N_terminus_smooth_factor + 1)):
                L_chem_shift_diff_abs_list = []
                L_residue_number = L_list[-L_number][0]
                L_residue_name = L_list[-L_number][1]
                L_chem_shift_diff_abs = L_list[-L_number][2]

                L_used_atoms = L_list[-L_number][3]
                L_complete_atoms = L_list[-L_number][4]
                L_abs_or_true = L_list[-L_number][5]
                L_first_smooth = L_list[-L_number][6]
                if L_chem_shift_diff_abs is not None:
                    L_chem_shift_diff_abs_list.append(L_chem_shift_diff_abs)
                Old_first_residue = L_residue_number
                D_residue_number = L_residue_number
                plus_done = 0
                D_counter = 1
                while not plus_done:
                    D_residue_number -= 1
                    if abs(D_residue_number - Old_first_residue) <= central_residue:
                        for L_entry in L_list:
                            N_residue_number = L_entry[0]
                            if N_residue_number == D_residue_number:
                                N_chem_shift_diff_abs = L_entry[2]
                                if N_chem_shift_diff_abs is not None:
                                    L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                Old_first_residue = N_residue_number
                                D_counter += 1
                                if D_counter == central_residue:
                                    plus_done = 1
                    else:
                        plus_done = 1
                if negative_switch == 1:
                    Old_first_residue = L_residue_number
                    D_residue_number = L_residue_number
                    minus_done = 0
                    D_counter = 1
                    while not minus_done:
                        D_residue_number += 1
                        if abs(D_residue_number - Old_first_residue) <= central_residue and D_residue_number < Real_last_residue:
                            for L_entry in L_list:
                                N_residue_number = L_entry[0]
                                if N_residue_number == D_residue_number:
                                    N_chem_shift_diff_abs = L_entry[2]
                                    if N_chem_shift_diff_abs is not None:
                                        L_chem_shift_diff_abs_list.append(N_chem_shift_diff_abs)
                                    Old_first_residue = N_residue_number
                                    D_counter += 1
                                    if D_counter == central_residue - 1:
                                        plus_done = 1
                        else:
                            minus_done = 1

                if len(L_chem_shift_diff_abs_list) > 0:
                    N_chem_shift_diff_abs_mean = lmean(L_chem_shift_diff_abs_list)
                    L_all.append([L_residue_number, L_residue_name, N_chem_shift_diff_abs_mean, L_used_atoms, L_complete_atoms, L_abs_or_true, L_first_smooth])

                negative_switch = 1

        return L_all

    def __combo(self, l_all_residues, s_best_ten_coef_corr_true, m_coeff_list):
        s_md_result, s_list_of_files = self.__rci_list_build(l_all_residues)
        (
            s_best_ten_coef_corr_true,
            s_list_of_files,
            s_Pearson_coeff,
            s_Spearman_coeff,
        ) = self.__rci_only(s_list_of_files, s_md_result, s_best_ten_coef_corr_true, m_coeff_list)
        return (s_best_ten_coef_corr_true, s_list_of_files, s_Pearson_coeff, s_Spearman_coeff)

    def __rci_list_build(self, l_all_residues):
        """ Function to build a list with chemical shifts for a stand-alone version.
        """

        l_md_result = []
        file_name = None
        for element in l_all_residues:
            residue_number = element[0]
            residue_name = element[1]
            value_true = element[2] / 1.125
            atoms = element[3]
            atoms_real = element[4]
            data_type = element[5]
            l_smooth_type = element[6]
            ratios = []
            complete_data_switch = 1
            for atomname in atoms_real:
                if atomname not in atoms:
                    complete_data_switch = self.incomplete_data_use
            if complete_data_switch == 1:
                if residue_number not in self.__noshift_res:
                    parameter_list = ([file_name] + [atoms, data_type, l_smooth_type] + [None, 0] + [value_true])
                    ratios.append(parameter_list)
                    l_md_result.append([residue_number, residue_name, ratios])

        l_list_of_files = [None]

        return (l_md_result, l_list_of_files)

    def __rci_only(self, m_list_of_files, m_md_result, m_best_ten_coef_corr_true, l_coeff_list):
        """ Function to create a list of RCI values for further analysis.
        """

        for filename in m_list_of_files:
            correlation_true_MD = []
            correlation_true_Dynamr = []
            md_data = []
            dynamr_data = []
            for entry in m_md_result:
                residue_number = entry[0]
                residue_name = entry[1]
                if len(entry[2]) > 0:
                    for cell in entry[2]:
                        file_name = cell[0]
                        m_atoms_real = cell[1]
                        m_data_type = cell[2]
                        m_smooth_type = cell[3]
                        if file_name is filename:
                            dynamr_value = cell[-1]
                            MD_value = cell[-2]
                            correlation_true_Dynamr.append(dynamr_value)
                            correlation_true_MD.append(MD_value)
                            md_data.append([residue_number, residue_name, MD_value])
                            dynamr_data.append([residue_number, residue_name, dynamr_value])

            dynamr_to_md_ratio = 0
            entry_found_true_corr = 0
            counter = 0
            corel_coef = l_Pearson_coeff = l_Spearman_coeff = 0
            if len(correlation_true_Dynamr) > 0 and len(correlation_true_MD) > 0:
                compare_list_corr_true = [abs(corel_coef), filename, "correlation", corel_coef, l_coeff_list, md_data, dynamr_data, dynamr_to_md_ratio]

                for item in m_best_ten_coef_corr_true:
                    if len(item) > 0:
                        name_of_file = item[0]
                        if name_of_file is filename:
                            if len(item[1]) < self.__number_of_best_matches:
                                if corel_coef not in m_best_ten_coef_corr_true[counter][5]:
                                    m_best_ten_coef_corr_true[counter][1].append(compare_list_corr_true)
                                    m_best_ten_coef_corr_true[counter][5].append(corel_coef)
                                    m_best_ten_coef_corr_true[counter][1].sort()
                            else:
                                m_best_ten_coef_corr_true[counter][1].sort()
                                if corel_coef not in m_best_ten_coef_corr_true[counter][5]:
                                    if abs(corel_coef) > m_best_ten_coef_corr_true[counter][1][0][0]:
                                        m_best_ten_coef_corr_true[counter][1][0] = compare_list_corr_true
                                        m_best_ten_coef_corr_true[counter][5].append(corel_coef)
                                        m_best_ten_coef_corr_true[counter][1].sort()

                            entry_found_true_corr = 1
                    counter += 1
                if entry_found_true_corr == 0:
                    m_best_ten_coef_corr_true.append([filename, [compare_list_corr_true], m_atoms_real, m_data_type, m_smooth_type, [corel_coef]])

        return (m_best_ten_coef_corr_true, m_list_of_files, l_Pearson_coeff, l_Spearman_coeff)

    def calculate(self, bmrb_to_aa_list, assignment, B_Cys, noshift_res):  # pylint: disable=unused-argument,inconsistent-return-statements
        """ Calculate Random Coil Index of given assignments
        """

        self.__bmrb_to_aa_list = bmrb_to_aa_list
        # self.__assignment = assignment
        self.__B_Cys = B_Cys
        self.__noshift_res = noshift_res

        self.__csi_list = self.__read_csi_pseudo()

        self.__firstresidue = self.__bmrb_to_aa_list[0][1]
        self.__lastresidue = self.__bmrb_to_aa_list[-1][1]

        self.__excluded_residues = []

        self.__simpred_lines = []

        for number in range(self.__firstresidue, self.__lastresidue + 1):
            NoNextProPro = 0

            res_i = number
            res_i_minus_1 = number - 1
            res_i_plus_1 = number + 1
            res_i_minus_2 = number - 2
            res_i_plus_2 = number + 2

            res_i_found = 0
            res_i_minus_1_found = 0
            res_i_plus_1_found = 0

            res_i_found = 0
            # res_i_minus_2_found = 0
            # res_i_plus_2_found = 0

            dyna_simp_res_num_i = None
            dyna_simp_res_name_i = None
            dyna_simp_res_num_i_minus_1 = None
            dyna_simp_res_name_i_minus_1 = None
            dyna_simp_res_num_i_plus_1 = None
            dyna_simp_res_name_i_plus_1 = None
            dyna_simp_res_num_i_minus_2 = None
            dyna_simp_res_name_i_minus_2 = None
            dyna_simp_res_num_i_plus_2 = None
            dyna_simp_res_name_i_plus_2 = None

            res_num_i_sec_str_found = 0
            res_num_i_minus_1_sec_str_found = 0
            res_num_i_plus_1_sec_str_found = 0
            res_num_i_minus_2_sec_str_found = 0
            res_num_i_plus_2_sec_str_found = 0

            preceed_res_cor_N = None
            preceed_res_cor_CO = None
            preceed_res_cor_CA = None
            preceed_res_cor_CB = None
            preceed_res_cor_NH = None
            preceed_res_cor_HA = None

            preceed_preceed_res_cor_N = None
            preceed_preceed_res_cor_CO = None
            preceed_preceed_res_cor_CA = None
            preceed_preceed_res_cor_CB = None
            preceed_preceed_res_cor_NH = None
            preceed_preceed_res_cor_HA = None

            next_res_cor_N = None
            next_res_cor_CO = None
            next_res_cor_CA = None
            next_res_cor_CB = None
            next_res_cor_NH = None
            next_res_cor_HA = None

            next_next_res_cor_N = None
            next_next_res_cor_CO = None
            next_next_res_cor_CA = None
            next_next_res_cor_CB = None
            next_next_res_cor_NH = None
            next_next_res_cor_HA = None

            final_N = None
            final_CO = None
            final_CA = None
            final_CB = None
            final_NH = None
            final_HA = None

            res_i_csi_sec_str = "C"
            res_i_minus_1_csi_sec_str = "C"
            res_i_plus_1_csi_sec_str = "C"
            res_i_minus_2_csi_sec_str = "C"
            res_i_plus_2_csi_sec_str = "C"

            for bmrb_to_aa_entry in self.__bmrb_to_aa_list:
                dyna_simp_res_num = bmrb_to_aa_entry[1]
                dyna_simp_res_name = self.__aa_names_full_all_CAP[bmrb_to_aa_entry[0][0:3]]

                # BCys res name correction
                if dyna_simp_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                    dyna_simp_res_name = "B"
                if dyna_simp_res_num == res_i:
                    dyna_simp_res_num_i = dyna_simp_res_num
                    dyna_simp_res_name_i = dyna_simp_res_name
                    res_i_found = 1
                    for csi_item in self.__csi_list:
                        if csi_item[0] != "#":
                            # res_num_i_sec_str = "C"
                            csi_res_num = csi_item[0]
                            csi_res_name = csi_item[1]
                            # BCys res name correction
                            if csi_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                                csi_res_name = "B"

                            csi_sec_str = csi_item[2]
                            if csi_res_num == dyna_simp_res_num_i and csi_res_name == dyna_simp_res_name_i:
                                res_i_csi_sec_str = csi_sec_str
                                res_num_i_sec_str_found = 1
                            elif csi_res_num == dyna_simp_res_num_i and csi_res_name != dyna_simp_res_name_i:
                                logwrite = "Residue %s is %s in CSI index whereas it is %s in Simpred output of Dynamr\n" % (csi_res_num, csi_res_name, dyna_simp_res_name_i)
                                if self.__verbose:
                                    self.__lfh.write(logwrite)

                if dyna_simp_res_num == res_i_minus_1:
                    # res_num_i_minus_1_sec_str = "C"
                    dyna_simp_res_num_i_minus_1 = dyna_simp_res_num
                    dyna_simp_res_name_i_minus_1 = dyna_simp_res_name
                    res_i_minus_1_found = 1
                    for csi_item in self.__csi_list:
                        if csi_item[0] != "#":
                            csi_res_num = csi_item[0]
                            csi_res_name = csi_item[1]
                            # BCys res name correction
                            if csi_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                                csi_res_name = "B"
                            csi_sec_str = csi_item[2]
                            if csi_res_num == dyna_simp_res_num_i_minus_1 and csi_res_name == dyna_simp_res_name_i_minus_1:
                                res_i_minus_1_csi_sec_str = csi_sec_str
                                res_num_i_minus_1_sec_str_found = 1
                            elif csi_res_num == dyna_simp_res_num_i_minus_1 and csi_res_name != dyna_simp_res_name_i_minus_1:
                                logwrite = "Residue %s is %s in CSI index whereas it is %s in Simpred output of Dynamr\n" % (csi_res_num, csi_res_name, dyna_simp_res_name_i_minus_1)  # noqa: E501
                                if self.__verbose:
                                    self.__lfh.write(logwrite)

                if dyna_simp_res_num == res_i_minus_2:
                    # res_num_i_minus_2_sec_str = "C"
                    dyna_simp_res_num_i_minus_2 = dyna_simp_res_num
                    dyna_simp_res_name_i_minus_2 = dyna_simp_res_name
                    # res_i_minus_2_found = 1
                    for csi_item in self.__csi_list:
                        if csi_item[0] != "#":
                            csi_res_num = csi_item[0]
                            csi_res_name = csi_item[1]
                            # BCys res name correction
                            if csi_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                                csi_res_name = "B"
                            csi_sec_str = csi_item[2]
                            if csi_res_num == dyna_simp_res_num_i_minus_2 and csi_res_name == dyna_simp_res_name_i_minus_2:
                                res_i_minus_2_csi_sec_str = csi_sec_str
                                res_num_i_minus_2_sec_str_found = 1
                            elif csi_res_num == dyna_simp_res_num_i_minus_2 and csi_res_name != dyna_simp_res_name_i_minus_2:
                                logwrite = "Residue %s is %s in CSI index whereas it is %s in Simpred output of Dynamr\n" % (csi_res_num, csi_res_name, dyna_simp_res_name_i_minus_2)  # noqa: E501
                                if self.__verbose:
                                    self.__lfh.write(logwrite)

                if dyna_simp_res_num == res_i_plus_1:
                    # res_num_i_plus_1_sec_str = "C"
                    dyna_simp_res_num_i_plus_1 = dyna_simp_res_num
                    dyna_simp_res_name_i_plus_1 = dyna_simp_res_name
                    res_i_plus_1_found = 1
                    for csi_item in self.__csi_list:
                        if csi_item[0] != "#":
                            csi_res_num = csi_item[0]
                            csi_res_name = csi_item[1]
                            # BCys res name correction
                            if csi_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                                csi_res_name = "B"
                            csi_sec_str = csi_item[2]
                            if csi_res_num == dyna_simp_res_num_i_plus_1 and csi_res_name == dyna_simp_res_name_i_plus_1:
                                res_i_plus_1_csi_sec_str = csi_sec_str
                                res_num_i_plus_1_sec_str_found = 1
                            elif csi_res_num == dyna_simp_res_num_i_plus_1 and csi_res_name != dyna_simp_res_name_i_plus_1:
                                logwrite = "Residue %s is %s in CSI index whereas it is %s in Simpred output of Dynamr\n" % (csi_res_num, csi_res_name, dyna_simp_res_name_i_plus_1)
                                if self.__verbose:
                                    self.__lfh.write(logwrite)

                if dyna_simp_res_num == res_i_plus_2:
                    # res_num_i_plus_2_sec_str = "C"
                    dyna_simp_res_num_i_plus_2 = dyna_simp_res_num
                    dyna_simp_res_name_i_plus_2 = dyna_simp_res_name
                    res_i_plus_1_found = 2
                    for csi_item in self.__csi_list:
                        if csi_item[0] != "#":
                            csi_res_num = csi_item[0]
                            csi_res_name = csi_item[1]
                            # BCys res name correction
                            if csi_res_num in self.__B_Cys and self.B_Cys_switch == 1:
                                csi_res_name = "B"
                            csi_sec_str = csi_item[2]
                            if csi_res_num == dyna_simp_res_num_i_plus_2 and csi_res_name == dyna_simp_res_name_i_plus_2:
                                res_i_plus_2_csi_sec_str = csi_sec_str
                                res_num_i_plus_2_sec_str_found = 1
                            elif csi_res_num == dyna_simp_res_num_i_plus_2 and csi_res_name != dyna_simp_res_name_i_plus_2:
                                logwrite = "Residue %s is %s in CSI index whereas it is %s in Simpred output of Dynamr\n" % (csi_res_num, csi_res_name, dyna_simp_res_name_i_plus_2)
                                if self.__verbose:
                                    self.__lfh.write(logwrite)

            if self.NoNextProPro_flag == 1 and ((dyna_simp_res_name_i_plus_1 == dyna_simp_res_name_i == "P") or (dyna_simp_res_name_i_plus_1 == dyna_simp_res_name_i_plus_2 == "P")):  # noqa: E501
                NoNextProPro = 1

            if dyna_simp_res_num_i is not None and dyna_simp_res_name_i is not None:
                N_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__N_place]
                CO_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__CO_place]
                CA_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__CA_place]
                CB_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__CB_place]
                NH_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__NH_place]
                HA_coil_value = self.random_coil_dic[dyna_simp_res_name_i][self.__HA_place]

            if self.preceed_res_effect_flag == 1 and dyna_simp_res_num_i_minus_1 is not None and dyna_simp_res_name_i_minus_1 is not None:
                if res_i_csi_sec_str == res_i_minus_1_csi_sec_str and res_num_i_minus_1_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    preceed_res_cor_N = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__N_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CO = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CO_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CA_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CB = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CB_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_NH = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__NH_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_HA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__HA_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                elif res_num_i_minus_1_sec_str_found == 0:
                    preceed_res_cor_N = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CO = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CB = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_NH = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_HA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
                elif res_num_i_sec_str_found == 0:
                    preceed_res_cor_N = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__N_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CO = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CO_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CA_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_CB = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CB_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_NH = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__NH_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                    preceed_res_cor_HA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__HA_place][self.__sec_str_place[res_i_minus_1_csi_sec_str]]
                if res_i_csi_sec_str != res_i_minus_1_csi_sec_str and res_num_i_minus_1_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    preceed_res_cor_N = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CO = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_CB = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_NH = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_res_cor_HA = self.preceed_res_effect[dyna_simp_res_name_i_minus_1][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
            else:
                preceed_res_cor_N = 0.0
                preceed_res_cor_CO = 0.0
                preceed_res_cor_CA = 0.0
                preceed_res_cor_CB = 0.0
                preceed_res_cor_NH = 0.0
                preceed_res_cor_HA = 0.0

            if self.preceed_res_effect_flag != 1:
                preceed_res_cor_N = 0.0
                preceed_res_cor_CO = 0.0
                preceed_res_cor_CA = 0.0
                preceed_res_cor_CB = 0.0
                preceed_res_cor_NH = 0.0
                preceed_res_cor_HA = 0.0

            if self.preceed_res_effect_flag == 1 and self.preceed_preceed_res_effect_flag == 1 and dyna_simp_res_num_i_minus_2 is not None and dyna_simp_res_name_i_minus_2 is not None:  # noqa: E501
                if res_i_csi_sec_str == res_i_minus_2_csi_sec_str and res_num_i_minus_2_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    preceed_preceed_res_cor_N = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__N_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CO = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CO_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CA_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CB = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CB_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_NH = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__NH_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_HA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__HA_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                elif res_num_i_minus_2_sec_str_found == 0:
                    preceed_preceed_res_cor_N = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CO = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CB = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_NH = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_HA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
                elif res_num_i_sec_str_found == 0:
                    preceed_preceed_res_cor_N = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__N_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CO = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CO_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CA_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_CB = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CB_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_NH = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__NH_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                    preceed_preceed_res_cor_HA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__HA_place][self.__sec_str_place[res_i_minus_2_csi_sec_str]]
                if res_i_csi_sec_str != res_i_minus_2_csi_sec_str and res_num_i_minus_2_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    preceed_preceed_res_cor_N = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CO = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_CB = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_NH = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    preceed_preceed_res_cor_HA = self.preceed_preceed_res_effect[dyna_simp_res_name_i_minus_2][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
            else:
                preceed_preceed_res_cor_N = 0.0
                preceed_preceed_res_cor_CO = 0.0
                preceed_preceed_res_cor_CA = 0.0
                preceed_preceed_res_cor_CB = 0.0
                preceed_preceed_res_cor_NH = 0.0
                preceed_preceed_res_cor_HA = 0.0

            if self.preceed_preceed_res_effect_flag != 1:
                preceed_preceed_res_cor_N = 0.0
                preceed_preceed_res_cor_CO = 0.0
                preceed_preceed_res_cor_CA = 0.0
                preceed_preceed_res_cor_CB = 0.0
                preceed_preceed_res_cor_NH = 0.0
                preceed_preceed_res_cor_HA = 0.0

            if self.next_res_effect_flag == 1 and dyna_simp_res_num_i_plus_1 is not None and dyna_simp_res_name_i_plus_1 is not None and NoNextProPro != 1:
                if res_i_csi_sec_str == res_i_plus_1_csi_sec_str and res_num_i_plus_1_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    next_res_cor_N = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__N_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CO = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CO_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CA_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CB = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CB_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_NH = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__NH_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_HA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__HA_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                elif res_num_i_plus_1_sec_str_found == 0:
                    next_res_cor_N = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CO = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CB = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_NH = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_HA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
                elif res_num_i_sec_str_found == 0:
                    next_res_cor_N = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__N_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CO = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CO_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CA_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_CB = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CB_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_NH = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__NH_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                    next_res_cor_HA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__HA_place][self.__sec_str_place[res_i_plus_1_csi_sec_str]]
                if res_i_csi_sec_str != res_i_plus_1_csi_sec_str and res_num_i_plus_1_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    next_res_cor_N = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CO = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_CB = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_NH = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_res_cor_HA = self.next_res_effect[dyna_simp_res_name_i_plus_1][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
            else:
                next_res_cor_N = 0.0
                next_res_cor_CO = 0.0
                next_res_cor_CA = 0.0
                next_res_cor_CB = 0.0
                next_res_cor_NH = 0.0
                next_res_cor_HA = 0.0

            if self.next_res_effect_flag != 1:
                next_res_cor_N = 0.0
                next_res_cor_CO = 0.0
                next_res_cor_CA = 0.0
                next_res_cor_CB = 0.0
                next_res_cor_NH = 0.0
                next_res_cor_HA = 0.0

            if self.next_res_effect_flag == 1 and self.next_next_res_effect_flag == 1 and dyna_simp_res_num_i_plus_2 is not None and dyna_simp_res_name_i_plus_2 is not None and NoNextProPro != 1:  # noqa: E501
                if res_i_csi_sec_str == res_i_plus_2_csi_sec_str and res_num_i_plus_2_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    next_next_res_cor_N = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__N_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CO = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CO_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CA_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CB = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CB_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_NH = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__NH_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_HA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__HA_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                elif res_num_i_plus_2_sec_str_found == 0:
                    next_next_res_cor_N = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CO = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CB = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_NH = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_HA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
                elif res_num_i_sec_str_found == 0:
                    next_next_res_cor_N = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__N_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CO = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CO_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CA_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_CB = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CB_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_NH = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__NH_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                    next_next_res_cor_HA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__HA_place][self.__sec_str_place[res_i_plus_2_csi_sec_str]]
                if res_i_csi_sec_str != res_i_plus_2_csi_sec_str and res_num_i_plus_2_sec_str_found == 1 and res_num_i_sec_str_found == 1:
                    next_next_res_cor_N = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__N_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CO = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CO_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CA_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_CB = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__CB_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_NH = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__NH_place][self.__sec_str_place[res_i_csi_sec_str]]
                    next_next_res_cor_HA = self.next_next_res_effect[dyna_simp_res_name_i_plus_2][self.__HA_place][self.__sec_str_place[res_i_csi_sec_str]]
            else:
                next_next_res_cor_N = 0.0
                next_next_res_cor_CO = 0.0
                next_next_res_cor_CA = 0.0
                next_next_res_cor_CB = 0.0
                next_next_res_cor_NH = 0.0
                next_next_res_cor_HA = 0.0

            if self.next_next_res_effect_flag != 1:
                next_next_res_cor_N = 0.0
                next_next_res_cor_CO = 0.0
                next_next_res_cor_CA = 0.0
                next_next_res_cor_CB = 0.0
                next_next_res_cor_NH = 0.0
                next_next_res_cor_HA = 0.0

            if N_coil_value is not None:
                final_N = N_coil_value + next_res_cor_N + preceed_res_cor_N + next_next_res_cor_N + preceed_preceed_res_cor_N
            if CO_coil_value is not None:
                final_CO = CO_coil_value + next_res_cor_CO + preceed_res_cor_CO + next_next_res_cor_CO + preceed_preceed_res_cor_CO
            if CA_coil_value is not None:
                final_CA = CA_coil_value + next_res_cor_CA + preceed_res_cor_CA + next_next_res_cor_CA + preceed_preceed_res_cor_CA
            if CB_coil_value is not None:
                final_CB = CB_coil_value + next_res_cor_CB + preceed_res_cor_CB + next_next_res_cor_CB + preceed_preceed_res_cor_CB
            if NH_coil_value is not None:
                final_NH = NH_coil_value + next_res_cor_NH + preceed_res_cor_NH + next_next_res_cor_NH + preceed_preceed_res_cor_NH
            if HA_coil_value is not None:
                final_HA = HA_coil_value + next_res_cor_HA + preceed_res_cor_HA + next_next_res_cor_HA + preceed_preceed_res_cor_HA

            self.__simpred_lines.append([dyna_simp_res_num_i, dyna_simp_res_name_i, final_CA, final_CB, final_CO, final_N, final_NH, final_HA])

            if res_i_found == 0:
                logwrite = "Warning. Residue i,%s, has not been found\n" % res_i
                if self.__verbose:
                    self.__lfh.write(logwrite)
            if res_i_minus_1_found == 0:
                logwrite = "Warning. Residue i-1,%s, for residue %s has not been found\n" % (res_i_minus_1, res_i)
                if self.__verbose:
                    self.__lfh.write(logwrite)
            if res_i_plus_1_found == 0:
                logwrite = "Warning. Residue i+1,%s, for residue %s has not been found\n" % (res_i_plus_1, res_i)
                if self.__verbose:
                    self.__lfh.write(logwrite)
            if res_num_i_minus_1_sec_str_found == 0:
                logwrite = "Warning. Secondary structure type of residue i-1,%s,(for residue %s) has not been found. Random coil type is used.\n" % (res_i_plus_1, res_i)
                if self.__verbose:
                    self.__lfh.write(logwrite)
            if res_num_i_plus_1_sec_str_found == 0:
                logwrite = "Warning. Secondary structure type of residue i+1,%s,(for residue %s) has not been found. Random coil type is used.\n" % (res_i_minus_1, res_i)
                if self.__verbose:
                    self.__lfh.write(logwrite)
            if res_num_i_sec_str_found == 0:
                logwrite = "Warning. Secondary structure type of residue i,%s, has not been found. Random coil type is used.\n" % res_i
                if self.__verbose:
                    self.__lfh.write(logwrite)

        self.__number_of_best_matches += 1

        self.__smooth_types = []
        if self.three_point_smooth_flag == 1:
            self.__smooth_types.append([None, 1, 3, 1])
        if self.five_point_smooth_flag == 1:
            self.__smooth_types.append([None, 2, 5, 1])
        if self.double_three_point_smooth_flag == 1:
            self.__smooth_types.append([None, 1, 3, 2])

        self.__all_smoothed = []
        # self.__all_atoms_no_smooth = []
        self.__all_atoms_no_smooth_gap_fill = []

        self.__atom_list = self.__write_atom_list()

        self.__real_BMRB_first_res = 9999
        self.__real_BMRB_last_res = 0

        for atom_type in self.__atom_list:
            no_smooth_list = []
            no_smooth_list2 = []

            for v in assignment:
                residue_name_init, residue_number, bmrb_atom_name, bmrb_shift = (v[0], v[1], v[2], v[4])
                residue_name = self.__aa_names_full_all_CAP[residue_name_init]

                if bmrb_atom_name == atom_type or ("HA" in bmrb_atom_name and atom_type == "HA"):

                    if residue_number < self.__real_BMRB_first_res:
                        self.__real_BMRB_first_res = residue_number
                    if residue_number > self.__real_BMRB_last_res:
                        self.__real_BMRB_last_res = residue_number

                    if residue_number in self.__B_Cys and self.B_Cys_switch == 1:
                        residue_name = "B"
                    elif residue_number in self.__B_Cys and self.B_Cys_switch == 2 and atom_type == "CB":
                        residue_name = "X"

                    simpred_shift = None
                    for split_entry in self.__simpred_lines:

                        if residue_name == split_entry[1] and residue_number == split_entry[0]:
                            simpred_shift = split_entry[self.__atom_place_in_simpred[atom_type]]

                            if simpred_shift is not None:
                                chem_shift_diff = bmrb_shift - simpred_shift

                                if 0.0 < chem_shift_diff < self.early_floor_value_pos / self.__Hertz_correction_dic[atom_type]:
                                    chem_shift_diff = self.early_floor_value_pos / self.__Hertz_correction_dic[atom_type]
                                elif 0.0 > chem_shift_diff > self.early_floor_value_neg / self.__Hertz_correction_dic[atom_type]:
                                    chem_shift_diff = self.early_floor_value_neg / self.__Hertz_correction_dic[atom_type]

                                chem_shift_diff_abs = abs(chem_shift_diff)

                                if residue_name == "B":
                                    residue_name = "C"

                                no_smooth_list.append([residue_number, residue_name, bmrb_shift, simpred_shift, chem_shift_diff, chem_shift_diff_abs])
                                no_smooth_list2.append([residue_number, residue_name, bmrb_shift, simpred_shift, chem_shift_diff, chem_shift_diff_abs, atom_type])

            if self.gap_fill_flag == 0:
                list_for_smooth = no_smooth_list2
            elif self.gap_fill_flag == 1:
                list_for_smooth = self.__gap_fill(no_smooth_list2, atom_type)
            else:
                list_for_smooth = self.__gap_fill2(no_smooth_list2, atom_type)

            self.__all_atoms_no_smooth_gap_fill.append(list_for_smooth)

            for smoothtype in self.__smooth_types:
                smoothed_data = self.__smoothing(smoothtype, list_for_smooth, atom_type)
                self.__all_smoothed.extend(smoothed_data)

        if self.five_point_smooth_flag == 1:
            five_point_smooth_list = []
            for lists in self.__all_smoothed:
                smooth_factor = lists[8]
                if smooth_factor == 5:
                    five_point_smooth_list.append(lists)

        if self.three_point_smooth_flag == 1:
            three_point_smooth_list = []
            for lists in self.__all_smoothed:
                smooth_factor = lists[8]
                if smooth_factor == 3:
                    three_point_smooth_list.append(lists)

        if self.five_point_smooth_flag == 1:
            return self.__grid_search(five_point_smooth_list, "5_pnt_smooth", self.COp_list, self.CAp_list, self.CBp_list, self.HAp_list, self.Hp_list, self.Np_list, self.final_smooth)  # noqa: E501

        if self.three_point_smooth_flag == 1:
            return self.__grid_search(three_point_smooth_list, "3_pnt_smooth", self.COp_list, self.CAp_list, self.CBp_list, self.HAp_list, self.Hp_list, self.Np_list, self.final_smooth)  # noqa: E501

        if self.no_smoothing_flag == 1:
            return self.__grid_search(self.__all_atoms_no_smooth_gap_fill,
                                      "no_smooth", self.COp_list,
                                      self.CAp_list, self.CBp_list,
                                      self.HAp_list, self.Hp_list,
                                      self.Np_list, self.final_smooth)
