class AsyncMalwareSettings:
    def __init__(self, session):
        super().__init__()
        self._session = session
    
    async def getNetworkSecurityMalwareSettings(self, networkId: str):
        """
        **Returns all supported malware settings for an MX network**
        https://developer.cisco.com/meraki/api/#!get-network-security-malware-settings
        
        - networkId (string)
        """

        metadata = {
            'tags': ['Malware settings'],
            'operation': 'getNetworkSecurityMalwareSettings',
        }
        resource = f'/networks/{networkId}/security/malwareSettings'

        return await self._session.get(metadata, resource)

    async def updateNetworkSecurityMalwareSettings(self, networkId: str, mode: str, **kwargs):
        """
        **Set the supported malware settings for an MX network**
        https://developer.cisco.com/meraki/api/#!update-network-security-malware-settings
        
        - networkId (string)
        - mode (string): Set mode to 'enabled' to enable malware prevention, otherwise 'disabled'
        - allowedUrls (array): The urls that should be permitted by the malware detection engine. If omitted, the current config will remain unchanged. This is available only if your network supports AMP allow listing
        - allowedFiles (array): The sha256 digests of files that should be permitted by the malware detection engine. If omitted, the current config will remain unchanged. This is available only if your network supports AMP allow listing
        """

        kwargs.update(locals())

        if 'mode' in kwargs:
            options = ['enabled', 'disabled']
            assert kwargs['mode'] in options, f'''"mode" cannot be "{kwargs['mode']}", & must be set to one of: {options}'''

        metadata = {
            'tags': ['Malware settings'],
            'operation': 'updateNetworkSecurityMalwareSettings',
        }
        resource = f'/networks/{networkId}/security/malwareSettings'

        body_params = ['mode', 'allowedUrls', 'allowedFiles']
        payload = {k.strip(): v for (k, v) in kwargs.items() if k.strip() in body_params}

        return await self._session.put(metadata, resource, payload)

