"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const toml_1 = require("@iarna/toml");
const param_case_1 = require("param-case");
const pascal_case_1 = require("pascal-case");
const snake_case_1 = require("snake-case");
/**
 * @stability stable
 */
class ConfigurationMapper {
    constructor(props) {
        this.props = props;
    }
    /**
     * @stability stable
     */
    static withDefaults(props) {
        const { globalConfiguration, runnerConfiguration, dockerConfiguration, machineConfiguration, autoscalingConfigurations, cacheConfiguration, } = props;
        return new ConfigurationMapper({
            globalConfiguration: {
                concurrent: 10,
                checkInterval: 0,
                logFormat: "runner",
                logLevel: "info",
                ...globalConfiguration,
            },
            runnerConfiguration: {
                name: "gitlab-runner",
                url: "https://gitlab.com",
                limit: 10,
                outputLimit: 52428800,
                executor: "docker+machine",
                environment: ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                ...runnerConfiguration,
            },
            dockerConfiguration: {
                tlsVerify: false,
                image: "docker:19.03.5",
                privileged: true,
                capAdd: ["CAP_SYS_ADMIN"],
                waitForServicesTimeout: 300,
                disableCache: false,
                volumes: ["/certs/client", "/cache"],
                shmSize: 0,
                ...dockerConfiguration,
            },
            machineConfiguration: {
                idleCount: 0,
                idleTime: 300,
                maxBuilds: 20,
                machineDriver: "amazonec2",
                machineName: "gitlab-runner-%s",
                ...machineConfiguration,
                machineOptions: {
                    requestSpotInstance: true,
                    spotPrice: 0.03,
                    ...machineConfiguration.machineOptions,
                },
            },
            autoscalingConfigurations: autoscalingConfigurations.length
                ? autoscalingConfigurations
                : [
                    {
                        periods: ["* * 7-22 * * mon-fri *"],
                        idleCount: 1,
                        idleTime: 1800,
                        timezone: "Etc/UTC",
                    },
                ],
            cacheConfiguration: {
                type: "s3",
                shared: true,
                ...cacheConfiguration,
            },
        });
    }
    /**
     * @stability stable
     */
    static fromProps(props) {
        return new ConfigurationMapper(props);
    }
    /**
     * @stability stable
     */
    toToml() {
        return toml_1.stringify(this._toJsonMap());
    }
    /**
     * @internal
     */
    _toJsonMap() {
        const { globalConfiguration, runnerConfiguration, dockerConfiguration, machineConfiguration, autoscalingConfigurations, cacheConfiguration, } = this.props;
        const result = toJsonMap(globalConfiguration, snake_case_1.snakeCase);
        const runner = toJsonMap(runnerConfiguration, snake_case_1.snakeCase);
        result.runners = [runner];
        runner.docker = toJsonMap(dockerConfiguration, snake_case_1.snakeCase);
        const machine = toJsonMap(machineConfiguration, pascal_case_1.pascalCase);
        machine.MachineOptions = toProperties(machineConfiguration.machineOptions, (key) => `amazonec2-${param_case_1.paramCase(key)}`);
        machine.autoscaling = autoscalingConfigurations.map((autoscaling) => toJsonMap(autoscaling, pascal_case_1.pascalCase));
        runner.machine = machine;
        const cache = toJsonMap(cacheConfiguration, pascal_case_1.pascalCase);
        delete cache.S3;
        cache.s3 = toJsonMap(cacheConfiguration.s3, pascal_case_1.pascalCase);
        runner.cache = cache;
        return result;
    }
}
exports.ConfigurationMapper = ConfigurationMapper;
_a = JSII_RTTI_SYMBOL_1;
ConfigurationMapper[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.ConfigurationMapper", version: "0.0.155" };
/**
 * Transforms configuration objects to JsonMap. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 */
function toJsonMap(configuration, inflector) {
    const result = {};
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result[inflector(key)] = value;
    }
    return result;
}
/**
 * Transforms configuration objects to a property array. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 * @example
 * // returns ["foo=bar"]
 * toProperties({foo: "bar", (s) => s});
 */
function toProperties(configuration, inflector) {
    const result = [];
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result.push(`${inflector(key)}=${value}`);
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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