"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 * @stability stable
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter.
     *
     * @stability stable
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:.
 *
 * - Setting session variables and configuration settings.
 *
 * @stability stable
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster.
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 *
 * @stability stable
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance.
     *
     * @param instance RDS database instance.
     * @stability stable
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster.
     *
     * @param cluster RDS database cluster.
     * @stability stable
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     *
     * @stability stable
     */
    bind(_) {
        var _a, _b, _c, _d;
        const engine = (_b = (_a = this.dbInstance) === null || _a === void 0 ? void 0 : _a.engine) !== null && _b !== void 0 ? _b : (_c = this.dbCluster) === null || _c === void 0 ? void 0 : _c.engine;
        if (!engine) {
            const errorResource = (_d = this.dbCluster) !== null && _d !== void 0 ? _d : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy.
 *
 * @stability stable
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     *
     * @stability stable
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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