"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).
 *
 * @stability stable
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string, for example "5.6.mysql_aurora.1.78.3.6".
     * @param auroraMajorVersion the major version of the engine, defaults to "5.6".
     * @stability stable
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/**
 * Version "5.6.10a".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/**
 * Version "5.6.mysql_aurora.1.17.9".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/**
 * Version "5.6.mysql_aurora.1.19.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/**
 * Version "5.6.mysql_aurora.1.19.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/**
 * Version "5.6.mysql_aurora.1.19.2".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/**
 * Version "5.6.mysql_aurora.1.19.5".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/**
 * Version "5.6.mysql_aurora.1.19.6".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/**
 * Version "5.6.mysql_aurora.1.20.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/**
 * Version "5.6.mysql_aurora.1.20.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/**
 * Version "5.6.mysql_aurora.1.21.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/**
 * Version "5.6.mysql_aurora.1.22.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/**
 * Version "5.6.mysql_aurora.1.22.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/**
 * Version "5.6.mysql_aurora.1.22.1.3".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/**
 * Version "5.6.mysql_aurora.1.22.2".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 *
 * @stability stable
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string, for example "5.7.mysql_aurora.2.78.3.6".
     * @param auroraMysqlMajorVersion the major version of the engine, defaults to "5.7".
     * @stability stable
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/**
 * Version "5.7.12".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/**
 * Version "5.7.mysql_aurora.2.03.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/**
 * Version "5.7.mysql_aurora.2.03.3".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/**
 * Version "5.7.mysql_aurora.2.03.4".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/**
 * Version "5.7.mysql_aurora.2.04.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/**
 * Version "5.7.mysql_aurora.2.04.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/**
 * Version "5.7.mysql_aurora.2.04.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/**
 * Version "5.7.mysql_aurora.2.04.3".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/**
 * Version "5.7.mysql_aurora.2.04.4".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/**
 * Version "5.7.mysql_aurora.2.04.5".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/**
 * Version "5.7.mysql_aurora.2.04.6".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/**
 * Version "5.7.mysql_aurora.2.04.7".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/**
 * Version "5.7.mysql_aurora.2.04.8".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/**
 * Version "5.7.mysql_aurora.2.05.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/**
 * Version "5.7.mysql_aurora.2.06.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/**
 * Version "5.7.mysql_aurora.2.07.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/**
 * Version "5.7.mysql_aurora.2.07.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/**
 * Version "5.7.mysql_aurora.2.07.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/**
 * Version "5.7.mysql_aurora.2.08.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/**
 * Version "5.7.mysql_aurora.2.08.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/**
 * Version "5.7.mysql_aurora.2.08.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/**
 * Version "5.7.mysql_aurora.2.09.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 *
 * @stability stable
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string, for example "9.6.25.1".
     * @param auroraPostgresMajorVersion the major version of the engine, for example "9.6".
     * @stability stable
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/**
 * Version "9.6.8".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/**
 * Version "9.6.9".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/**
 * Version "9.6.11".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/**
 * Version "9.6.12".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/**
 * Version "9.6.16".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/**
 * Version "9.6.17".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/**
 * Version "10.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/**
 * Version "10.5".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/**
 * Version "10.6".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/**
 * Version "10.7".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * Version "10.11".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.12".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/**
 * Version "11.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * Version "11.6".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.7".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.8".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _a, _b, _c;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_a = this.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) {
            if (options.s3ImportRole && !((_b = config.features) === null || _b === void 0 ? void 0 : _b.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_c = config.features) === null || _c === void 0 ? void 0 : _c.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine.
 *
 * Provides mapping to the serverless application
 * used for secret rotation.
 *
 * @stability stable
 */
class DatabaseClusterEngine {
    /**
     * Creates a new plain Aurora database cluster engine.
     *
     * @stability stable
     */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora MySQL database cluster engine.
     *
     * @stability stable
     */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora PostgreSQL database cluster engine.
     *
     * @stability stable
     */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link aurora()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraMysql()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraPostgres()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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