import json
from http.server import HTTPServer, BaseHTTPRequestHandler
from urllib.parse import urlparse, parse_qs

try:
    from protobuf.klog_pb2 import LogGroup
    from compress import Lz4Compressor
    from const import *
except ImportError:
    from .protobuf.klog_pb2 import LogGroup
    from .compress import Lz4Compressor
    from .const import *


class KLogHandler(BaseHTTPRequestHandler):
    STAT = dict()
    MODE = dict()

    def do_GET(self):
        self._code("", "", 404)

    def do_POST(self):
        p = urlparse(self.path)
        qa = parse_qs(p.query)
        params = {k: v[0] for k, v in qa.items()}

        data = None
        content_length = int(self.headers.get('content-length', '0'))
        if content_length > 0:
            data = self.rfile.read(content_length)

        if p.path == '/PutLogs':
            p = params.get("ProjectName", "")
            l = params.get("LogPoolName", "")
            self.put_logs(p, l, data)
        elif p.path == '/_bulk':
            self.bulk(data)
        elif p.path == '/_set_mode':
            self.MODE["mode"] = params.get('mode')
        elif p.path == '/_stat':
            self.send_response(200)
            self.send_header('content-type', 'application/json')
            self.end_headers()
            self.wfile.write(json.dumps(self.STAT).encode())
        elif p.path == '/_clear_stat':
            self.send_response(200)
            self.end_headers()
            self.STAT.clear()
            self.MODE.clear()
        else:
            self._code("", "", 404)

    def put_logs(self, p, l, data):
        self._add_stat(p, l, "requests", 1)

        mode = self.MODE.get('mode')
        if mode == 'ProjectOrLogPoolNotExist':
            self._code(p, l, 400)
            self._err_code('ProjectOrLogPoolNotExist')
            return
        elif mode == 'InternalServerError':
            self._code(p, l, 500)
            return

        if self.headers.get('Authorization') == '':
            self._code(p, l, 400)
            self._err_code('MissingAuthorization')
            return
        elif self.headers.get('x-klog-compress-type') == 'lz4':
            compressor = Lz4Compressor()
            try:
                data = compressor.decompress(data)
            except Exception as e:
                print(str(e))
                self._code(p, l, 400)
                self._err_code('PostBodyUncompressError')
                return

        lg = LogGroup()

        try:
            lg.ParseFromString(data)
        except Exception as e:
            print(str(e))
            self._code(p, l, 400)
            self._err_code('InvalidProtobufData')
            return self._code(p, l, 500)

        if lg.ByteSize() > 3 << 20:
            self._code(p, l, 400)
            self._err_code('PostBodyTooLarge')
            return

        if len(lg.logs) > MAX_BULK_SIZE:
            self._code(p, l, 400)
            self._err_code('MaxBulkSizeExceeded')
            return

        for pb_log in lg.logs:
            if len(pb_log.contents) > MAX_KEY_COUNT:
                self._code(p, l, 400)
                self._err_code('MaxKeyCountExceeded')
                return

            for content in pb_log.contents:
                if len(content.key) > MAX_KEY_SIZE:
                    self._code(p, l, 400)
                    self._err_code('MaxKeySizeExceeded')
                    return

                if len(content.value) > MAX_VALUE_SIZE:
                    self._code(p, l, 400)
                    self._err_code('MaxValueSizeExceeded')
                    return

            self._add_stat(p, l, "contents", len(pb_log.contents))

        self._add_stat(p, l, "logs", len(lg.logs))
        self._code(p, l, 200)

    def bulk(self, data):
        self._add_stat('', '', "requests", 1)

        mode = self.MODE.get('mode')
        if mode == 'InternalServerError':
            self._code('', '', 500)
            return
        elif mode == 'BadJsonReturn':
            bad_json = '{"a: 1,"b": 2}'
            self._code('', '', 200)
            self.wfile.write(bad_json.encode())
            return
        elif mode == 'BadRequest':
            ret = {
                "error": {
                    "root_cause": [
                        {
                            "type": "index_not_found_exception",
                            "reason": "no such index [mock-index]",
                            "resource.type": "index_or_alias",
                            "resource.id": "mock-index",
                            "index_uuid": "_na_",
                            "index": "mock-index"
                        }
                    ],
                    "type": "index_not_found_exception",
                    "reason": "no such index [mock-index]",
                    "resource.type": "index_or_alias",
                    "resource.id": "mock-index",
                    "index_uuid": "_na_",
                    "index": "mock-index"
                },
                "status": 400
            }
            self._code('', '', 400)
            self.wfile.write(json.dumps(ret).encode())
            return

        ret = {
            'took': 50,
            'errors': False,
            'items': []
        }

        count = 0
        err_count = 0
        pre_index = ''
        for line in data.decode().split("\n"):
            if not line:
                continue
            dic = json.loads(line.encode())
            index = dic.get('index', {}).get('_index')
            if not index:
                self._add_stat('', '', 'contents', len(dic))
                self._add_stat('', pre_index, 'contents', len(dic))
            else:
                count += 1
                if mode == 'PartiallyError' and count % 2 == 0:
                    ret['errors'] = True
                    err_count += 1
                    rotate = err_count % 4
                    status = 0
                    if rotate == 0:
                        status = 0
                    elif rotate == 1:
                        status = 400
                    elif rotate == 2:
                        status = 429
                    elif rotate == 3:
                        status = 500

                    item = {
                        "_index": index,
                        "_type": "_doc",
                        "_id": "5",
                        "status": status,
                        "error": {
                            "type": "doc_status_{}".format(status),
                            "reason": "mock_doc_error_reason",
                            "index_uuid": "aAsFqTI0Tc2W0LCWgPNrOA",
                            "shard": "0",
                            "index": "index1"
                        }
                    }
                else:
                    item = {
                        "_index": index,
                        "_type": "_doc",
                        "_id": "1",
                        "_version": 1,
                        "result": "created",
                        "_shards": {
                            "total": 2,
                            "successful": 1,
                            "failed": 0
                        },
                        "status": 201,
                        "_seq_no": 0,
                        "_primary_term": 1
                    }
                    self._add_stat('', index, 'logs', 1)
                    self._add_stat('', '', 'logs', 1)
                ret['items'].append({"index": item})
                pre_index = index

        self._code('', '', 200)
        self.wfile.write(json.dumps(ret).encode())

    def _err_code(self, err_code, err_msg=''):
        dic = {
            "ErrorCode": err_code,
            "ErrorMessage": err_msg,
        }
        self.wfile.write(json.dumps(dic).encode())

    def _code(self, project_name, log_pool_name, code):
        self.send_response(code)
        self.end_headers()
        self._add_stat(project_name, log_pool_name, str(code), 1)
        if project_name or log_pool_name:
            self._add_stat('', '', str(code), 1)

    def _get_pool_stat(self, project_name, log_pool_name):
        key = '{}__{}'.format(project_name, log_pool_name)
        if key not in self.STAT:
            self.STAT[key] = dict()
        return self.STAT[key]

    def _set_stat(self, project_name, log_pool_name, name, value):
        pool_stat = self._get_pool_stat(project_name, log_pool_name)
        pool_stat[name] = value

    def _add_stat(self, project_name, log_pool_name, name, delta):
        pool_stat = self._get_pool_stat(project_name, log_pool_name)
        if name not in pool_stat:
            pool_stat[name] = delta
        else:
            pool_stat[name] += delta


def main():
    port = 8765
    srv = HTTPServer(('', port), KLogHandler)
    print('Server started on port %s' % port)
    srv.serve_forever()


if __name__ == '__main__':
    main()
