import { IAutoScalingGroup } from "@aws-cdk/aws-autoscaling";
import { IMachineImage, InstanceType, ISecurityGroup } from "@aws-cdk/aws-ec2";
import { CfnInstanceProfile, IRole } from "@aws-cdk/aws-iam";
import { IBucket } from "@aws-cdk/aws-s3";
import { Construct, Stack } from "@aws-cdk/core";
import { AutoscalingConfiguration, DockerConfiguration, GlobalConfiguration, MachineConfiguration } from "../runner-configuration";
import { CacheProps } from "./cache";
import { Network, NetworkProps } from "./network";
/**
 * Properties of the Gitlab Runner.
 *
 * You have to provide at least the GitLab's Runner's authentication token.
 *
 * @stability stable
 */
export interface GitlabRunnerAutoscalingProps extends GlobalConfiguration {
    /**
     * The GitLab Runner’s authentication token, which is obtained during runner registration.
     *
     * @see {@link https://docs.gitlab.com/ee/api/runners.html#registration-and-authentication-tokens}
     * @stability stable
     */
    readonly gitlabToken: string;
    /**
     * GitLab instance URL.
     *
     * @default "https://gitlab.com"
     * @stability stable
     */
    readonly gitlabUrl?: string;
    /**
     * @stability stable
     */
    readonly cache?: GitlabRunnerAutoscalingCacheProps;
    /**
     * The network configuration for the Runner.
     *
     * If not set, the defaults will be used.
     *
     * @stability stable
     * @link NetworkProps
     */
    readonly network?: NetworkProps;
    /**
     * The manager EC2 instance configuration.
     *
     * If not set, the defaults will be used.
     *
     * @stability stable
     * @link GitlabRunnerAutoscalingManagerProps
     */
    readonly manager?: GitlabRunnerAutoscalingManagerProps;
    /**
     * @stability stable
     */
    readonly runners?: GitlabRunnerAutoscalingRunnerProps;
}
/**
 * The distributed GitLab runner S3 cache.
 *
 * Either pass an existing bucket or override default options.
 *
 * @see {@link https://docs.gitlab.com/runner/configuration/advanced-configuration.html#the-runnerscaches3-section}
 * @stability stable
 */
export interface GitlabRunnerAutoscalingCacheProps {
    /**
     * An existing S3 bucket used as runner's cache.
     *
     * @stability stable
     */
    readonly bucket?: IBucket;
    /**
     * If no existing S3 bucket is provided, a S3 bucket will be created.
     *
     * @stability stable
     */
    readonly options?: CacheProps;
}
/**
 * @stability stable
 */
export interface GitlabRunnerAutoscalingManagerProps {
    /**
     * An Amazon Machine Image ID for the Manager EC2 instance.
     *
     * If empty the latest Amazon 2 Image will be looked up.
     *
     * @stability stable
     */
    readonly machineImage?: IMachineImage;
    /**
     * Instance type for manager EC2 instance.
     *
     * It's a combination of a class and size.
     *
     * @default InstanceType.of(InstanceClass.T3, InstanceSize.NANO)
     * @stability stable
     */
    readonly instanceType?: InstanceType;
    /**
     * A set of security credentials that you use to prove your identity when connecting to an Amazon EC2 instance.
     *
     * You won't be able to ssh into an instance without the Key Pair.
     *
     * @stability stable
     */
    readonly keyPairName?: string;
}
/**
 * The runner EC2 instances configuration.
 *
 * If not set, the defaults will be used.
 *
 * @stability stable
 * @link GitlabRunnerAutoscalingProps
 */
export interface GitlabRunnerAutoscalingRunnerProps {
    /**
     * Instance type for runner EC2 instances.
     *
     * It's a combination of a class and size.
     *
     * @default InstanceType.of(InstanceClass.T3, InstanceSize.MICRO)
     * @stability stable
     */
    readonly instanceType?: InstanceType;
    /**
     * An Amazon Machine Image ID for the Runners EC2 instances.
     *
     * If empty the latest Ubuntu 20.04 focal will be looked up.
     *
     * @see https://cloud-images.ubuntu.com/locator/ec2/
     * @stability stable
     */
    readonly machineImage?: IMachineImage;
    /**
     * Optionally pass an IAM role, that get's assigned to the EC2 runner instances.
     *
     * @stability stable
     */
    readonly role?: IRole;
    /**
     * Limit how many jobs can be handled concurrently by this registered runner.
     *
     * 0 (default) means do not limit.
     *
     * @default 10
     * @stability stable
     */
    readonly limit?: number;
    /**
     * Maximum build log size in kilobytes.
     *
     * Default is 4096 (4MB).
     *
     * @default 52428800 (50GB)
     * @stability stable
     */
    readonly outputLimit?: number;
    /**
     * Append or overwrite environment variables.
     *
     * @default ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"]
     * @stability stable
     */
    readonly environment?: string[];
    /**
     * Optional docker configuration.
     *
     * @stability stable
     */
    readonly docker?: DockerConfiguration;
    /**
     * Optional docker machine configuration.
     *
     * @stability stable
     */
    readonly machine?: MachineConfiguration;
    /**
     * Optional autoscaling configuration.
     *
     * @stability stable
     */
    readonly autoscaling?: AutoscalingConfiguration[];
}
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @stability stable
 * @example
 *
 * <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 * env: {
 * account: "000000000000",
 * region: "us-east-1",
 * }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 * gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
export declare class GitlabRunnerAutoscaling extends Construct {
    /**
     * @stability stable
     */
    readonly network: Network;
    /**
     * @stability stable
     */
    readonly cacheBucket: IBucket;
    /**
     * @stability stable
     */
    readonly runners: GitlabRunnerAutoscalingRunners;
    /**
     * @stability stable
     */
    readonly manager: GitlabRunnerAutoscalingManager;
    /**
     * @stability stable
     */
    constructor(scope: Stack, id: string, props: GitlabRunnerAutoscalingProps);
}
/**
 * @stability stable
 */
export interface GitlabRunnerAutoscalingRunners {
    /**
     * @stability stable
     */
    readonly securityGroupName: string;
    /**
     * @stability stable
     */
    readonly securityGroup: ISecurityGroup;
    /**
     * @stability stable
     */
    readonly role: IRole;
    /**
     * @stability stable
     */
    readonly instanceProfile: CfnInstanceProfile;
    /**
     * @stability stable
     */
    readonly instanceType: InstanceType;
    /**
     * @stability stable
     */
    readonly machineImage: IMachineImage;
}
/**
 * @stability stable
 */
export interface GitlabRunnerAutoscalingManager {
    /**
     * @stability stable
     */
    readonly securityGroup: ISecurityGroup;
    /**
     * @stability stable
     */
    readonly instanceType: InstanceType;
    /**
     * @stability stable
     */
    readonly machineImage: IMachineImage;
    /**
     * @stability stable
     */
    readonly autoScalingGroup: IAutoScalingGroup;
    /**
     * @stability stable
     */
    readonly role: IRole;
}
