#                                  Apache License
#                            Version 2.0, January 2004
#                         http://www.apache.org/licenses/

#    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

#    1. Definitions.

#       "License" shall mean the terms and conditions for use, reproduction,
#       and distribution as defined by Sections 1 through 9 of this document.

#       "Licensor" shall mean the copyright owner or entity authorized by
#       the copyright owner that is granting the License.

#       "Legal Entity" shall mean the union of the acting entity and all
#       other entities that control, are controlled by, or are under common
#       control with that entity. For the purposes of this definition,
#       "control" means (i) the power, direct or indirect, to cause the
#       direction or management of such entity, whether by contract or
#       otherwise, or (ii) ownership of fifty percent (50%) or more of the
#       outstanding shares, or (iii) beneficial ownership of such entity.

#       "You" (or "Your") shall mean an individual or Legal Entity
#       exercising permissions granted by this License.

#       "Source" form shall mean the preferred form for making modifications,
#       including but not limited to software source code, documentation
#       source, and configuration files.

#       "Object" form shall mean any form resulting from mechanical
#       transformation or translation of a Source form, including but
#       not limited to compiled object code, generated documentation,
#       and conversions to other media types.

#       "Work" shall mean the work of authorship, whether in Source or
#       Object form, made available under the License, as indicated by a
#       copyright notice that is included in or attached to the work
#       (an example is provided in the Appendix below).

#       "Derivative Works" shall mean any work, whether in Source or Object
#       form, that is based on (or derived from) the Work and for which the
#       editorial revisions, annotations, elaborations, or other modifications
#       represent, as a whole, an original work of authorship. For the purposes
#       of this License, Derivative Works shall not include works that remain
#       separable from, or merely link (or bind by name) to the interfaces of,
#       the Work and Derivative Works thereof.

#       "Contribution" shall mean any work of authorship, including
#       the original version of the Work and any modifications or additions
#       to that Work or Derivative Works thereof, that is intentionally
#       submitted to Licensor for inclusion in the Work by the copyright owner
#       or by an individual or Legal Entity authorized to submit on behalf of
#       the copyright owner. For the purposes of this definition, "submitted"
#       means any form of electronic, verbal, or written communication sent
#       to the Licensor or its representatives, including but not limited to
#       communication on electronic mailing lists, source code control systems,
#       and issue tracking systems that are managed by, or on behalf of, the
#       Licensor for the purpose of discussing and improving the Work, but
#       excluding communication that is conspicuously marked or otherwise
#       designated in writing by the copyright owner as "Not a Contribution."

#       "Contributor" shall mean Licensor and any individual or Legal Entity
#       on behalf of whom a Contribution has been received by Licensor and
#       subsequently incorporated within the Work.

#    2. Grant of Copyright License. Subject to the terms and conditions of
#       this License, each Contributor hereby grants to You a perpetual,
#       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
#       copyright license to reproduce, prepare Derivative Works of,
#       publicly display, publicly perform, sublicense, and distribute the
#       Work and such Derivative Works in Source or Object form.

#    3. Grant of Patent License. Subject to the terms and conditions of
#       this License, each Contributor hereby grants to You a perpetual,
#       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
#       (except as stated in this section) patent license to make, have made,
#       use, offer to sell, sell, import, and otherwise transfer the Work,
#       where such license applies only to those patent claims licensable
#       by such Contributor that are necessarily infringed by their
#       Contribution(s) alone or by combination of their Contribution(s)
#       with the Work to which such Contribution(s) was submitted. If You
#       institute patent litigation against any entity (including a
#       cross-claim or counterclaim in a lawsuit) alleging that the Work
#       or a Contribution incorporated within the Work constitutes direct
#       or contributory patent infringement, then any patent licenses
#       granted to You under this License for that Work shall terminate
#       as of the date such litigation is filed.

#    4. Redistribution. You may reproduce and distribute copies of the
#       Work or Derivative Works thereof in any medium, with or without
#       modifications, and in Source or Object form, provided that You
#       meet the following conditions:

#       (a) You must give any other recipients of the Work or
#           Derivative Works a copy of this License; and

#       (b) You must cause any modified files to carry prominent notices
#           stating that You changed the files; and

#       (c) You must retain, in the Source form of any Derivative Works
#           that You distribute, all copyright, patent, trademark, and
#           attribution notices from the Source form of the Work,
#           excluding those notices that do not pertain to any part of
#           the Derivative Works; and

#       (d) If the Work includes a "NOTICE" text file as part of its
#           distribution, then any Derivative Works that You distribute must
#           include a readable copy of the attribution notices contained
#           within such NOTICE file, excluding those notices that do not
#           pertain to any part of the Derivative Works, in at least one
#           of the following places: within a NOTICE text file distributed
#           as part of the Derivative Works; within the Source form or
#           documentation, if provided along with the Derivative Works; or,
#           within a display generated by the Derivative Works, if and
#           wherever such third-party notices normally appear. The contents
#           of the NOTICE file are for informational purposes only and
#           do not modify the License. You may add Your own attribution
#           notices within Derivative Works that You distribute, alongside
#           or as an addendum to the NOTICE text from the Work, provided
#           that such additional attribution notices cannot be construed
#           as modifying the License.

#       You may add Your own copyright statement to Your modifications and
#       may provide additional or different license terms and conditions
#       for use, reproduction, or distribution of Your modifications, or
#       for any such Derivative Works as a whole, provided Your use,
#       reproduction, and distribution of the Work otherwise complies with
#       the conditions stated in this License.

#    5. Submission of Contributions. Unless You explicitly state otherwise,
#       any Contribution intentionally submitted for inclusion in the Work
#       by You to the Licensor shall be under the terms and conditions of
#       this License, without any additional terms or conditions.
#       Notwithstanding the above, nothing herein shall supersede or modify
#       the terms of any separate license agreement you may have executed
#       with Licensor regarding such Contributions.

#    6. Trademarks. This License does not grant permission to use the trade
#       names, trademarks, service marks, or product names of the Licensor,
#       except as required for reasonable and customary use in describing the
#       origin of the Work and reproducing the content of the NOTICE file.

#    7. Disclaimer of Warranty. Unless required by applicable law or
#       agreed to in writing, Licensor provides the Work (and each
#       Contributor provides its Contributions) on an "AS IS" BASIS,
#       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
#       implied, including, without limitation, any warranties or conditions
#       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
#       PARTICULAR PURPOSE. You are solely responsible for determining the
#       appropriateness of using or redistributing the Work and assume any
#       risks associated with Your exercise of permissions under this License.

#    8. Limitation of Liability. In no event and under no legal theory,
#       whether in tort (including negligence), contract, or otherwise,
#       unless required by applicable law (such as deliberate and grossly
#       negligent acts) or agreed to in writing, shall any Contributor be
#       liable to You for damages, including any direct, indirect, special,
#       incidental, or consequential damages of any character arising as a
#       result of this License or out of the use or inability to use the
#       Work (including but not limited to damages for loss of goodwill,
#       work stoppage, computer failure or malfunction, or any and all
#       other commercial damages or losses), even if such Contributor
#       has been advised of the possibility of such damages.

#    9. Accepting Warranty or Additional Liability. While redistributing
#       the Work or Derivative Works thereof, You may choose to offer,
#       and charge a fee for, acceptance of support, warranty, indemnity,
#       or other liability obligations and/or rights consistent with this
#       License. However, in accepting such obligations, You may act only
#       on Your own behalf and on Your sole responsibility, not on behalf
#       of any other Contributor, and only if You agree to indemnify,
#       defend, and hold each Contributor harmless for any liability
#       incurred by, or claims asserted against, such Contributor by reason
#       of your accepting any such warranty or additional liability.

#    END OF TERMS AND CONDITIONS

#    APPENDIX: How to apply the Apache License to your work.

#       To apply the Apache License to your work, attach the following
#       boilerplate notice, with the fields enclosed by brackets "[]"
#       replaced with your own identifying information. (Don't include
#       the brackets!)  The text should be enclosed in the appropriate
#       comment syntax for the file format. We also recommend that a
#       file or class name and description of purpose be included on the
#       same "printed page" as the copyright notice for easier
#       identification within third-party archives.

#    Copyright [yyyy] [name of copyright owner]

#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at

#        http://www.apache.org/licenses/LICENSE-2.0

#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from typing import List, Tuple
from torch import nn
import torch.nn.functional as F
from functools import partial
from fastai.basic_train import LearnerCallback
import numpy as np
import torch
from arcgis.learn._utils.classified_tiles import calculate_intersection, calculate_union, confusion_matrix, calculate_precision, calculate_recall, calculate_f1, calculate_metrics



def expand_outputs(preds,trues):
    if preds.shape == trues.shape:
        return trues
    encoded_trues=preds.detach()*0
    encoded_trues.scatter_(1, trues, 1)
    return encoded_trues


def dice(preds, *targs, eps=1e-8, iou=False, soft=False, ignore_classes=[0], weighted=False, average ='micro', **kwargs, ):
    """
    Calculates dice coefficient over a batch.
    =====================   ===========================================
    **Argument**            **Description**
    ---------------------   -------------------------------------------
    preds                   Required torch.tensor.
                            Predictions form a segmentation model.
                            file.
    ---------------------   -------------------------------------------
    targs                   Required torch.tensor.
                            A batch of ground truth segmentation mask.
    ---------------------   -------------------------------------------
    eps                     Optional float.
                            A very small floating point number to avoid deviding
                            by zero errors.
                            Default : 1e-08
    ---------------------   -------------------------------------------
    iou                     Optional Bool.
                            False: Compute Dice coefficient.
                            True:  Compute mean IOU
    ---------------------   -------------------------------------------
    ignore_classes=[0],     Optional List.
                            A list of classes(ignore_mapped_class) on which needs to be skipped for
                            dice calculation.
    ---------------------   -------------------------------------------
    weighted=False          Optional Bool.
                            Not implemented
                            Default False.
    ---------------------   -------------------------------------------
    average='micro'         Optional str.
                            'micro': micro dice coefficient is calculate.
                            'macro': macro dice coefficient is calculate.
                            Default 'micro'.                            
    =====================   ===========================================
    :returns: Dice Coefficient->Rank 0 torch.tensor
    """
    if (
            isinstance(preds, (List, Tuple))
            and isinstance(targs, (List, Tuple))
            and len(preds) == len(targs)
    ):
        return torch.FloatTensor(
            [
                dice(pred, targ, ignore_classes=ignore_class)
                for pred, targ, ignore_class in zip(preds, targs, ignore_classes)
            ]
        )
    if isinstance(preds, (List, Tuple)):
        preds = preds[0]
    if isinstance(targs, (List, Tuple)) and len(targs) == 1:
        targs = targs[0].long()
    if len(targs.size()) == 3:
        targs = targs.unsqueeze(1)
    num = preds.size(0)
    classes = preds.size(1)
    preds = F.softmax(preds, dim=1)
    keep_classes = [v for v in range(classes) if v not in ignore_classes]
    encoded_targs = expand_outputs(preds, targs)

    if weighted:
        # need to work on a weighted metric and loss function
        pass

    if not soft:  # for metric calculation
        preds = preds.argmax(dim=1)
        preds = torch.nn.functional.one_hot(preds, classes).permute(0, 3, 1, 2)

    if average == 'macro':
        preds= preds.contiguous().float().view(num, classes, -1)[:,keep_classes,:]
        encoded_targs = encoded_targs.float().view(num, classes, -1)[:,keep_classes,:]
        intersection = calculate_intersection(preds, encoded_targs, mode='per_class')
        union = calculate_union(preds, encoded_targs, intersection, mode='per_class')
        if not iou: score = 2. * intersection / (union + intersection + eps)
        else: score = intersection / (union + eps)
        return (score.sum(1)/((union!=0).sum(1).float()+1e-08)).mean()
        
    if average == 'micro':
        preds= preds.contiguous().float().view(num, classes, -1)[:,keep_classes,:].view(num,-1)
        encoded_targs = encoded_targs.float().view(num, classes, -1)[:,keep_classes,:].view(num,-1)
        intersection = calculate_intersection(preds, encoded_targs, mode='mean')
        union = calculate_union(preds, encoded_targs, intersection, mode='mean')
        if not iou: score = 2. * intersection / (union + intersection + eps)
        else: score = intersection / (union + eps)
        score = score[union != 0.]
        mean_per_img = score.mean(dim=0)
        return mean_per_img

# Below loss functions are implementations from https://github.com/simongrest/farm-pin-crop-detection-challenge 
class FocalLoss(nn.Module):
    def __init__(self, crit, alpha=1, gamma=2):
        super().__init__()
        self.alpha = alpha
        self.gamma = gamma
        self.crit = crit
    
    def forward(self, inputs, targets, reduction='mean'):
        loss = self.crit(inputs, targets)
        pt = torch.exp(-loss)
        F_loss = self.alpha * (1-pt)**self.gamma * loss

        if reduction is None:
            return F_loss
        else:
            return torch.mean(F_loss)

class DiceLoss(nn.Module):
    def __init__(self, crit, pct, weighted_dice, dice_average):
        super().__init__()
        self.crit = crit
        self.pct = pct
        self.weighted_dice = weighted_dice
        self.dice_average = dice_average
    
    def forward(self, inputs, targets, **kwargs):
        weighted = self.weighted_dice
        if weighted:
            ignore_mapped_class = kwargs.get('ignore_mapped_class', [])
        else:
            ignore_mapped_class = kwargs.get('ignore_mapped_class', [0])

        dice_complement = self.crit(inputs, targets)
        if isinstance(inputs, tuple): # handling for aux_loss=True 
            inputs=inputs[0]
        dice_c=dice(inputs, targets, soft=True, weighted=weighted, ignore_classes=ignore_mapped_class, average=self.dice_average)
        
        dice_loss = torch.clamp(1 - dice_c,0,1)
        loss = (1-self.pct) * dice_complement + self.pct * dice_loss
        return loss

class MixUpCallback(LearnerCallback):
    "Callback that creates the mixed-up input and target."
    def __init__(self, learn, alpha=0.4, stack_x=False, stack_y=True):
        super().__init__(learn)
        self.alpha,self.stack_x,self.stack_y = alpha,stack_x,stack_y
    
    def on_train_begin(self, **kwargs):
        if self.stack_y: self.learn.loss_func = MixUpLoss(self.learn.loss_func)
        
    def on_batch_begin(self, last_input, last_target, train, **kwargs):
        "Applies mixup to `last_input` and `last_target` if `train`."
        if not train: return
        lambd = np.random.beta(self.alpha, self.alpha, last_target.size(0))
        lambd = np.concatenate([lambd[:,None], 1-lambd[:,None]], 1).max(1)
        lambd = last_input.new(lambd)

        shuffle = torch.randperm(last_target.size(0)).to(last_input.device)
        x1, y1 = last_input[shuffle], last_target[shuffle]
        if self.stack_x:
            new_input = [last_input, last_input[shuffle], lambd]
        else: 
            out_shape = [lambd.size(0)] + [1 for _ in range(len(x1.shape) - 1)]
            new_input = (last_input * lambd.view(out_shape) + x1 * (1-lambd).view(out_shape))
        if self.stack_y:
            
            new_lambd = torch.distributions.utils.broadcast_all(lambd[:,None,None,None], last_target)[0]
            
            #new_target = torch.cat([last_target[:,None].float(), y1[:,None].float(), new_lambd[:,None].float()], 1)
            new_target = torch.stack([last_target.float(), y1.float(), new_lambd.float()], 1)
        else:
            if len(last_target.shape) == 2:
                lambd = lambd.unsqueeze(1).float()
            new_target = last_target.float() * lambd + y1.float() * (1-lambd)
            
        return {'last_input': new_input, 'last_target': new_target}  
    
    def on_train_end(self, **kwargs):
        if self.stack_y: self.learn.loss_func = self.learn.loss_func.get_old()
        

class MixUpLoss(nn.Module):
    "Adapt the loss function `crit` to go with mixup."
    
    def __init__(self, crit, reduction='mean'):
        super().__init__()
        if hasattr(crit, 'reduction'): 
            self.crit = crit
            self.old_red = crit.reduction
            setattr(self.crit, 'reduction', 'none')
        else: 
            self.crit = partial(crit, reduction='none')
            self.old_crit = crit
        self.reduction = reduction
        
    def forward(self, output, target):
        if len(target.size()) >= 5:
            loss1, loss2 = self.crit(output,target[:,0].long()), self.crit(output,target[:,1].long())
            lambd = target[:,2].contiguous().view(-1)
            d = (loss1 * lambd  + loss2 * (1-lambd)).mean()
        else:  d = self.crit(output, target)
        if self.reduction == 'mean': return d.mean()
        elif self.reduction == 'sum':            return d.sum()
        return d
    
    def get_old(self):
        if hasattr(self, 'old_crit'):  return self.old_crit
        elif hasattr(self, 'old_red'): 
            setattr(self.crit, 'reduction', self.old_red)
            return self.crit
