# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['snakefmt', 'snakefmt.parser']

package_data = \
{'': ['*']}

install_requires = \
['black>=23.1.0,<24.0.0', 'click>=8.0.0,<9.0.0', 'toml>=0.10.2,<0.11.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=1.7.0,<5.0']}

entry_points = \
{'console_scripts': ['snakefmt = snakefmt.snakefmt:main']}

setup_kwargs = {
    'name': 'snakefmt',
    'version': '0.8.1',
    'description': 'The uncompromising Snakemake code formatter',
    'long_description': '# Snakefmt\n\n![GitHub Workflow Status](https://img.shields.io/github/actions/workflow/status/snakemake/snakefmt/ci.yaml?branch=master)\n[![codecov](https://codecov.io/gh/snakemake/snakefmt/branch/master/graph/badge.svg)](https://codecov.io/gh/snakemake/snakefmt)\n[![PyPI](https://img.shields.io/pypi/v/snakefmt)](https://pypi.org/project/snakefmt/)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/snakefmt)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nThis repository provides formatting for [Snakemake][snakemake] files. It follows the\ndesign and specifications of [Black][black].\n\n> **⚠️WARNING⚠️**: `snakefmt` modifies files in-place by default, thus we strongly\n> recommend ensuring your files are under version control before doing any formatting.\n> You\n> can also pipe the file in from stdin, which will print it to the screen, or use the\n> `--diff` or `--check` options. See [Usage](#usage) for more details.\n\n[TOC]: #\n\n# Table of Contents\n- [Install](#install)\n  - [PyPi](#pypi)\n  - [Conda](#conda)\n  - [Containers](#containers)\n  - [Local](#local)\n- [Example File](#example-file)\n- [Usage](#usage)\n  - [Basic Usage](#basic-usage)\n  - [Full Usage](#full-usage)\n- [Configuration](#configuration)\n- [Integration](#integration)\n    - [Editor Integration](#editor-integration)\n    - [Version Control Integration](#version-control-integration)\n    - [Github Actions](#github-actions)\n- [Plug Us](#plug-us)\n- [Changes](#changes)\n- [Contributing](#contributing)\n- [Cite](#cite)\n\n\n## Install\n\n### PyPi\n\n```sh\npip install snakefmt\n```\n\n### Conda\n\n[![Conda (channel only)](https://img.shields.io/conda/vn/bioconda/snakefmt)](https://anaconda.org/bioconda/snakefmt)\n[![bioconda version](https://anaconda.org/bioconda/snakefmt/badges/platforms.svg)](https://anaconda.org/bioconda/snakefmt)\n\n```sh\nconda install -c bioconda snakefmt\n```\n\n### Containers\n\nAs `snakefmt` has a Conda recipe, there is a matching image built for each version by\nBiocontainers.\n\nIn the following examples, all tags (`<tag>`) can be found\n[here](https://quay.io/repository/biocontainers/snakefmt?tab=tags).\n\n#### Docker\n\n```shell\n$ docker run -it "quay.io/biocontainers/snakefmt:<tag>" snakefmt --help\n```\n\n#### Singularity\n\n```shell\n$ singularity exec "docker://quay.io/biocontainers/snakefmt:<tag>" snakefmt --help\n```\n\n### Local\n\nThese instructions include [installing `poetry`](https://python-poetry.org/docs/#installation).\n```sh\n# install poetry\ncurl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python3\n\ngit clone https://github.com/snakemake/snakefmt && cd snakefmt\n# install snakefmt in a new environment\npoetry install\n# activate the environment so snakefmt is available on your PATH\npoetry shell\n```\n\n## Example File\n\nInput\n\n```python\nfrom snakemake.utils import min_version\nmin_version("5.14.0")\nconfigfile: "config.yaml" # snakemake keywords are treated like classes i.e. 2 newlines\nSAMPLES = [\'s1\', \'s2\'] # strings are normalised\nCONDITIONS = ["a", "b", "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglong"] # long lines are wrapped\ninclude: "rules/foo.smk" # 2 newlines\n\nrule all:\n    input: "data/results.txt" # newlines after keywords enforced and trailing comma\n\nrule gets_separated_by_two_newlines:\n    input:\n        files = expand("long/string/to/data/files/gets_broken_by_black/{sample}.{condition}",sample=SAMPLES, condition=CONDITIONS)\nif True:\n    rule can_be_inside_python_code:\n        input: "parameters", "get_indented"\n        threads: 4 # Numeric params stay unindented\n        params: key_val = "PEP8_formatted"\n        run:\n\n                print("weirdly_spaced_string_gets_respaced")\n\n```\n\n\nOutput\n\n```python\nfrom snakemake.utils import min_version\n\nmin_version("5.14.0")\n\n\nconfigfile: "config.yaml" # snakemake keywords are treated like classes i.e. 2 newlines\n\n\nSAMPLES = ["s1", "s2"] # strings are normalised\nCONDITIONS = [\n    "a",\n    "b",\n    "longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglong",\n]  # long lines are wrapped\n\n\ninclude: "rules/foo.smk" # 2 newlines\n\n\nrule all:\n    input:\n        "data/results.txt", # newlines after keywords enforced and trailing comma\n\n\nrule gets_separated_by_two_newlines:\n    input:\n        files=expand(\n            "long/string/to/data/files/gets_broken_by_black/{sample}.{condition}",\n            sample=SAMPLES,\n            condition=CONDITIONS,\n        ),\n\n\nif True:\n\n    rule can_be_inside_python_code:\n        input:\n            "parameters",\n            "get_indented",\n        threads: 4 # Numeric params stay unindented\n        params:\n            key_val="PEP8_formatted",\n        run:\n            print("weirdly_spaced_string_gets_respaced")\n\n```\n\n\n## Usage\n\n### Basic Usage\n\nFormat a single Snakefile.\n\n```shell\nsnakefmt Snakefile\n```\n\nFormat all Snakefiles within a directory.\n\n```shell\nsnakefmt workflows/\n```\n\nFormat a file but write the output to stdout.\n\n```shell\nsnakefmt - < Snakefile\n```\n\n### Full Usage\n\n```\n$ snakefmt --help\nUsage: snakefmt [OPTIONS] [SRC]...\n\n  The uncompromising Snakemake code formatter.\n\n  SRC specifies directories and files to format. Directories will be\n  searched for file names that conform to the include/exclude patterns\n  provided.\n\n  Files are modified in-place by default; use diff, check, or  `snakefmt - <\n  Snakefile` to avoid this.\n\nOptions:\n  -l, --line-length INT  Lines longer than INT will be wrapped.  [default: 88]\n  --check                Don\'t write the files back, just return the status.\n                         Return code 0 means nothing would change. Return code\n                         1 means some files would be reformatted. Return code\n                         123 means there was an error.\n\n  -d, --diff             Don\'t write the files back, just output a diff for\n                         each file to stdout.\n\n  --compact-diff         Same as --diff but only shows lines that would change\n                         plus a few lines of context.\n\n  --include PATTERN      A regular expression that matches files and\n                         directories that should be included on recursive\n                         searches.  An empty value means all files are\n                         included regardless of the name.  Use forward slashes\n                         for directories on all platforms (Windows, too).\n                         Exclusions are calculated first, inclusions later.\n                         [default: (\\.smk$|^Snakefile)]\n\n  --exclude PATTERN      A regular expression that matches files and\n                         directories that should be excluded on recursive\n                         searches.  An empty value means no paths are\n                         excluded. Use forward slashes for directories on all\n                         platforms (Windows, too). Exclusions are calculated\n                         first, inclusions later.  [default: (\\.snakemake|\\.eg\n                         gs|\\.git|\\.hg|\\.mypy_cache|\\.nox|\\.tox|\\.venv|\\.svn|_\n                         build|buck-out|build|dist)]\n\n  -c, --config PATH      Read configuration from PATH. By default, will try to\n                         read from `./pyproject.toml`\n\n  -h, --help             Show this message and exit.\n  -V, --version          Show the version and exit.\n  -v, --verbose          Turns on debug-level logging.\n\n```\n\n## Configuration\n\n`snakefmt` is able to read project-specific default values for its command line options\nfrom a `pyproject.toml` file. In addition, it will also load any [`black`\nconfigurations][black-config] you have in the same file.\n\nBy default, `snakefmt` will search in the parent directories of the formatted file(s)\nfor a file called `pyproject.toml` and use any configuration there.\nIf your configuration file is located somewhere else or called something different,\nspecify it using `--config`.\n\nAny options you pass on the command line will take precedence over default values in the\nconfiguration file.\n\n#### Example\n\n`pyproject.toml`\n\n```toml\n[tool.snakefmt]\nline_length = 90\ninclude = \'\\.smk$|^Snakefile|\\.py$\'\n\n# snakefmt passes these options on to black\n[tool.black]\nskip_string_normalization = true\n```\n\nIn this example we increase the `--line-length` value and also include python (`*.py`)\nfiles for formatting - this effectively runs `black` on them. `snakefmt` will also pass\non the `[tool.black]` settings, internally, to `black`.\n\n\n## Integration\n\n\n### Editor Integration\n\nFor instructions on how to integrate `snakefmt` into your editor of choice, refer to\n[`docs/editor_integration.md`](docs/editor_integration.md)\n\n### Version Control Integration\n\n`snakefmt` supports [pre-commit](https://pre-commit.com/), a framework for managing git pre-commit hooks. Using this framework you can run `snakefmt` whenever you commit a `Snakefile` or `*.smk` file. `Pre-commit` automatically creates an isolated virtual environment with `snakefmt` and will stop the commit if `snakefmt` would modify the file. You then review, stage, and re-commit these changes. Pre-commit is especially useful if you don\'t have access to a CI/CD system like GitHub actions.\n\nTo do so, create the file `.pre-commit-config.yaml` in the root of your project directory with the following:\n\n```yaml\nrepos:\n  - repo: https://github.com/snakemake/snakefmt\n    rev: 0.5.0 # Replace by any tag/version ≥0.2.4 : https://github.com/snakemake/snakefmt/releases\n    hooks:\n      - id: snakefmt\n```\n\nThen [install pre-commit](https://pre-commit.com/#installation) and initialize the pre-commit hooks by running `pre-commit install` (Note you need to run this step once per clone of your repository). Additional pre-commit hooks can be found [here](https://pre-commit.com/hooks.html).\n\n### GitHub Actions\n\n[GitHub Actions](https://github.com/features/actions) in combination with [super-linter](https://github.com/github/super-linter) allows you to automatically run `snakefmt` on all Snakefiles in your repository e.g. whenever you push a new commit.\n\nTo do so, create the file `.github/workflows/linter.yml` in your repository:\n```yaml\n---\nname: Lint Code Base\n\non:\n  push:\n  pull_request:\n    branches: [master]\n\njobs:\n  build:\n    name: Lint Code Base\n    runs-on: ubuntu-latest\n\n    steps:\n      - name: Checkout Code\n        uses: actions/checkout@v2\n\n      - name: Lint Code Base\n        uses: github/super-linter@v3\n        env:\n          VALIDATE_ALL_CODEBASE: false\n          DEFAULT_BRANCH: master\n          GITHUB_TOKEN: ${{ secrets.GITHUB_TOKEN }}\n\n          VALIDATE_SNAKEMAKE_SNAKEFMT: true\n```\n\nAdditional configuration parameters can be specified by creating `.github/linters/.snakefmt.toml`:\n```toml\n[tool.black]\nskip_string_normalization = true\n```\n\nFor more information check the `super-linter` readme.\n\n## Plug Us\n\nIf you can\'t get enough of badges, then feel free to show others you\'re using `snakefmt`\nin your project.\n\n[![Code style: snakefmt](https://img.shields.io/badge/code%20style-snakefmt-000000.svg)](https://github.com/snakemake/snakefmt)\n\n#### Markdown\n\n```md\n[![Code style: snakefmt](https://img.shields.io/badge/code%20style-snakefmt-000000.svg)](https://github.com/snakemake/snakefmt)\n```\n\n#### ReStructuredText\n\n```rst\n.. image:: https://img.shields.io/badge/code%20style-snakefmt-000000.svg\n    :target: https://github.com/snakemake/snakefmt\n```\n\n## Changes\n\nSee [`CHANGELOG.md`][changes].\n\n## Contributing\n\nSee [CONTRIBUTING.md][contributing].\n\n## Cite\n\n[![DOI][doi-shield]][doi]\n\n```Bibtex\n@article{snakemake2021,\n  doi = {10.12688/f1000research.29032.2},\n  url = {https://doi.org/10.12688/f1000research.29032.2},\n  year = {2021},\n  month = apr,\n  publisher = {F1000 Research Ltd},\n  volume = {10},\n  pages = {33},\n  author = {Felix M\\"{o}lder and Kim Philipp Jablonski and Brice Letcher and Michael B. Hall and Christopher H. Tomkins-Tinch and Vanessa Sochat and Jan Forster and Soohyun Lee and Sven O. Twardziok and Alexander Kanitz and Andreas Wilm and Manuel Holtgrewe and Sven Rahmann and Sven Nahnsen and Johannes K\\"{o}ster},\n  title = {Sustainable data analysis with Snakemake},\n  journal = {F1000Research}\n}\n```\n\n\n[snakemake]: https://snakemake.readthedocs.io/\n[black]: https://black.readthedocs.io/en/stable/\n[black-config]: https://github.com/psf/black#pyprojecttoml\n[pyproject]: https://github.com/snakemake/snakefmt/blob/master/pyproject.toml\n[poetry]: https://python-poetry.org\n[contributing]: CONTRIBUTING.md\n[changes]: CHANGELOG.md\n[doi-shield]: https://img.shields.io/badge/DOI-10.12688%2Ff1000research.29032.2-brightgreen.svg\n[doi]: https://doi.org/10.12688/f1000research.29032.2\n',
    'author': 'Michael Hall',
    'author_email': 'michael@mbh.sh',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/snakemake/snakefmt',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
