"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources.
 *
 * @stability stable
 */
class NagSuppressions {
    /**
     * Add cdk-nag suppressions to the Stack.
     *
     * @param stack the Stack to apply the suppression to.
     * @param suppressions a list of suppressions to apply to the stack.
     * @stability stable
     */
    static addStackSuppressions(stack, suppressions) {
        var _b;
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${stack.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const currentSuppressions = (_b = stack.templateOptions.metadata) === null || _b === void 0 ? void 0 : _b.cdk_nag;
        if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
            newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
        }
        if (stack.templateOptions.metadata) {
            stack.templateOptions.metadata.cdk_nag = {
                rules_to_suppress: newSuppressions,
            };
        }
        else {
            stack.templateOptions.metadata = {
                cdk_nag: { rules_to_suppress: newSuppressions },
            };
        }
    }
    /**
     * Add cdk-nag suppressions to the construct if it is a CfnResource.
     *
     * @param construct the IConstruct to apply the suppression to.
     * @param suppressions a list of suppressions to apply to the resource.
     * @param applyToChildren apply the suppressions to this construct and all of its children if they exist (default:false).
     * @stability stable
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${construct.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const constructs = applyToChildren ? construct.node.findAll() : [construct];
        for (const child of constructs) {
            if (child.node.defaultChild instanceof monocdk_1.CfnResource) {
                const resource = child.node.defaultChild;
                const currentSuppressions = resource.getMetadata('cdk_nag');
                if (Array.isArray(currentSuppressions === null || currentSuppressions === void 0 ? void 0 : currentSuppressions.rules_to_suppress)) {
                    newSuppressions.unshift(...currentSuppressions.rules_to_suppress);
                }
                resource.addMetadata('cdk_nag', {
                    rules_to_suppress: newSuppressions,
                });
            }
        }
    }
    /**
     * Locate a construct by it's path and add cdk-nag suppressions if it both exists and is a CfnResource.
     *
     * @param stack the Stack the construct belongs to.
     * @param path the path of the construct in the provided stack.
     * @param suppressions a list of suppressions to apply to the resource.
     * @param applyToChildren apply the suppressions to this construct and all of its children if they exist (default:false).
     * @stability stable
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        for (const child of stack.node.findAll()) {
            const fixedPath = path.replace(/^\//, '');
            if (child.node.path === fixedPath ||
                child.node.path + '/Resource' === fixedPath) {
                NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
            }
        }
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "monocdk-nag.NagSuppressions", version: "0.1.13" };
//# sourceMappingURL=data:application/json;base64,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