"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveIfPrimitive = exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 *
 * @stability stable
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel[NagMessageLevel["WARN"] = 0] = "WARN";
    NagMessageLevel[NagMessageLevel["ERROR"] = 1] = "ERROR";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule sets.
 *
 * @stability stable
 */
class NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
    }
    /**
     * Create a rule to be used in the NagPack.
     *
     * @param params The.
     * @stability stable
     * @IApplyRule interface with rule details
     */
    applyRule(params) {
        var _b, _c, _d;
        let resourceIgnores = (_b = params.node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
        resourceIgnores = resourceIgnores ? resourceIgnores : [];
        let stackIgnores = (_d = (_c = monocdk_1.Stack.of(params.node).templateOptions.metadata) === null || _c === void 0 ? void 0 : _c.cdk_nag) === null || _d === void 0 ? void 0 : _d.rules_to_suppress;
        stackIgnores = stackIgnores ? stackIgnores : [];
        const allIgnores = resourceIgnores.concat(stackIgnores);
        try {
            if (!params.rule(params.node)) {
                const reason = this.ignoreRule(allIgnores, params.ruleId);
                if (reason) {
                    if (this.logIgnores === true) {
                        const message = this.createMessage(SUPPRESSION_ID, `${params.ruleId} was triggered but suppressed.`, `Provided reason: "${reason}"`);
                        monocdk_1.Annotations.of(params.node).addInfo(message);
                    }
                }
                else {
                    const message = this.createMessage(params.ruleId, params.info, params.explanation);
                    if (params.level == NagMessageLevel.ERROR) {
                        monocdk_1.Annotations.of(params.node).addError(message);
                    }
                    else if (params.level == NagMessageLevel.WARN) {
                        monocdk_1.Annotations.of(params.node).addWarning(message);
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID);
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${params.ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored
     * @param ignores ignores listed in cdkNag metadata
     * @param ruleId the id of the rule to ignore
     * @returns the reason the rule was ignored, or an empty string
     */
    ignoreRule(ignores, ruleId) {
        for (let ignore of ignores) {
            if (ignore.id &&
                ignore.reason &&
                JSON.stringify(ignore.reason).length >= 10) {
                if (ignore.id == ruleId) {
                    return ignore.reason;
                }
            }
            else {
                throw Error(`Improperly formatted cdk_nag rule suppression detected: ${JSON.stringify(ignore)}. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered
     * @param ruleId the id of the rule
     * @param info why the rule was triggered
     * @param explanation why the rule exists
     * @returns string
     */
    createMessage(ruleId, info, explanation) {
        let message = `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}` : message;
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "0.1.13" };
/**
 * Use in cases where a primitive value must be known to pass a rule
 * https://developer.mozilla.org/en-US/docs/Glossary/Primitive
 * @param node the CfnResource to check
 * @param parameter the value to attempt to resolve
 * @returns Return a value if resolves to a primitive data type, otherwise throw an error.
 */
function resolveIfPrimitive(node, parameter) {
    const resolvedValue = monocdk_1.Stack.of(node).resolve(parameter);
    if (resolvedValue === Object(resolvedValue)) {
        throw Error(`The parameter resolved to to a non-primitive value "${JSON.stringify(resolvedValue)}", therefore the rule could not be validated.`);
    }
    else {
        return resolvedValue;
    }
}
exports.resolveIfPrimitive = resolveIfPrimitive;
//# sourceMappingURL=data:application/json;base64,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