"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PCIDSS321Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const dms_1 = require("./rules/dms");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const secretsmanager_1 = require("./rules/secretsmanager");
const sns_1 = require("./rules/sns");
const vpc_1 = require("./rules/vpc");
/**
 * Check for PCI DSS 3.2.1 compliance. Based on the PCI DSS 3.2.1 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-pci-dss.html.
 *
 * @stability stable
 */
class PCIDSS321Checks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkDMS(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWCacheEnabledAndEncrypted',
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control ID: 3.4).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWExecutionLoggingEnabled',
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-APIGWSSLEnabled',
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.pciDss321APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-AutoscalingGroupELBHealthCheckRequired',
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks - (Control ID: 2.2).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.pciDss321AutoscalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-AutoscalingLaunchConfigPublicIpDisabled',
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.pciDss321AutoscalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailCloudWatchLogsEnabled',
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.3, 10.5.4).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailEncryptionEnabled',
            info: 'The trail does not have encryption enabled - (Control IDs: 2.2, 3.4, 10.5).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudTrailLogFileValidationEnabled',
            info: 'The trail does not have log file validation enabled - (Control IDs: 2.2, 10.5.2, 10.5, 10.5.5, 11.5).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.pciDss321CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudWatchLogGroupEncrypted',
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control ID: 3.4).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.pciDss321CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-CloudWatchLogGroupRetentionPeriod',
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: 3.1, 10.7).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.pciDss321CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-DMSReplicationNotPublic',
            info: 'The DMS replication instance is public - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.pciDss321DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstanceNoPublicIp',
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstanceProfileAttached',
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: 2.2, 7.1.1, 7.2.1).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2InstancesInVPC',
            info: 'The EC2 instance is not within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2RestrictedCommonPorts',
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-EC2RestrictedSSH',
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.pciDss321EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-ECSTaskDefinitionUserForHostMode',
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control ID: 7.2.1).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.pciDss321ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EFSEncrypted',
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.pciDss321EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-ALBHttpToHttpsRedirection',
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 2.3, 4.1, 8.2.1).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBACMCertificateRequired',
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 4.1, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBLoggingEnabled',
            info: 'The ELB does not have logging enabled - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBTlsHttpsListenersOnly',
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-ELBv2ACMCertificateRequired',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.pciDss321ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-EMRKerberosEnabled',
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 7.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.pciDss321EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMNoInlinePolicy',
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMPolicyNoStatementsWithAdminAccess',
            info: 'The IAM policy grants admin access - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMPolicyNoStatementsWithFullAccess',
            info: 'The IAM policy grants full access - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMUserGroupMembership',
            info: 'The IAM user does not belong to any group(s) - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-IAMUserNoPolicies',
            info: 'The IAM policy is attached at the user level - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.pciDss321IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-LambdaInsideVPC',
            info: 'The Lambda function is not VPC enabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 2.2.2).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.pciDss321LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchEncryptedAtRest',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchInVPCOnly',
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchLogsToCloudWatch',
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-OpenSearchNodeToNodeEncryption',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.pciDss321OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSAutomaticMinorVersionUpgradeEnabled',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control ID: 6.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSAutomaticMinorVersionUpgradeEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSInstancePublicAccess',
            info: 'The RDS DB Instance allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSLoggingEnabled',
            info: 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RDSStorageEncrypted',
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.pciDss321RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterConfiguration',
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 3.4, 8.2.1, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterMaintenanceSettings',
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control ID: 6.2).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                              ',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftClusterPublicAccess',
            info: 'The Redshift cluster allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-RedshiftEnhancedVPCRoutingEnabled',
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 1.2, 1.3, 1.3.1, 1.3.2).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.pciDss321RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketLevelPublicAccessProhibited',
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketLoggingEnabled',
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketPublicReadProhibited',
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketPublicWriteProhibited',
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketReplicationEnabled',
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: 2.2, 10.5.3).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketServerSideEncryptionEnabled',
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 2.2, 3.4, 8.2.1, 10.5).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3BucketVersioningEnabled',
            info: 'The S3 Bucket does not have versioning enabled - (Control ID: 10.5.3).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-S3DefaultEncryptionKMS',
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 3.4, 8.2.1, 10.5).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.pciDss321S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerEndpointConfigurationKMSKeyConfigured',
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerNotebookInstanceKMSKeyConfigured',
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-SageMakerNotebookNoDirectInternetAccess',
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.pciDss321SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SecretsManagerUsingKMSKey',
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.pciDss321SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-SNSEncryptedKMS',
            info: 'The SNS topic does not have KMS encryption enabled - (Control ID: 8.2.1).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.pciDss321SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCDefaultSecurityGroupClosed',
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.1, 2.2, 2.2.2).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.pciDss321VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCNoUnrestrictedRouteToIGW',
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.pciDss321VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            ruleId: 'PCI.DSS.321-VPCSubnetAutoAssignPublicIpDisabled',
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.pciDss321VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
}
exports.PCIDSS321Checks = PCIDSS321Checks;
_a = JSII_RTTI_SYMBOL_1;
PCIDSS321Checks[_a] = { fqn: "monocdk-nag.PCIDSS321Checks", version: "0.1.13" };
//# sourceMappingURL=data:application/json;base64,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