"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
const monocdk_1 = require("monocdk");
const aws_ec2_1 = require("monocdk/aws-ec2");
const nag_pack_1 = require("../../../nag-pack");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-2a, CM-2(2), CM-6a, CM-7b, CM-8(6), CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28))
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
function default_1(node) {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}
exports.default = default_1;
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_pack_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_pack_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_pack_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_pack_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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