"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const analytics_1 = require("./rules/analytics");
const application_integration_1 = require("./rules/application_integration");
const index_1 = require("./rules/compute/index");
const index_2 = require("./rules/databases/index");
const developer_tools_1 = require("./rules/developer_tools");
const index_3 = require("./rules/machine_learning/index");
const management_and_governance_1 = require("./rules/management_and_governance");
const media_services_1 = require("./rules/media_services");
const index_4 = require("./rules/network_and_delivery/index");
const security_and_compliance_1 = require("./rules/security_and_compliance");
const serverless_1 = require("./rules/serverless");
const index_5 = require("./rules/storage/index");
/**
 * Check Best practices based on AWS Solutions Security Matrix.
 *
 * @stability stable
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc23,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc26,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc27,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc28,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEc29,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcr1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsEcs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of The CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2a',
            info: 'The ALB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb2a,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB2e',
            info: 'The CLB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb2e,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_1.awsSolutionsElb5,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsS3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_5.awsSolutionsEfs1,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds10,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds11,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds13,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds14,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRds16,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: index_2.awsSolutionsDdb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDdb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsAec6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsN5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS1',
            info: 'The Redshift cluster parameter group must have the "require_ssl" parameter enabled.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs9,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsRs10,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_2.awsSolutionsDoc5,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsVpc3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsCfr1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: index_4.awsSolutionsCfr2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsCfr6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_4.awsSolutionsApig6,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: management_and_governance_1.awsSolutionsAs3,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: index_3.awsSolutionsSm3,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsAth1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsEmr6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: analytics_1.awsSolutionsKda3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKds1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsKdf1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsMsk6,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsOs9,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: analytics_1.awsSolutionsQs1,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsIam5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: security_and_compliance_1.awsSolutionsCog2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsCog7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-KMS5',
            info: 'The KMS Symmetric key does not have Key Rotation enabled.',
            explanation: 'KMS Key Rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: security_and_compliance_1.awsSolutionsKms5,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: serverless_1.awsSolutionsSf2,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSns2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs2,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: application_integration_1.awsSolutionsSqs3,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: media_services_1.awsSolutionsMs1,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs7,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs8,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: media_services_1.awsSolutionsMs10,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleId: 'AwsSolutions-CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb3,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsCb4,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: developer_tools_1.awsSolutionsCb5,
            node: node,
        });
        this.applyRule({
            ruleId: 'AwsSolutions-C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: developer_tools_1.awsSolutionsC91,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "0.1.13" };
//# sourceMappingURL=data:application/json;base64,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