<!--
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
-->

# Rules

A list of all the available rules by Rule Pack

## AWS Solutions

The [AWS Solutions Library](https://aws.amazon.com/solutions/) offers a collection of cloud-based solutions for dozens of technical and business problems, vetted for you by AWS. The AwsSolutions Rules Pack contains many of the checks that the are used as part of the vetting process.

### Warnings

| Rule ID           | Cause                                                                                                                                                  | Explanation                                                                                                                                                                                                                                                                                                          |
| ----------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------ | -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| AwsSolutions-CB3  | The CodeBuild project has privileged mode enabled.                                                                                                     | Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.                                                   |
| AwsSolutions-CB5  | The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image. | Explaining differences/edits to Docker images helps operators better understand system dependencies.                                                                                                                                                                                                                 |
| AwsSolutions-CFR1 | The CloudFront distribution may require Geo restrictions.                                                                                              | Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.                                                                                                                                    |
| AwsSolutions-CFR2 | The CloudFront distribution may require integration with AWS WAF.                                                                                      | The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.                                                                                                                                                    |
| AwsSolutions-COG2 | The Cognito user pool does not require MFA.                                                                                                            | Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.                                                                                                                                                  |
| AwsSolutions-DDB3 | The DynamoDB table does not have Point-in-time Recovery enabled.                                                                                       | DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.                |
| AwsSolutions-KDA3 | The Kinesis Data Analytics Flink Application does not have checkpointing enabled.                                                                      | Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.                                                                                                                                                                                            |
| AwsSolutions-MS4  | The MediaStore container does not define a metric policy to send metrics to CloudWatch.                                                                | Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.                                                                                                                                                                                      |
| AwsSolutions-MS7  | The MediaStore container does not define a container policy.                                                                                           | Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.                                                                                                                                   |
| AwsSolutions-MS8  | The MediaStore container does not define a CORS policy.                                                                                                | Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.                                                                                                                                        |
| AwsSolutions-MS10 | The MediaStore container does not define a lifecycle policy.                                                                                           | Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.                                                                                                                                                                                                      |
| AwsSolutions-VPC3 | A Network ACL or Network ACL entry has been implemented.                                                                                               | Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage. |

### Errors

| Rule ID            | Cause                                                                                                                                                                                     | Explanation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| ------------------ | ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| AwsSolutions-AEC1  | The ElastiCache cluster is not provisioned in a VPC.                                                                                                                                      | Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-AEC3  | The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.                                                                                               | Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-AEC4  | The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.                                                                                                                | The cluster should use a Multi-AZ deployment configuration for high availability.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| AwsSolutions-AEC5  | The ElastiCache cluster uses the default endpoint port.                                                                                                                                   | Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-AEC6  | The ElastiCache Redis cluster does not use Redis AUTH for user authentication.                                                                                                            | Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.                                                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-APIG1 | The API does not have access logging enabled.                                                                                                                                             | Enabling access logs helps operators view who accessed an API and how the caller accessed the API.                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |
| AwsSolutions-APIG4 | The API does not implement authorization.                                                                                                                                                 | In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-APIG6 | The REST API Stage does not have CloudWatch logging enabled for all methods.                                                                                                              | Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-AS1   | The Auto Scaling Group does not have a cooldown period.                                                                                                                                   | A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.                                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-AS2   | The Auto Scaling Group does not have properly configured health checks.                                                                                                                   | The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-AS3   | The Auto Scaling Group does not have notifications configured for all scaling events.                                                                                                     | Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-ATH1  | The Athena workgroup does not encrypt query results.                                                                                                                                      | Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-CB4   | The CodeBuild project does not use an AWS KMS key for encryption.                                                                                                                         | Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-C91   | The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.                                                                                                      | SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.                                                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-CFR3  | The CloudFront distributions does not have access logging enabled.                                                                                                                        | Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| AwsSolutions-CFR5  | The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.                                                                                                         | Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.                                                                                                                                                                                                                                                                                                                                  |
| AwsSolutions-CFR6  | The CloudFront distribution does not use an origin access identity an S3 origin.                                                                                                          | Origin access identities help with security by restricting any direct access to objects through S3 URLs.                                                                                                                                                                                                                                                                                                                                                                                                                                                               |
| AwsSolutions-COG1  | The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters. | Strong password policies increase system security by encouraging users to create reliable and secure passwords.                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-COG3  | The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.                                                                                                                 | Advanced security features enable the system to detect and act upon malicious sign-in attempts.                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-COG7  | The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.                                                                   | In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-DDB4  | The DAX cluster does not have server-side encryption enabled.                                                                                                                             | Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.                                                                                                                                                                                                                                                                                                                                                                                               |
| AwsSolutions-DOC1  | The Document DB cluster does not have encryption at rest enabled.                                                                                                                         | Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-DOC2  | The Document DB cluster uses the default endpoint port.                                                                                                                                   | Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).                                                                                                                                                                                                                                                                                                                                                                                                                       |
| AwsSolutions-DOC3  | The Document DB cluster does not have the username and password stored in Secrets Manager.                                                                                                | Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise. |
| AwsSolutions-DOC4  | The Document DB cluster does not have a reasonable minimum backup retention period configured.                                                                                            | The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-DOC5  | The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.                                                                                  | This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.                                                                                                                                                                                                                                                                     |
| AwsSolutions-EC23  | The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.                                                                                                                           | Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.                                                                                                     |
| AwsSolutions-EC26  | The EBS volume has encryption disabled.                                                                                                                                                   | With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.                                                                                                                                                                                                                                                                                                                                           |
| AwsSolutions-EC27  | The Security Group does not have a description.                                                                                                                                           | Descriptions help simplify operations and remove any opportunities for operator errors.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-EC28  | The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.                                                                                              | Monitoring data helps make better decisions on architecting and managing compute resources.                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-EC29  | The EC2 instance is not part of an ASG and has Termination Protection disabled.                                                                                                           | Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| AwsSolutions-ECR1  | The ECR Repository allows open access.                                                                                                                                                    | Removing \* principals in an ECR Repository helps protect against unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-ECS4  | The ECS Cluster has CloudWatch Container Insights disabled.                                                                                                                               | CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.                                                                                                                                                                                                                                                                                                                                                                                                                         |
| AwsSolutions-ECS7  | The ECS Task Definition does not have awslogs logging enabled at the minimum.                                                                                                             | Container logging allows operators to view and aggregate the logs from the container.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| AwsSolutions-EFS1  | The EFS is not configured for encryption at rest.                                                                                                                                         | By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.                                                                                                                                                                                                                       |
| AwsSolutions-ELB1  | The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.                                                                                                                        | HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-ELB2a | The ALB does not have access logs enabled.                                                                                                                                                | Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-ELB2e | The CLB does not have access logs enabled.                                                                                                                                                | Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-ELB3  | The CLB does not have connection draining enabled.                                                                                                                                        | With Connection Draining feature enabled, if an EC2 backend instance fails health checks the CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.                                                                                                                                                                                                                                                                                          |
| AwsSolutions-ELB4  | The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.                                                                                                 | CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-ELB5  | The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.                                                                                          | The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| AwsSolutions-EMR2  | The EMR cluster does not have S3 logging enabled.                                                                                                                                         | Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.                                                                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-EMR6  | The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.                                                                                                        | SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.                                                                                                                                                                                                                                                                                                                                      |
| AwsSolutions-IAM4  | The IAM user, role, or group uses AWS managed policies.                                                                                                                                   | An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-IAM5  | The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.                                                             | Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-KDF1  | The Kinesis Data Firehose delivery stream does have server-side encryption enabled.                                                                                                       | This allows the system to meet strict regulatory requirements and enhance the security of system data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-KDS1  | The Kinesis Data Stream does not has server-side encryption enabled.                                                                                                                      | Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.                                                                                                                                                                                                                                                                                                                                     |
| AwsSolutions-KMS5  | The KMS Symmetric key does not have Key Rotation enabled.                                                                                                                                 | KMS Key Rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.                                                                                                                                                                                                                                                                                                                     |
| AwsSolutions-MS1   | The MediaStore container does not have container access logging enabled.                                                                                                                  | The container should have access logging enabled to provide detailed records for the requests that are made to the container.                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| AwsSolutions-MSK2  | The MSK cluster uses plaintext communication between clients and brokers.                                                                                                                 | TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.                                                                                                                                                                                                                                                                                                                                                                                                                                                           |
| AwsSolutions-MSK3  | The MSK cluster uses plaintext communication between brokers.                                                                                                                             | TLS communication secures data-in-transit by encrypting the connection between brokers.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-MSK6  | The MSK cluster does not send broker logs to a supported destination.                                                                                                                     | Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.                                                                                                                                                                                                                                                                                               |
| AwsSolutions-N1    | The Neptune DB cluster is not deployed in a Multi-AZ configuration.                                                                                                                       | The cluster should use a Multi-AZ deployment configuration for high availability.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| AwsSolutions-N2    | The Neptune DB instance does have Auto Minor Version Upgrade enabled.                                                                                                                     | The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.                                                                                                                                                                                                                                                                                                                                                                                               |
| AwsSolutions-N3    | The Neptune DB cluster does not have a reasonable minimum backup retention period configured.                                                                                             | The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-N4    | The Neptune DB cluster does not have encryption at rest enabled.                                                                                                                          | Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-N5    | The Neptune DB cluster does not have IAM Database Authentication enabled.                                                                                                                 | With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| AwsSolutions-OS1   | The OpenSearch Service domain is not provisioned inside a VPC.                                                                                                                            | Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.                                                                                                                                                                                                                                                                                                           |
| AwsSolutions-OS2   | The OpenSearch Service domain does not have node-to-node encryption enabled.                                                                                                              | Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.                                                                                                                                                                                                                                                                                                                                                                |
| AwsSolutions-OS3   | The OpenSearch Service domain does not only grant access via allowlisted IP addresses.                                                                                                    | Using allowlisted IP addresses helps protect the domain against unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-OS4   | The OpenSearch Service domain does not use dedicated master nodes.                                                                                                                        | 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-OS5   | The OpenSearch Service domain does not allow for unsigned requests or anonymous access.                                                                                                   | Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-OS7   | The OpenSearch Service domain does not have Zone Awareness enabled.                                                                                                                       | Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.                                                                                                                                                                                                                                                                                  |
| AwsSolutions-OS8   | The OpenSearch Service domain does not have encryption at rest enabled.                                                                                                                   | Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-OS9   | The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.                                                                         | These logs enable operators to gain full insight into the performance of these operations.                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
| AwsSolutions-QS1   | The Quicksight data sources connection is not configured to use SSL.                                                                                                                      | SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.                                                                                                                                                                                                                                                                                                                                                                   |
| AwsSolutions-RDS2  | The RDS instance or Aurora DB cluster does not have storage encryption enabled.                                                                                                           | Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.                                                                                                                                                                                                                                                                                                                                                                                                                  |
| AwsSolutions-RDS6  | The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.                                                                                               | With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.                                                                                                                                                                                                                                                                                                                                                                               |
| AwsSolutions-RDS10 | The RDS instance or Aurora DB cluster does not have deletion protection enabled.                                                                                                          | The deletion protection feature helps protect the database from being accidentally deleted.                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-RDS11 | The RDS instance or Aurora DB cluster uses the default endpoint port.                                                                                                                     | Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).                                                                                                                                                                                                                                                                                                                                                                  |
| AwsSolutions-RDS13 | The RDS instance is not configured for automated backups.                                                                                                                                 | Automated backups allow for point-in-time recovery.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| AwsSolutions-RDS14 | The RDS Aurora MySQL cluster does not have Backtrack enabled.                                                                                                                             | The RDS Aurora MySQL cluster does not have Backtrack enabled.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| AwsSolutions-RDS16 | The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.                                                                           | This allows operators to use CloudWatch to view logs to help diagnose problems in the database.                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-RS1   | The Redshift cluster parameter group must have the "require_ssl" parameter enabled.                                                                                                       | Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.                                                                                                                                                                                                                                                                                                                                                                                                                               |
| AwsSolutions-RS2   | The Redshift cluster is not provisioned in a VPC.                                                                                                                                         | Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.                                                                                                                                                                                                                                                                                                                                                                                                           |
| AwsSolutions-RS3   | The Redshift cluster uses the default "awsuser" username.                                                                                                                                 | Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.                                                                                                                                                                                                                                                                                                                                                                                                         |
| AwsSolutions-RS4   | The Redshift cluster uses the default endpoint port.                                                                                                                                      | Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).                                                                                                                                                                                                                                                                                                                                                                                                                       |
| AwsSolutions-RS5   | The Redshift cluster does not have audit logging enabled.                                                                                                                                 | Audit logging helps operators troubleshoot issues and ensure security.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-RS6   | The Redshift cluster does not have encryption at rest enabled.                                                                                                                            | Encrypting data-at-rest protects data confidentiality.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-RS8   | The Redshift cluster is publicly accessible.                                                                                                                                              | Disabling public accessibility helps minimize security risks.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| AwsSolutions-RS9   | The Redshift cluster does not have version upgrade enabled.                                                                                                                               | Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.                                                                                                                                                                                                                                                                                                                                                                                                                                                    |
| AwsSolutions-RS10  | The Redshift cluster does not have a retention period for automated snapshots configured.                                                                                                 | The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-S1    | The S3 Bucket has server access logs disabled.                                                                                                                                            | The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| AwsSolutions-S2    | The S3 Bucket does not have public access restricted and blocked.                                                                                                                         | The bucket should have public access restricted and blocked to prevent unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| AwsSolutions-S3    | The S3 Bucket does not default encryption enabled.                                                                                                                                        | The bucket should minimally have SSE enabled to help protect data-at-rest.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             |
| AwsSolutions-SF1   | The Step Function does not log "ALL" events to CloudWatch Logs.                                                                                                                           | Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| AwsSolutions-SF2   | The Step Function does not have X-Ray tracing enabled.                                                                                                                                    | X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.                                                                                                                                                                                                                                                                                                                                                        |
| AwsSolutions-SM1   | The SageMaker notebook instance is not provisioned inside a VPC.                                                                                                                          | Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.                                                                                                                                                                                                                                                                                                                                                                                                                                           |
| AwsSolutions-SM2   | The SageMaker notebook instance does not have an encrypted storage volume.                                                                                                                | Encrypting storage volumes helps protect SageMaker data-at-rest.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| AwsSolutions-SM3   | The SageMaker notebook instance has direct internet access enabled.                                                                                                                       | Disabling public accessibility helps minimize security risks.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| AwsSolutions-SNS2  | The SNS Topic does not have server-side encryption enabled.                                                                                                                               | Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-SQS2  | The SQS Queue does not have server-side encryption enabled.                                                                                                                               | Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| AwsSolutions-SQS3  | The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.                                                                  | Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.                                                                                                                                                                                                                                                                                                                                                                                                                                  |

## HIPAA Security

The [Operational Best Practices for HIPAA Security](https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html) Conformance Pack offers a collection of AWS Config Rules that enable customers to align to a subset of the HIPAA Security standards. The HIPAA Security Rules Pack implements many of the checks within the conformance pack.

### Warnings

| Rule ID                                                                                                                                         | Cause                                                                | Explanation                                                                                                                                                                                                                                                                                                                          | Relevant Control ID(s) |
| ----------------------------------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ | ---------------------- |
| [HIPAA.Security-VPCDefaultSecurityGroupClosed](https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html) | The VPC's default security group allows inbound or outbound traffic. | When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources. | 164.312(e)(1)          |

### Errors

| Rule ID                                                                                                                                                                            | Cause                                                                                                                                                                             | Explanation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Relevant Control ID(s)                                                                                                                                          |
| ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| [HIPAA.Security-ALBHttpDropInvalidHeaderEnabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html)                               | The ALB does not have invalid HTTP header dropping enabled.                                                                                                                       | Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                             | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-ALBHttpToHttpsRedirection](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html)                                      | The ALB's HTTP listeners are not configured to redirect to HTTPS.                                                                                                                 | To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                            | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-APIGWCacheEnabledAndEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html)                                   | The API Gateway stage does not have caching enabled and encrypted for all methods.                                                                                                | To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                               | 164.312(a)(2)(iv), 164.312(e)(2)(ii))                                                                                                                           |
| [HIPAA.Security-APIGWExecutionLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html)                                      | The API Gateway stage does not have execution logging enabled for all methods.                                                                                                    | API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.                                                                                                                                                                                                                                                                                                                                                                                                             | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-APIGWSSLEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-ssl-enabled.html)                                                                 | The API Gateway REST API stage is not configured with SSL certificates.                                                                                                           | Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.                                                                                                                                                                                                                                                                                                                                                                                                            | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-APIGWXrayEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-xray-enabled.html)                                                               | The API Gateway REST API stage does not have X-Ray enabled.                                                                                                                       | AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.                                                                                                                                  | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-AutoscalingGroupELBHealthCheckRequired](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html)                  | The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured.                                                                                 | The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.                                                                                                                                                             | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-config-public-ip-disabled.html)               | The Auto Scaling launch configuration does not have public IP addresses disabled.                                                                                                 | If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.                                                                                                                                                                                                                                                                                          | 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                |
| [HIPAA.Security-CloudTrailCloudWatchLogsEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html)                               | The trail does not have CloudWatch logs enabled.                                                                                                                                  | Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.                                                                                                                                                                                                                                                                                                                                                                                                    | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [HIPAA.Security-CloudTrailEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html)                                         | The trail does not have encryption enabled.                                                                                                                                       | Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.                                                                                                                                                                                                                                                                                                                                                                                                                                              | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-CloudTrailLogFileValidationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html)                         | The trail does not have log file validation enabled.                                                                                                                              | Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.                                                                                                                              | 164.312(c)(1), 164.312(c)(2)                                                                                                                                    |
| [HIPAA.Security-CloudWatchAlarmAction](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html)                                                | The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled.                                                             | Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.                                                                                                                                                                                                                                                                                                               | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-CloudWatchLogGroupEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html)                                         | The CloudWatch Log Group is not encrypted with an AWS KMS key.                                                                                                                    | To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-CloudWatchLogGroupRetentionPeriod](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                               | The CloudWatch Log Group does not have an explicit retention period configured.                                                                                                   | Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.                                                                                                                                                                                                                                                                                                                | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-CodeBuildProjectEnvVarAwsCred](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html)                               | The Codebuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.                               | Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                                                                 | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                                                                     |
| [HIPAA.Security-CodeBuildProjectSourceRepoUrl](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html)                              | The Codebuild project which utilizes either a GitHub or BitBucket repository does not utilize OAUTH.                                                                              | OAUTH is the most secure method of authenticating your Codebuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.                                                                                                                                                                                                                                                                                                                                                 | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                                                                     |
| [HIPAA.Security-DMSReplicationNotPublic](https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html)                                                 | The DMS replication instance is public.                                                                                                                                           | DMS replication instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-DynamoDBPITREnabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html)                                                          | The DynamoDB table does not have Point-in-time Recovery enabled.                                                                                                                  | The recovery maintains continuous backups of your table for the last 35 days.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [HIPAA.Security-EC2EBSOptimizedInstance](https://docs.aws.amazon.com/config/latest/developerguide/ebs-optimized-instance.html)                                                     | The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled.                                                                                     | An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.                                                                                                                                                                                                                                                      | 164.308(a)(7)(i))                                                                                                                                               |
| [HIPAA.Security-EC2InstanceDetailedMonitoringEnabled](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-detailed-monitoring-enabled.html)                      | The EC2 instance does not have detailed monitoring enabled.                                                                                                                       | Enable this rule to help improve Amazon Elastic Compute Cloud (Amazon EC2) instance monitoring on the Amazon EC2 console, which displays monitoring graphs with a 1-minute period for the instance.                                                                                                                                                                                                                                                                                                                                                                          | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-EC2InstanceNoPublicIps](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html)                                                   | The EC2 instance is associated with a public IP address.                                                                                                                          | Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                         | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-EC2InstanceProfileAttached](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-profile-attached.html)                                           | The EC2 instance does not have an instance profile attached.                                                                                                                      | EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.                                                                                                                                                                                                                                                                                                                                                                                                       | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                         |
| [HIPAA.Security-EC2InstancesInVPC](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html)                                                             | The EC2 instance is not within a VPC.                                                                                                                                             | Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access. | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-EC2RestrictedCommonPorts](https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html)                                                   | The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).                        | Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.                                                                                                                                                                                                                                                                                                                            | 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)        |
| [HIPAA.Security-EC2RestrictedSSH](https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html)                                                                    | The Security Group allows unrestricted SSH access.                                                                                                                                | Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.                                                                                                                                                                                                                                                                                                                                                                                                                                                | 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)        |
| [HIPAA.Security-ECSTaskDefinitionUserForHostMode](https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-user-for-host-mode-check.html)                      | The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty. | If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.                                                                                                                                                                                                                                                                                         | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                                               |
| [HIPAA.Security-EFSEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html)                                                                   | The EFS does not have encryption at rest enabled.                                                                                                                                 | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).                                                                                                                                                                                                                                                                                                                                                                                                                                   | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-ElastiCacheRedisClusterAutomaticBackup](https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html)            | The ElastiCache Redis cluster does not retain automatic backups for at least 15 days.                                                                                             | Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.                                                                                                                                                                                                                                                                                                                                                                                                                 | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/beanstalk-enhanced-health-reporting-enabled.html)         | The Elastic Beanstalk environment does not have enhanced health reporting enabled.                                                                                                | AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.                                                                                                                                                                                       | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-ElasticBeanstalkManagedUpdatesEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elastic-beanstalk-managed-updates-enabled.html)                    | The Elastic Beanstalk environment does not have managed updates enabled.                                                                                                          | Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.                                                                                                                                                                                                                                                                                                  | 164.308(a)(5)(ii)(A)                                                                                                                                            |
| [HIPAA.Security-ELBACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html)                                             | The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                                         | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-ELBCrossZoneLoadBalancingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html)                             | The CLB does not balance traffic between at least 2 Availability Zones.                                                                                                           | Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.                                                                                                                                                                                                                          | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [HIPAA.Security-ELBDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html)                                       | The ALB, NLB, or GLB does not have deletion protection enabled.                                                                                                                   | This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                                                                                                                                  | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [HIPAA.Security-ELBLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html)                                                              | The ELB does not have logging enabled.                                                                                                                                            | Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client\'s IP address, latencies, request paths, and server responses.                                                                                                                                                                                                                | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-ELBTlsHttpsListenersOnly](https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html)                                              | The CLB does not restrict its listeners to only the SSL and HTTPS protocols.                                                                                                      | Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                         | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-ELBv2ACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elbv2-acm-certificate-required.html)                                         | The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                   | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-EMRKerberosEnabled](https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html)                                                            | The EMR cluster does not have Kerberos enabled.                                                                                                                                   | The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.                                                                                                                                                                                                                                                                                                                                                                                     | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-IAMUserNoPolicies](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html)                                                       | The IAM policy is attached at the user level.                                                                                                                                     | Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.                                                                                                                                                                                                                                                                                                                                                                                                                                   | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-IAMPolicyNoStatementsWithAdminAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html)                    | The IAM policy grants admin access.                                                                                                                                               | AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.                                                                                                                                                                                                                | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-IAMPolicyNoStatementsWithFullAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-full-access.html)                      | The IAM policy grants full access.                                                                                                                                                | Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                                                                                                        | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-IAMGroupMembership](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html)                                                 | The IAM user does not belong to any group(s).                                                                                                                                     | AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                           | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-IAMUserNoPolicies](https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html)                                                       | The IAM Group, User, or Role contains an inline policy.                                                                                                                           | AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.                                                                                                                                                                                                                                                                                                                                                                                               | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [HIPAA.Security-LambdaConcurrency](https://docs.aws.amazon.com/config/latest/developerguide/lambda-concurrency-check.html)                                                         | The Lambda function is not configured with function-level concurrent execution limits.                                                                                            | Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.                                                                                                                                                                                                                                                                                                                                                                                       | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-LambdaDlq](https://docs.aws.amazon.com/config/latest/developerguide/lambda-dlq-check.html)                                                                         | The Lambda function is not configured with a dead-letter configuration.                                                                                                           | Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.                                                                                                                                                                                                                                                                                                                                                                                                             | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-LambdaInsideVPC](https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html)                                                                  | The Lambda function is not VPC enabled.                                                                                                                                           | Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.                                                                                                                                                                                                                                                                                                                                                                                                             | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-OpenSearchEncryptedAtRest](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html)                                          | The OpenSearch Service domain does not have encryption at rest enabled.                                                                                                           | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                             | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-OpenSearchInVPCOnly](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html)                                                      | The OpenSearch Service domain is not running within a VPC.                                                                                                                        | VPCs help secure your AWS resources and provide an extra layer of protection.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-OpenSearchLogsToCloudWatch](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-logs-to-cloudwatch.html)                                        | The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs.                                                                                | Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.                                                                                                                                                                                                                                                                                                                              | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [HIPAA.Security-OpenSearchNodeToNodeEncryption](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html)                         | The OpenSearch Service domain does not have node-to-node encryption enabled.                                                                                                      | Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                                 | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [HIPAA.Security-RDSAutomaticMinorVersionUpgradeEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-automatic-minor-version-upgrade-enabled.html)                 | The RDS DB instance does not have automatic minor version upgrades enabled.                                                                                                       | Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.                                                                                                                                                                                                                                                                                                          | 164.308(a)(5)(ii)(A)                                                                                                                                            |
| [HIPAA.Security-RDSEnhancedMonitoringEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html)                                       | The RDS DB instance does not enhanced monitoring enabled.                                                                                                                         | Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.                                                                                                                                                                                                                                                                                                                                                                                                                 | 164.312(b)                                                                                                                                                      |
| [HIPAA.Security-RDSInstanceBackupEnabled](https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html)                                                | The RDS DB instance does not have backup enabled.                                                                                                                                 | The backup feature of Amazon RDS creates backups of your databases and transaction logs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [HIPAA.Security-RDSInstanceDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html)                      | The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.                                                                                               | Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                                                                   | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [HIPAA.Security-RDSInstanceMultiAZSupport](https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html)                                                     | The non-Aurora RDS DB instance does not have multi-AZ support enabled.                                                                                                            | Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.                                        | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [HIPAA.Security-RDSInstancePublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html)                                           | The RDS DB instance allows public access.                                                                                                                                         | Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                            | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-RDSLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html)                                                              | The RDS DB instance does not have all CloudWatch log types exported.                                                                                                              | To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.                                                                                                                                                                                                                                                                                                                     | 164.308(a)(3)(ii)(A), 164.308(a)(5)(ii)(C)                                                                                                                      |
| [HIPAA.Security-RDSStorageEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html)                                                          | The RDS DB instance or Aurora DB cluster does not have storage encrypted.                                                                                                         | Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                    | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-RedshiftBackupEnabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-backup-enabled.html)                                                      | The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days.                                                              | To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.                                                                                                                                                                                                                                     | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [HIPAA.Security-RedshiftClusterConfiguration](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html)                                  | The Redshift cluster does not have encryption or audit logging enabled.                                                                                                           | To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.                                                                                                                                                                                                                                                                  | 164.312(a)(2)(iv), 164.312(b), 164.312(e)(2)(ii)                                                                                                                |
| [HIPAA.Security-RedshiftClusterMaintenanceSettings](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html)                      | The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured.                         | Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                                | 164.308(a)(5)(ii)(A), 164.308(a)(7)(ii)(A)                                                                                                                      |
| [HIPAA.Security-RedshiftClusterPublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html)                                   | The Redshift cluster allows public access.                                                                                                                                        | Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-RedshiftEnhancedVPCRoutingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-enhanced-vpc-routing-enabled.html)                            | The Redshift cluster does not have enhanced VPC routing enabled.                                                                                                                  | Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.                                                                                                                                                                                                                                                                | 164.312(e)(1)                                                                                                                                                   |
| [HIPAA.Security-S3BucketLevelPublicAccessProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-level-public-access-prohibited.html)                       | The S3 bucket does not prohibit public access through bucket level settings.                                                                                                      | Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-S3BucketLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html)                                                   | The S3 Buckets does not have server access logs enabled.                                                                                                                          | Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.                                                                                                             | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [HIPAA.Security-S3BucketPublicReadProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html)                                      | The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.                                                                | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-S3BucketPublicWriteProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html)                                    | The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.                                                               | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-S3BucketReplicationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html)                                           | The S3 Bucket does not have replication enabled.                                                                                                                                  | Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.                                                                                                                                                                                                                                                                                                       | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [HIPAA.Security-S3BucketServerSideEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html)                       | The S3 Bucket does not have default server-side encryption enabled.                                                                                                               | Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 164.312(a)(2)(iv), 164.312(c)(2), 164.312(e)(2)(ii)                                                                                                             |
| [HIPAA.Security-S3BucketVersioningEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html)                                             | The S3 Bucket does not have versioning enabled.                                                                                                                                   | Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.                                                                                                                                                                                                                                                                                                                                                               | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B), 164.312(c)(1), 164.312(c)(2)                                                                      |
| [HIPAA.Security-S3DefaultEncryptionKMS](https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html)                                                   | The S3 Bucket is not encrypted with a KMS Key by default.                                                                                                                         | Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data                                                                                                                                                                                                                                                                                                                                                           | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-SageMakerEndpointConfigurationKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html) | The SageMaker resource endpoint is not encrypted with a KMS key.                                                                                                                  | Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-SageMakerNotebookInstanceKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html)           | The SageMaker notebook is not encrypted with a KMS key.                                                                                                                           | Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-SageMakerNotebookNoDirectInternetAccess](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html)               | The SageMaker notebook does not disable direct internet access.                                                                                                                   | By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [HIPAA.Security-SecretsManagerUsingKMSKey](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-using-cmk.html)                                                 | The secret is not encrypted with a KMS Customer managed key.                                                                                                                      | To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                              | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-SNSEncryptedKMS](https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html)                                                                  | The SNS topic does not have KMS encryption enabled.                                                                                                                               | To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                     | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [HIPAA.Security-VPCNoUnrestrictedRouteToIGW](https://docs.aws.amazon.com/config/latest/developerguide/no-unrestricted-route-to-igw.html)                                           | The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').                                                                                  | Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.                                                                                                                                                                                                                                                                                                                                                | 164.312(e)(1)                                                                                                                                                   |
| [HIPAA.Security-VPCSubnetAutoAssignPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/subnet-auto-assign-public-ip-disabled.html)                          | The subnet auto-assigns public IP addresses.                                                                                                                                      | Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.                                                                                                                                                                                                                                                 | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |

### Excluded Rules

Unimplemented rules from the AWS HIPAA Security Conformance Pack.

| AWS Config Rule Name                                                                                                                                                             | Summary                                                                                                                                                | Relevant Control ID(s)                                                                                                                                          |
| -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------ | --------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| [access-keys-rotated](https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html)                                                                   | IAM active access keys are rotated within the number of days specified in maxAccessKeyAge.                                                             | 164.308(a)(4)(ii)(B)                                                                                                                                            |
| [cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html)                                                                           | The AWS Account has CloudTrail enabled.                                                                                                                | 164.308(a)(3)(ii)(A), 164.308(a)(5)(ii)(C), 164.312(b)                                                                                                          |
| [cloudtrail-s3-dataevents-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html)                                               | The AWS Account has at least once AWS CloudTrail that logs Amazon S3 data events for all S3 buckets.                                                   | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [dynamodb-autoscaling-enabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html)                                                       | DynamoDB tables have autoscaling enabled.                                                                                                              | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [dynamodb-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html)                                                                 | DynamoDB tables are in an AWS Backup Plan.                                                                                                             | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [dynamodb-table-encrypted-kms](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html)                                                       | DynamoDB tables are encrypted in KMS.<sup>[1](#dynamodb-table-encrypted-kms)</sup>                                                                     | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [dynamodb-throughput-limit-check](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-throughput-limit-check.html)                                                 | DynamoDB throughput is approaching the maximum limit for the AWS Account.                                                                              | 164.312(b)                                                                                                                                                      |
| [ebs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html)                                                                           | EBS volumes are in an AWS Backup Plan.                                                                                                                 | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [ebs-snapshot-public-restorable-check](https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html)                                       | EBS snapshots can not be publicly restored.                                                                                                            | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [ec2-ebs-encryption-by-default](https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html)                                                     | Default encryption for EBS volumes is enabled at the AWS Account level.                                                                                | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [ec2-imdsv2-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html)                                                                               | EC2 instances have IMDSV2 (instance metadata service version 2) enabled.                                                                               | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                                                                     |
| [ec2-instance-managed-by-systems-manager](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html)                                 | EC2 instances are managed by Systems Manager.                                                                                                          | 164.308(a)(5)(ii)(B)                                                                                                                                            |
| [ec2-managedinstance-association-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html) | EC2 instance Systems Manager association compliances are COMPLIANT after the association execution.                                                    | 164.308(a)(5)(ii)(B)                                                                                                                                            |
| [ec2-managedinstance-patch-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html)             | The EC2 instance's Systems Manager patch compliances are COMPLIANT after the patch execution.                                                          | 164.308(a)(5)(ii)(A), 164.308(a)(5)(ii)(B)                                                                                                                      |
| [efs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html)                                                                           | EFSs are in an AWS Backup plan.                                                                                                                        | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [emr-master-no-public-ip](https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html)                                                                 | EMR clusters' master nodes have no public IP.                                                                                                          | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [encrypted-volumes](https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html)                                                                             | EC2 instances are only attached to encrypted EBS volumes.                                                                                              | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [guardduty-enabled-centralized](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html)                                                     | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                                              | 164.308(a)(5)(ii)(C), 164.308(a)(6)(ii), 164.312(b)                                                                                                             |
| [guardduty-non-archived-findings](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-non-archived-findings.html)                                                 | Amazon GuardDuty has archived low/medium/high severity findings older than the specified number in the daysLowSev/daysMediumSev/daysHighSev parameter. | 164.312(b)                                                                                                                                                      |
| [iam-group-has-users-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html)                                                             | IAM groups have at least one IAM user.                                                                                                                 | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [iam-password-policy](https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html)                                                                         | Account password policy for IAM users meet the specified requirements indicated in the parameters.                                                     | 164.308(a)(4)(ii)(B), 164.308(a)(5)(ii)(D), 164.312(d)                                                                                                          |
| [iam-root-access-key-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html)                                                             | The Account IAM Root User has an access key(s).                                                                                                        | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(a)(2)(i)                                             |
| [iam-user-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html)                                                                       | IAM Users have MFA enabled.                                                                                                                            | 164.308(a)(4)(ii)(B), 164.312(d)                                                                                                                                |
| [iam-user-unused-credentials-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html)                                             | IAM User passwords and active access keys have been used within a specified number of days.                                                            | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)                                                               |
| [lambda-function-public-access-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html)                               | Lambda function policies do not allow public access.                                                                                                   | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [mfa-enabled-for-iam-console-access](https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html)                                           | IAM Users have MFA enabled for console access.                                                                                                         | 164.312(d)                                                                                                                                                      |
| [multi-region-cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html)                                                 | The AWS Account has at least one multi-region CloudTrail enabled.                                                                                      | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [rds-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html)                                                                           | RDS DB instances are in an AWS Backup Plan.                                                                                                            | 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)                                                                                                    |
| [rds-snapshot-encrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html)                                                                   | RDS snapshots are encrypted.                                                                                                                           | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [rds-snapshots-public-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html)                                                 | RDS snapshots are not public.                                                                                                                          | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [redshift-cluster-kms-enabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-kms-enabled.html)                                                       | Redshift clusters are encrypted with one of the specified KMS keys.                                                                                    | 164.312(a)(2)(iv), 164.312(e)(2)(ii)                                                                                                                            |
| [redshift-require-tls-ssl](https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html)                                                              | Redshift clusters require TLS/SSL encryption.                                                                                                          | 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                                           |
| [root-account-hardware-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html)                                             | The AWS Account root user is hardware MFA enabled.                                                                                                     | 164.312(d)                                                                                                                                                      |
| [root-account-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html)                                                               | The AWS Account root user is MFA enabled.                                                                                                              | 164.312(d)                                                                                                                                                      |
| [s3-account-level-public-access-blocks-periodic](https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks-periodic.html)                   | The AWS Account blocks S3 public access.                                                                                                               | 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)                                                                      |
| [s3-bucket-policy-grantee-check](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html)                                                   | S3 Buckets have access restricted to a list of provided AWS principals, federated users, service principals, IP addresses, or VPCs.                    | 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1) |
| [s3-bucket-ssl-requests-only](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html)                                                         | S3 Buckets only allow SSL requests.                                                                                                                    | 164.312(a)(2)(iv), 164.312(c)(2), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)                                                                            |
| [secretsmanager-rotation-enabled-check](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-rotation-enabled-check.html)                                     | Secrets have rotation enabled.                                                                                                                         | 164.308(a)(4)(ii)(B)                                                                                                                                            |
| [secretsmanager-scheduled-rotation-success-check](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-scheduled-rotation-success-check.html)                 | Secrets succesfully rotate per their rotation schedules.                                                                                               | 164.308(a)(4)(ii)(B)                                                                                                                                            |
| [secretsmanager-secret-periodic-rotation](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-periodic-rotation.html)                                 | All secrets have been rotated in the past 90 days.                                                                                                     | 164.308(a)(4)(ii)(B)                                                                                                                                            |
| [secretsmanager-secret-unused](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-unused.html)                                                       | All secrets have been accessed in the past 90 days.                                                                                                    | 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C)                                                                                                                      |
| [securityhub-enabled](https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html)                                                                         | The AWS Account has Security Hub enabled.                                                                                                              | 164.312(b)                                                                                                                                                      |
| [ssm-document-not-public](https://docs.aws.amazon.com/config/latest/developerguide/ssm-document-not-public.html)                                                                 | SSM documents are not public.                                                                                                                          | 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)        |
| [vpc-flow-logs-enabled](https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html)                                                                     | VPC flow logs are found and enabled for VPCs.                                                                                                          | 164.308(a)(3)(ii)(A), 164.312(b)                                                                                                                                |
| [vpc-sg-open-only-to-authorized-ports](https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html)                                       | Security Groups only allow inbound 0.0.0.0/0 from authorized TCP or UDP ports                                                                          | 164.312(e)(1)                                                                                                                                                   |
| [vpc-vpn-2-tunnels-up](https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html)                                                                       | Site-to-Site VPN tunnels have an UP status                                                                                                             | 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)                                                                                                                          |
| [wafv2-logging-enabled](https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html)                                                                     | Logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control lists (ACLs).                                        | 164.312(b)                                                                                                                                                      |

## NIST 800-53 rev 4

The [Operational Best Practices for NIST 800-53 rev 4](https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html) Conformance Pack offers a collection of AWS Config Rules that enable customers to align to a subset of the NIST 800-53 standards. The NIST 800-53 rev 4 Rules Pack implements many of the checks within the conformance pack.

### Warnings

| Rule ID                                                                                                                                              | Cause                                                                | Explanation                                                                                                                                                                                                                                                                                                                          | Relevant Control ID(s) |
| ---------------------------------------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ | ---------------------- |
| [NIST.800.53.R4-EC2CheckDefaultSecurityGroupClosed](https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html) | The VPC's default security group allows inbound or outbound traffic. | When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources. | AC-4, SC-7, SC-7(3)    |

### Errors

| Rule ID                                                                                                                                                                    | Cause                                                                                                                                                      | Explanation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | Relevant Control ID(s)                                                                                                     |
| -------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------------------------------------------------------------- |
| [NIST.800.53.R4-ALBHttpDropInvalidHeaderEnabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html)                       | The ALB does not have invalid HTTP header dropping enabled.                                                                                                | Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                      | AC-17(2)                                                                                                                   |
| [NIST.800.53.R4-ALBHttpToHttpsRedirection](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html)                              | The ALB's HTTP listeners are not configured to redirect to HTTPS.                                                                                          | To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                     | AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23                                                                                |
| [NIST.800.53.R4-APIGWCacheEnabledAndEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html)                           | The API Gateway stage does not have caching enabled and encrypted for all methods.                                                                         | To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                        | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-APIGWExecutionLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html)                              | The API Gateway stage does not have execution logging enabled for all methods.                                                                             | API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.                                                                                                                                                                                                                                                                                                                                                                      | AU-2(a)(d), AU-3, AU-12(a)(c)                                                                                              |
| [NIST.800.53.R4-AutoScalingHealthCheck](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html)                          | The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks.                                                       | The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.                                                                                                                                                                                                                                                                                                                                                            | SC-5                                                                                                                       |
| [NIST.800.53.R4-CloudTrailCloudWatchLogsEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html)                       | The trail does not have CloudWatch logs enabled.                                                                                                           | Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.                                                                                                                                                                                                                                                                                                                                                             | AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c) |
| [NIST.800.53.R4-CloudTrailEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html)                                 | The trail does not have encryption enabled.                                                                                                                | Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.                                                                                                                                                                                                                                                                                                                                                                                                       | AU-9, SC-13, SC-28                                                                                                         |
| [NIST.800.53.R4-CloudTrailLogFileValidationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html)                 | The trail does not have log file validation enabled.                                                                                                       | Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.                                                                                       | AU-9, SC-13, SC-28                                                                                                         |
| [NIST.800.53.R4-CloudWatchAlarmAction](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html)                                        | The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled.                                      | Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.                                                                                                                                                                                                                                                                        | AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)                                |
| [NIST.800.53.R4-CloudWatchLogGroupEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html)                                 | The CloudWatch Log Group is not encrypted with an AWS KMS key.                                                                                             | To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.                                                                                                                                                                                                                                                                                                                                                                                                                           | AU-9, SC-13, SC-28                                                                                                         |
| [NIST.800.53.R4-CodeBuildCheckEnvVars](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html)                               | The Codebuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.        | Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.                                                                                                                                                                                                                                                                                                                                                                                          | AC-6, IA-5(7), SA-3(a)                                                                                                     |
| [NIST.800.53.R4-CodeBuildURLCheck](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html)                                  | The Codebuild project which utilizes either a GitHub or BitBucket repository does not utilize OAUTH.                                                       | OAUTH is the most secure method of authenticating your Codebuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.                                                                                                                                                                                                                                                                                                          | SA-3(a)                                                                                                                    |
| [NIST.800.53.R4-DMSReplicationNotPublic](https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html)                                         | The DMS replication instance is public.                                                                                                                    | DMS replication instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                         | AC-4                                                                                                                       |
| [NIST.800.53.R4-DynamoDBPITREnabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html)                                                  | The DynamoDB table does not have Point-in-time Recovery enabled.                                                                                           | The recovery maintains continuous backups of your table for the last 35 days.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | CP-9(b), CP-10, SI-12                                                                                                      |
| [NIST.800.53.R4-EC2CheckCommonPortsRestricted](https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html)                                      | The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333). | Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.                                                                                                                                                                                                                                                                                     | AC-4, CM-2, SC-7, SC-7(3)                                                                                                  |
| [NIST.800.53.R4-EC2CheckDetailedMonitoring](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-detailed-monitoring-enabled.html)                        | The EC2 instance does not have detailed monitoring enabled.                                                                                                | Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.                                                                                                                                                                                                                                                                                                                                                                                                                   | CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)                                                                                         |
| [NIST.800.53.R4-EC2CheckInsideVPC](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html)                                                     | The EC2 instance is not within a VPC.                                                                                                                      | Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.                                                                                                                                                                                                                                                                                                                                                                      | AC-4, SC-7, SC-7(3)                                                                                                        |
| [NIST.800.53.R4-EC2CheckNoPublicIPs](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html)                                              | The EC2 instance is associated with a public IP address.                                                                                                   | Amazon EC2 instances can contain sensitive information and access control is required for such resources.                                                                                                                                                                                                                                                                                                                                                                                                                             | AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                        |
| [NIST.800.53.R4-EC2CheckSSHRestricted](https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html)                                                       | The Security Group allows unrestricted SSH access.                                                                                                         | Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.                                                                                                                                                                                                                                                                                                                                                                                                         | AC-4, SC-7, SC-7(3)                                                                                                        |
| [NIST.800.53.R4-EFSEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html)                                                           | The EFS does not have encryption at rest enabled.                                                                                                          | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).                                                                                                                                                                                                                                                                                                                                                                                            | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-ElastiCacheRedisClusterAutomaticBackup](https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html)    | The ElastiCache Redis cluster does not retain automatic backups for at least 15 days.                                                                      | Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.                                                                                                                                                                                                                                                                                                                                                                          | CP-9(b), CP-10, SI-12                                                                                                      |
| [NIST.800.53.R4-ELBCrossZoneBalancing](https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html)                                | The CLB does not balance traffic between at least 2 Availability Zones.                                                                                    | Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.                                                                                                                                                                                   | SC-5, CP-10                                                                                                                |
| [NIST.800.53.R4-ELBDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html)                               | The ALB, NLB, or GLB does not have deletion protection enabled.                                                                                            | This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                                                                                           | CM-2, CP-10                                                                                                                |
| [NIST.800.53.R4-ELBListenersUseSSLOrHTTPS](https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html)                                     | The CLB does not restrict its listeners to only the SSL and HTTPS protocols.                                                                               | Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                  | AC-17(2), SC-7, SC-8, SC-8(1), SC-23                                                                                       |
| [NIST.800.53.R4-ELBLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html)                                                      | The ELB does not have logging enabled.                                                                                                                     | Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client\'s IP address, latencies, request paths, and server responses.                                                                                                                                                                         | AU-2(a)(d)                                                                                                                 |
| [NIST.800.53.R4-ELBUseACMCerts](https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html)                                                | The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                  | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                     | AC-17(2), SC-7, SC-8, SC-8(1), SC-13                                                                                       |
| [NIST.800.53.R4-EMRKerberosEnabled](https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html)                                                    | The EMR cluster does not have Kerberos enabled.                                                                                                            | The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.                                                                                                                                                                                                                                                                                                                                              | AC-2(j), AC-3, AC-5c, AC-6                                                                                                 |
| [NIST.800.53.R4-IAMGroupMembership](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html)                                         | The IAM user does not belong to any group(s).                                                                                                              | AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                    | AC-2(1), AC-2(j), AC-3, AC-6                                                                                               |
| [NIST.800.53.R4-IAMNoInlinePolicy](https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html)                                               | The IAM Group, User, or Role contains an inline policy.                                                                                                    | AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.                                                                                                                                                                                                                                                                                                                                                        | AC-6                                                                                                                       |
| [NIST.800.53.R4-IAMUserNoPolicies](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html)                                               | The IAM policy is attached at the user level.                                                                                                              | Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.                                                                                                                                                                                                                                                                                                                                                                                            | AC-2(j), AC-3, AC-5c, AC-6                                                                                                 |
| [NIST.800.53.R4-IAMPolicyNoStatementsWithAdminAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html)            | The IAM policy grants admin access.                                                                                                                        | AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.                                                                                                                                                                         | AC-2(1), AC-2(j), AC-3, AC-6                                                                                               |
| [NIST.800.53.R4-LambdaFunctionsInsideVPC](https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html)                                                 | The Lambda function is not VPC enabled.                                                                                                                    | Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.                                                                                                                                                                                                                                                                                                                                                                      | AC-4, SC-7, SC-7(3)                                                                                                        |
| [NIST.800.53.R4-OpenSearchEncryptedAtRest](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html)                                  | The OpenSearch Service domain does not have encryption at rest enabled.                                                                                    | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                      | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-OpenSearchNodeToNodeEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html)                  | The OpenSearch Service domain does not have node-to-node encryption enabled.                                                                               | Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                          | SC-7, SC-8, SC-8(1)                                                                                                        |
| [NIST.800.53.R4-OpenSearchRunningWithinVPC](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html)                                       | The OpenSearch Service domain is not running within a VPC.                                                                                                 | VPCs help secure your AWS resources and provide an extra layer of protection.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | AC-4, SC-7, SC-7(3)                                                                                                        |
| [NIST.800.53.R4-RDSEnhancedMonitoringEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html)                               | The RDS DB Instance does not enhanced monitoring enabled.                                                                                                  | Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.                                                                                                                                                                                                                                                                                                                                                                          | CA-7(a)(b)                                                                                                                 |
| [NIST.800.53.R4-RDSInstanceBackupEnabled](https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html)                                        | The RDS DB Instance does not have backup enabled.                                                                                                          | The backup feature of Amazon RDS creates backups of your databases and transaction logs.                                                                                                                                                                                                                                                                                                                                                                                                                                              | CP-9(b), CP-10, SI-12                                                                                                      |
| [NIST.800.53.R4-RDSInstanceDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html)              | The RDS DB Instance or Aurora Cluster does not have deletion protection enabled.                                                                           | Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                               | SC-5                                                                                                                       |
| [NIST.800.53.R4-RDSInstanceMultiAZSupport](https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html)                                             | The RDS DB Instance does not have multi-AZ support.                                                                                                        | Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete. | CP-10, SC-5, SC-36                                                                                                         |
| [NIST.800.53.R4-RDSInstancePublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html)                                   | The RDS DB Instance allows public access.                                                                                                                  | Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                     | AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                        |
| [NIST.800.53.R4-RDSLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html)                                                      | The RDS DB Instance does not have all CloudWatch log types exported.                                                                                       | To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.                                                                                                                                                                                                                                                                              | AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)                                                                            |
| [NIST.800.53.R4-RDSStorageEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html)                                                  | The RDS DB Instance or Aurora Cluster does not have storage encrypted.                                                                                     | Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-RedshiftClusterConfiguration](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html)                          | The Redshift cluster does not have encryption or audit logging enabled.                                                                                    | To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.                                                                                                                                                                                                                           | AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13                                                                     |
| [NIST.800.53.R4-RedshiftClusterPublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html)                           | The Redshift cluster allows public access.                                                                                                                 | Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                           | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                  |
| [NIST.800.53.R4-S3BucketDefaultLockEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-default-lock-enabled.html)                                  | The S3 Bucket does not have object lock enabled.                                                                                                           | Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                       | SC-28                                                                                                                      |
| [NIST.800.53.R4-S3BucketLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html)                                           | The S3 Buckets does not have server access logs enabled.                                                                                                   | Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.                                                                      | AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)                                                                                     |
| [NIST.800.53.R4-S3BucketPublicReadProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html)                              | The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.                                         | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                    | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                  |
| [NIST.800.53.R4-S3BucketPublicWriteProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html)                            | The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.                                        | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                    | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                  |
| [NIST.800.53.R4-S3BucketReplicationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html)                                   | The S3 Bucket does not have replication enabled.                                                                                                           | Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.                                                                                                                                                                                                                                                                | AU-9(2), CP-9(b), CP-10, SC-5, SC-36                                                                                       |
| [NIST.800.53.R4-S3BucketServerSideEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html)               | The S3 Bucket does not have default server-side encryption enabled.                                                                                        | Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                           | AU-9(2), CP-9(b), CP-10, SC-5, SC-36                                                                                       |
| [NIST.800.53.R4-S3BucketVersioningEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html)                                     | The S3 Bucket does not have versioning enabled.                                                                                                            | Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.                                                                                                                                                                                                                                                                                                                        | CP-10, SI-12                                                                                                               |
| [NIST.800.53.R4-SageMakerEndpointKMS](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html)                   | The SageMaker resource endpoint is not encrypted with a KMS key.                                                                                           | Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                  | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-SageMakerNotebookDirectInternetAccessDisabled](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html) | The SageMaker notebook does not disable direct internet access.                                                                                            | By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.                                                                                                                                                                                                                                                                                                                                                                                                                          | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                  |
| [NIST.800.53.R4-SageMakerNotebookKMS](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html)                        | The SageMaker notebook is not encrypted with a KMS key.                                                                                                    | Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                  | SC-13, SC-28                                                                                                               |
| [NIST.800.53.R4-SNSEncryptedKMS](https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html)                                                          | The SNS topic does not have KMS encryption enabled.                                                                                                        | To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                              | SC-13, SC-28                                                                                                               |

### Excluded Rules

Unimplemented rules from the AWS NIST 800-53 rev 4 Conformance Pack

| AWS Config Rule Name                                                                                                                                                             | Summary                                                                                                                                                                                                                                                                              | Relevant Control ID(s)                                                                                                                             |
| -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ | -------------------------------------------------------------------------------------------------------------------------------------------------- |
| [access-keys-rotated](https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html)                                                                         | IAM active access keys are rotated within the number of days specified in maxAccessKeyAge.                                                                                                                                                                                           | AC-2(1), AC-2(j)                                                                                                                                   |
| [acm-certificate-expiration-check](https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html)                                               | AWS Certificate Manager Certificates in your account are valid and not marked for expiration within the specified number of days.                                                                                                                                                    | AC-4, AC-17(2), SC-12                                                                                                                              |
| [alb-waf-enabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html)                                                                                 | Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).                                                                                                                                                                                                      | SC-7, SI-4(a)(b)(c)                                                                                                                                |
| [cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html)                                                                           | The AWS Account has CloudTrail enabled.                                                                                                                                                                                                                                              | AC-2(4)                                                                                                                                            |
| [cloudtrail-security-trail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-security-trail-enabled.html)                                             | The AWS Account has at least once AWS CloudTrail that records global service events, is a multi-region trail, has Log file validation enabled, encrypted with a KMS key, records events for reads and writes, records management events, and does not exclude any management events. | CM-2                                                                                                                                               |
| [cloudtrail-s3-dataevents-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html)                                               | The AWS Account has at least once AWS CloudTrail that logs Amazon S3 data events for all S3 buckets.                                                                                                                                                                                 | AC-2(g)                                                                                                                                            |
| [cmk-backing-key-rotation-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html)                                               | KMS backing keys have key rotation enabled and match to the key ID of the customer managed key.                                                                                                                                                                                      | SC-12                                                                                                                                              |
| [cw-loggroup-retention-period-check](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                                           | CloudWatch LogGroup retention period is set to specific number of days and is greater than the configured retention period.                                                                                                                                                          | AU-11, SI-12                                                                                                                                       |
| [dynamodb-autoscaling-enabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html)                                                       | DynamoDB has autoscaling enabled.                                                                                                                                                                                                                                                    | CP-10, SC-5                                                                                                                                        |
| [dynamodb-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html)                                                                 | DynamoDB tables are in an AWS Backup Plan.                                                                                                                                                                                                                                           | CP-9(b), CP-10, SI-12                                                                                                                              |
| [dynamodb-table-encrypted-kms](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html)                                                       | DynamoDB tables are encrypted in KMS.<sup>[1](#dynamodb-table-encrypted-kms)</sup>                                                                                                                                                                                                   | SC-13                                                                                                                                              |
| [ebs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html)                                                                           | EBS volumes are in an AWS Backup Plan.                                                                                                                                                                                                                                               | CP-9(b), CP-10, SI-12                                                                                                                              |
| [ebs-snapshot-public-restorable-check](https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html)                                       | EBS snapshots can be publicly restored.                                                                                                                                                                                                                                              | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                                          |
| [ec2-ebs-encryption-by-default](https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html)                                                     | Default encryption for EBS volumes is enabled at the AWS Account level.                                                                                                                                                                                                              | SC-28                                                                                                                                              |
| [ec2-imdsv2-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html)                                                                               | EC2 instances have IMDSV2 (instance metadata service version 2) enabled.                                                                                                                                                                                                             | AC-6                                                                                                                                               |
| [ec2-instance-managed-by-systems-manager](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html)                                 | EC2 instances are managed by Systems Manager.                                                                                                                                                                                                                                        | CM-2, CM-7(a), CM-8(1), CM-8(3)(a), SA-10, SA-3(a), SI-2(2), SI-7(1)                                                                               |
| [ec2-managedinstance-association-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html) | Managed EC2 instances are compliant with their association's standards.                                                                                                                                                                                                              | CM-2, CM-7(a), CM-8(3)(a), SI-2(2)                                                                                                                 |
| [ec2-managedinstance-patch-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html)             | EC2 instances are compliant with their patch requirements.                                                                                                                                                                                                                           | SI-7(1), SI-2(2), CM-8(3)(a)                                                                                                                       |
| [ec2-stopped-instance](https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html)                                                                       | EC2 instances have not been stopped for more than the allowed number of days.                                                                                                                                                                                                        | CM-2                                                                                                                                               |
| [ec2-volume-inuse-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html)                                                                   | When an instances is terminated, its associated EBS volumes are marked for deletion.                                                                                                                                                                                                 | CM-2, SC-4                                                                                                                                         |
| [efs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html)                                                                           | EFSs are in an AWS Backup plan.                                                                                                                                                                                                                                                      | CP-9(b), CP-10, SI-12                                                                                                                              |
| [emr-master-no-public-ip](https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html)                                                                 | EMR clusters' master nodes have no public IP.                                                                                                                                                                                                                                        | AC-4, AC-21(b), SC-7, SC-7(3)                                                                                                                      |
| [encrypted-volumes](https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html)                                                                             | EC2 instances are only attached to encrypted EBS volumes.                                                                                                                                                                                                                            | SC-13, SC-28                                                                                                                                       |
| [guardduty-enabled-centralized](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html)                                                     | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                                                                                                                                                                            | AC-17(1), AC-2(1), AC-2(12)(a), AC-2(4), AC-2(g), AU-6(1)(3), CA-7(a)(b), RA-5, SA-10, SI-4(1), SI-4(16), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c) |
| [guardduty-non-archived-findings](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-non-archived-findings.html)                                                 | Amazon GuardDuty does not have non archived low/medium/high severity findings older than the specified number in the daysLowSev/daysMediumSev/daysHighSev parameter.                                                                                                                 | IR-4(1), IR-6(1), IR-7(1), RA-5, SA-10                                                                                                             |
| [iam-group-has-users-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html)                                                             | IAM groups have at least one IAM user.                                                                                                                                                                                                                                               | AC-2(j)                                                                                                                                            |
| [iam-password-policy](https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html)                                                                         | The account password policy for IAM users meets the specified requirements indicated in the parameters.                                                                                                                                                                              | AC-2(1)                                                                                                                                            |
| [iam-root-access-key-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html)                                                             | The Account IAM Root User has an access key(s).                                                                                                                                                                                                                                      | AC-2(f), AC-2(j), AC-3, AC-6, AC-6(10)                                                                                                             |
| [iam-user-unused-credentials-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html)                                             | IAM User passwords and active access keys have been used within a specified number of days.                                                                                                                                                                                          | AC-2(1), AC-2(3), AC-2(f), AC-3, AC-6                                                                                                              |
| [iam-user-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html)                                                                       | AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.                                                                                                                                                                                             | IA-2(1)(2)(11)                                                                                                                                     |
| [internet-gateway-authorized-vpc-only](https://docs.aws.amazon.com/config/latest/developerguide/internet-gateway-authorized-vpc-only.html)                                       | Internet gateways (IGWs) are only attached to an authorized Amazon Virtual Private Cloud (VPCs).                                                                                                                                                                                     | AC-17(3), AC-4, SC-7, SC-7(3)                                                                                                                      |
| [kms-cmk-not-scheduled-for-deletion](https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html)                                           | AWS KMS keys are not scheduled for deletion in AWS Key Management Service (KMS).                                                                                                                                                                                                     | SC-12, SC-28                                                                                                                                       |
| [lambda-function-public-access-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html)                               | Public access to lambda functions is prohibited.                                                                                                                                                                                                                                     | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                                          |
| [mfa-enabled-for-iam-console-access](https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html)                                           | IAM Users have MFA enabled for console access.                                                                                                                                                                                                                                       | IA-2(1)(2)(11)                                                                                                                                     |
| [multi-region-cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html)                                                 | The AWS Account has at least one multi-region CloudTrail enabled.                                                                                                                                                                                                                    | AC-2(4), AU-12(a)(c), AU-2(a)(d), AU-3                                                                                                             |
| [rds-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html)                                                                           | RDS DB instances are in an AWS Backup Plan.                                                                                                                                                                                                                                          | CP-10, CP-9(b), SI-12                                                                                                                              |
| [rds-snapshot-encrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html)                                                                   | RDS snapshots are encrypted.                                                                                                                                                                                                                                                         | SC-28                                                                                                                                              |
| [rds-snapshots-public-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html)                                                 | RDS snapshots are not public.                                                                                                                                                                                                                                                        | AC-3, AC-4, AC-6, AC-21(b), SC-7/SC-7(3)                                                                                                           |
| [redshift-require-tls-ssl](https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html)                                                               | Redshift clusters require TLS/SSL encryption.                                                                                                                                                                                                                                        | AC-17(2), SC-7, SC-8, SC-8(1), SC-13                                                                                                               |
| [root-account-hardware-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html)                                             | The AWS Account root user is hardware MFA enabled.                                                                                                                                                                                                                                   | IA-2(1)(11)                                                                                                                                        |
| [root-account-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html)                                                               | The AWS Account root user is MFA enabled.                                                                                                                                                                                                                                            | AC-2(j), IA-2(1)(11)                                                                                                                               |
| [s3-account-level-public-access-blocks](https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks.html)                                     | The AWS Account blocks S3 public access.                                                                                                                                                                                                                                             | AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)                                                                                                          |
| [s3-bucket-policy-grantee-check](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html)                                                   | S3 Buckets have access restricted to a list of provided AWS principals, federated users, service principals, IP addresses, or VPCs.                                                                                                                                                  | AC-3, AC-6, SC-7, SC-7(3)                                                                                                                          |
| [s3-bucket-ssl-requests-only](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html)                                                         | S3 Buckets only allow SSL requests.                                                                                                                                                                                                                                                  | AC-17(2), SC-7, SC-8, SC-8(1), SC-13                                                                                                               |
| [secretsmanager-scheduled-rotation-success-check](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-scheduled-rotation-success-check.html)                 | AWS Secrets Manager secret rotation has triggered/started successfully as per the configured rotation schedule.                                                                                                                                                                      | AC-2(1), AC-2(j)                                                                                                                                   |
| [securityhub-enabled](https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html)                                                                         | The AWS Account has Security Hub enabled.                                                                                                                                                                                                                                            | AC-17(1), AC-2(1), AC-2(12)(a), AC-2(4), AC-2(g), AU-6(1)(3), CA-7(a)(b), SA-10, SI-4(16), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)                |
| [vpc-flow-logs-enabled](https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html)                                                                     | Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.                                                                                                                                                                                                         | AU-2(a)(d), AU-3, AU-12(a)(c)                                                                                                                      |
| [vpc-sg-open-only-to-authorized-ports](https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html)                                       | The VPC Security Group restricts IPv4 TCP traffic on unauthorized ports.<sup>[2](#vpc-sg-open-only-to-authorized-ports)</sup>                                                                                                                                                        | AC-4, SC-7, SC-7(3)                                                                                                                                |
| [vpc-vpn-2-tunnels-up](https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html)                                                                       | At least two redundant Site-to-Site VPN tunnels are implemented.                                                                                                                                                                                                                     | CP-10                                                                                                                                              |
| [wafv2-logging-enabled](https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html)                                                                     | Logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control list (ACLs).                                                                                                                                                                       | AU-2(a)(d), AU-3, AU-12(a)(c), SC-7, SI-4(a)(b)(c)                                                                                                 |

## NIST 800-53 rev 5

The [Operational Best Practices for NIST 800-53 rev 5](https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_5.html) Conformance Pack offers a collection of AWS Config Rules that enable customers to align to a subset of the NIST 800-53 standards. The NIST 800-53 rev 5 Rules Pack implements many of the checks within the conformance pack.

### Warnings

| Rule ID                                                                                                                                         | Cause                                                                | Explanation                                                                                                                                                                                                                                                                                                        | Relevant Control ID(s)                                                                                                                                                                                            |
| ----------------------------------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ | ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| [NIST.800.53.R5-VPCDefaultSecurityGroupClosed](https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html) | The VPC's default security group allows inbound or outbound traffic. | Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources. | AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28) |

### Errors

| Rule ID                                                                                                                                                                            | Cause                                                                                                                                                                             | Explanation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Relevant Control ID(s)                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| [NIST.800.53.R5-ALBHttpToHttpsRedirection](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html)                                      | The ALB's HTTP listeners are not configured to redirect to HTTPS.                                                                                                                 | To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                            | AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                             |
| [NIST.800.53.R5-APIGWCacheEnabledAndEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html)                                   | The API Gateway stage does not have caching enabled and encrypted for all methods.                                                                                                | To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                               | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-APIGWExecutionLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html)                                      | The API Gateway stage does not have execution logging enabled for all methods.                                                                                                    | API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.                                                                                                                                                                                                                                                                                                                                                                                                             | AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)                                                                                                                                                                |
| [NIST.800.53.R5-APIGWSSLEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-ssl-enabled.html)                                                                 | The API Gateway REST API stage is not configured with SSL certificates.                                                                                                           | Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.                                                                                                                                                                                                                                                                                                                                                                                                            | AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                             |
| [NIST.800.53.R5-AutoscalingGroupELBHealthCheckRequired](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html)                  | The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks.                                                                              | The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.                                                                                                                                                             | AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, CM-6a, CM-9b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a                                                                                                                                                                                                                                                                                                                                                         |
| [NIST.800.53.R5-AutoscalingLaunchConfigPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-config-public-ip-disabled.html)               | The Auto Scaling launch configuration does not have public IP addresses disabled.                                                                                                 | If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.                                                                                                                                                                                                                                                                                          | AC-3, AC-4(21), CM-6a, SC-7(3)                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| [NIST.800.53.R5-CloudTrailCloudWatchLogsEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html)                               | The trail does not have CloudWatch logs enabled.                                                                                                                                  | Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.                                                                                                                                                                                                                                                                                                                                                                                                    | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-4(1), AU-6(1), AU-6(3), AU-6(4), AU-6(5), AU-6(6), AU-6(9), AU-7(1), AU-8b, AU-9(7), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), AU-16, CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c) |
| [NIST.800.53.R5-CloudTrailEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html)                                         | The trail does not have encryption enabled.                                                                                                                                       | Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.                                                                                                                                                                                                                                                                                                                                                                                                                                              | AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                              |
| [NIST.800.53.R5-CloudTrailLogFileValidationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html)                         | The trail does not have log file validation enabled.                                                                                                                              | Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.                                                                                                                              | AU-9a, CM-6a, CM-9b, PM-11b, PM-17b, SA-1(1), SA-10(1), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-4d, SI-7a, SI-7(1), SI-7(3), SI-7(7)                                                                                                                                                                                                                                                                                                                                    |
| [NIST.800.53.R5-CloudWatchAlarmAction](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html)                                                | The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled.                                                             | Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.                                                                                                                                                                                                                                                                                                               | AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a, SI-4(12), SI-5b, SI-5(1)                                                                                                                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-CloudWatchLogGroupEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html)                                         | The CloudWatch Log Group is not encrypted with an AWS KMS key.                                                                                                                    | To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-CloudWatchLogGroupRetentionPeriod](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                               | The CloudWatch Log Group does not have an explicit retention period configured.                                                                                                   | Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.                                                                                                                                                                                                                                                                                                                | AC-16b, AT-4b, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-10, AU-11(1), AU-11, AU-12(1), AU-12(2), AU-12(3), AU-14a, AU-14b, CA-7b, PM-14a.1, PM-14b, PM-21b, PM-31, SC-28(2), SI-4(17), SI-12                                                                                                                                                                                                                                                                              |
| [NIST.800.53.R5-DMSReplicationNotPublic](https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html)                                                 | The DMS replication instance is public.                                                                                                                                           | DMS replication instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                                | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |
| [NIST.800.53.R5-DynamoDBPITREnabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html)                                                          | The DynamoDB table does not have Point-in-time Recovery enabled.                                                                                                                  | The recovery maintains continuous backups of your table for the last 35 days.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | CP-1(2), CP-2(5), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                      |
| [NIST.800.53.R5-EC2EBSOptimizedInstance](https://docs.aws.amazon.com/config/latest/developerguide/ebs-optimized-instance.html)                                                     | The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled.                                                                                     | An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.                                                                                                                                                                                                                                                      | CP-2(5), CP-9a, CP-9b, CP-9c, CP-10, SC-5(2)                                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-EC2InstanceNoPublicIp](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html)                                                    | The EC2 instance is associated with a public IP address.                                                                                                                          | Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                         | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |
| [NIST.800.53.R5-EC2InstanceProfileAttached](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-profile-attached.html)                                           | The EC2 instance does not have an instance profile attached.                                                                                                                      | EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.                                                                                                                                                                                                                                                                                                                                                                                                       | AC-3, CM-5(1)(a), CM-6a                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-EC2InstancesInVPC](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html)                                                             | The EC2 instance is not within a VPC.                                                                                                                                             | Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access. | AC-2(6), AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25                                                                                                                                                                                                                                           |
| [NIST.800.53.R5-EC2RestrictedCommonPorts](https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html)                                                   | The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).                        | Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.                                                                                                                                                                                                                                                                                                                            | AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-2a, CM-2(2), CM-6a, CM-7b, CM-8(6), CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)                                                                                                                                                                                                                       |
| [NIST.800.53.R5-EC2RestrictedSSH](https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html)                                                                    | The Security Group allows unrestricted SSH access.                                                                                                                                | Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.                                                                                                                                                                                                                                                                                                                                                                                                                                                | AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-9b, SC-7a, SC-7c, SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)                                                                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-ECSTaskDefinitionUserForHostMode](https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-user-for-host-mode-check.html)                      | The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty. | If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.                                                                                                                                                                                                                                                                                         | AC-3, AC-5b, CM-5(1)(a)                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-EFSEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html)                                                                   | The EFS does not have encryption at rest enabled.                                                                                                                                 | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).                                                                                                                                                                                                                                                                                                                                                                                                                                   | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-ElastiCacheRedisClusterAutomaticBackup](https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html)            | The ElastiCache Redis cluster does not retain automatic backups for at least 15 days.                                                                                             | Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.                                                                                                                                                                                                                                                                                                                                                                                                                 | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                      |
| [NIST.800.53.R5-ElasticBeanstalkEnhancedHealthReportingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/beanstalk-enhanced-health-reporting-enabled.html)         | The Elastic Beanstalk environment does not have enhanced health reporting enabled.                                                                                                | AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.                                                                                                                                                                                       | AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a                                                                                                                                                                                                                                                                                                                                                                       |
| [NIST.800.53.R5-ElasticBeanstalkManagedUpdatesEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elastic-beanstalk-managed-updates-enabled.html)                    | The Elastic Beanstalk environment does not have managed updates enabled.                                                                                                          | Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.                                                                                                                                                                                                                                                                                                  | SI-2c, SI-2d, SI-2(2), SI-2(5)                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| [NIST.800.53.R5-ELBACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html)                                             | The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                                         | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SC-23(5), SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                   |
| [NIST.800.53.R5-ELBCrossZoneLoadBalancingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html)                             | The CLB does not balance traffic between at least 2 Availability Zones.                                                                                                           | Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.                                                                                                                                                                                                                          | CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)                                                                                                                                                                                                                                                                                                                                     |
| [NIST.800.53.R5-ELBDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html)                                       | The ALB, NLB, or GLB does not have deletion protection enabled.                                                                                                                   | This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                                                                                                                                  | CA-7(4)(c), CM-2a, CM-2(2), CM-3a, CM-8(6), CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22                                                                                                                                                                                                                                                                                                                               |
| [NIST.800.53.R5-ELBLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html)                                                              | The ELB does not have logging enabled.                                                                                                                                            | Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client\'s IP address, latencies, request paths, and server responses.                                                                                                                                                                                                                | AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)                                                                                                                                                                |
| [NIST.800.53.R5-ELBTlsHttpsListenersOnly](https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html)                                              | The CLB does not restrict its listeners to only the SSL and HTTPS protocols.                                                                                                      | Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                         | AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1a.2, SI-1a.2, SI-1c.2, SI-1c.2                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-ELBv2ACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elbv2-acm-certificate-required.html)                                         | The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                   | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | SC-8(1), SC-23(5)                                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| [NIST.800.53.R5-IAMNoInlinePolicy](https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html)                                                       | The IAM Group, User, or Role contains an inline policy.                                                                                                                           | AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.                                                                                                                                                                                                                                                                                                                                                                                               | AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)                                                                                                           |
| [NIST.800.53.R5-IAMPolicyNoStatementsWithAdminAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html)                    | The IAM policy grants admin access.                                                                                                                                               | AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.                                                                                                                                                                                                                | AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-5b, AC-6, AC-6(2), AC-6(3), AC-6(10), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25                                                                          |
| [NIST.800.53.R5-IAMPolicyNoStatementsWithFullAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-full-access.html)                      | The IAM policy grants full access.                                                                                                                                                | Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                                                                                                        | AC-3, AC-5b, AC-6(2), AC-6(10), CM-5(1)(a)                                                                                                                                                                                                                                                                                                                                                                                                                              |
| [NIST.800.53.R5-IAMUserGroupMembership](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html)                                             | The IAM user does not belong to any group(s).                                                                                                                                     | AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                           | AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)                                                                                                           |
| [NIST.800.53.R5-IAMUserNoPolicies](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html)                                                       | The IAM policy is attached at the user level.                                                                                                                                     | Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.                                                                                                                                                                                                                                                                                                                                                                                                                                   | AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25                                                                                                    |
| [NIST.800.53.R5-LambdaConcurrency](https://docs.aws.amazon.com/config/latest/developerguide/lambda-concurrency-check.html)                                                         | The Lambda function is not configured with function-level concurrent execution limits.                                                                                            | Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.                                                                                                                                                                                                                                                                                                                                                                                       | AU-12(3), AU-14a, AU-14b, CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6                                                                                                                                                                                                                                                                                                                                                                                                    |
| [NIST.800.53.R5-LambdaDlq](https://docs.aws.amazon.com/config/latest/developerguide/lambda-dlq-check.html)                                                                         | The Lambda function is not configured with a dead-letter configuration.                                                                                                           | Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.                                                                                                                                                                                                                                                                                                                                                                                                             | AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a                                                                                                                                                                                                                                                                                                                                                                             |
| [NIST.800.53.R5-LambdaInsideVPC](https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html)                                                                  | The Lambda function is not VPC enabled.                                                                                                                                           | Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.                                                                                                                                                                                                                                                                                                                                                                                                             | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25                                                                                                                                                                                                                                     |
| [NIST.800.53.R5-OpenSearchEncryptedAtRest](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html)                                          | The OpenSearch Service domain does not have encryption at rest enabled.                                                                                                           | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                             | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-OpenSearchInVPCOnly](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html)                                                      | The OpenSearch Service domain is not running within a VPC.                                                                                                                        | VPCs help secure your AWS resources and provide an extra layer of protection.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25                                                                                                                                                                                                                                     |
| [NIST.800.53.R5-OpenSearchLogsToCloudWatch](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-logs-to-cloudwatch.html)                                        | The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs.                                                                                | Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.                                                                                                                                                                                                                                                                                                                              | AU-10                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| [NIST.800.53.R5-OpenSearchNodeToNodeEncryption](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html)                         | The OpenSearch Service domain does not have node-to-node encryption enabled.                                                                                                      | Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                                 | AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                                                   |
| [NIST.800.53.R5-RDSEnhancedMonitoringEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html)                                       | The RDS DB Instance does not enhanced monitoring enabled.                                                                                                                         | Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.                                                                                                                                                                                                                                                                                                                                                                                                                 | AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a                                                                                                                                                                                                                                                                                                                                                                             |
| [NIST.800.53.R5-RDSInstanceBackupEnabled](https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html)                                                | The RDS DB Instance does not have backup enabled.                                                                                                                                 | The backup feature of Amazon RDS creates backups of your databases and transaction logs.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                      |
| [NIST.800.53.R5-RDSInstanceDeletionProtectionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html)                      | The RDS DB Instance or Aurora Cluster does not have deletion protection enabled.                                                                                                  | Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.                                                                                                                                                                                                                                                                   | CA-7(4)(c), CM-3a, CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22, SI-13(5)                                                                                                                                                                                                                                                                                                                                              |
| [NIST.800.53.R5-RDSInstancePublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html)                                           | The RDS DB Instance allows public access.                                                                                                                                         | Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                            | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |
| [NIST.800.53.R5-RDSLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html)                                                              | The RDS DB Instance does not have all CloudWatch log types exported.                                                                                                              | To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.                                                                                                                                                                                                                                                                                                                     | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)                                                                   |
| [NIST.800.53.R5-RDSMultiAZSupport](https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html)                                                             | The RDS DB Instance does not have multi-AZ support.                                                                                                                               | Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.                                        | CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)                                                                                                                                                                                                                                                                                                                                     |
| [NIST.800.53.R5-RDSStorageEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html)                                                          | The RDS DB Instance or Aurora Cluster does not have storage encrypted.                                                                                                            | Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                       | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-RedshiftBackupEnabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-backup-enabled.html)                                                      | The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days.                                                              | To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.                                                                                                                                                                                                                                     | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                      |
| [NIST.800.53.R5-RedshiftClusterConfiguration](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html)                                  | The Redshift cluster does not have encryption or audit logging enabled.                                                                                                           | To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.                                                                                                                                                                                                                                                                  | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-9(3), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CP-9d, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c), SI-19(4)     |
| [NIST.800.53.R5-RedshiftClusterMaintenanceSettings](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html)                      | The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured                          | Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                                | CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-3(3), CP-9a, CP-9b, CP-9c, SC-5(2), SI-2c, SI-2d, SI-2(2), SI-2(5)                                                                                                                                                                                                                                                                                                                                                                 |
| [NIST.800.53.R5-RedshiftClusterPublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html)                                   | The Redshift cluster allows public access.                                                                                                                                        | Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                  | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |
| [NIST.800.53.R5-RedshiftEnhancedVPCRoutingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-enhanced-vpc-routing-enabled.html)                            | The Redshift cluster does not have enhanced VPC routing enabled.                                                                                                                  | Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.                                                                                                                                                                                                                                                                | AC-4(21), SC-7b                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| [NIST.800.53.R5-S3BucketLevelPublicAccessProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-level-public-access-prohibited.html)                       | The S3 bucket does not prohibit public access through bucket level settings.                                                                                                      | Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                        |
| [NIST.800.53.R5-S3BucketLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html)                                                   | The S3 Buckets does not have server access logs enabled.                                                                                                                          | Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.                                                                                                             | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)                                                     |
| [NIST.800.53.R5-S3BucketPublicReadProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html)                                      | The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.                                                                | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                      |
| [NIST.800.53.R5-S3BucketPublicWriteProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html)                                    | The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.                                                               | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                      |
| [NIST.800.53.R5-S3BucketReplicationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html)                                           | The S3 Bucket does not have replication enabled.                                                                                                                                  | Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.                                                                                                                                                                                                                                                                                                       | AU-9(2), CM-6a, CM-9b, CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                               |
| [NIST.800.53.R5-S3BucketServerSideEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html)                       | The S3 Bucket does not have default server-side encryption enabled.                                                                                                               | Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | AU-9(3), CM-6a, CM-9b, CP-9d, CP-9(8), PM-11b, SC-8(3), SC-8(4), SC-13a, SC-16(1), SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                   |
| [NIST.800.53.R5-S3BucketVersioningEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html)                                             | The S3 Bucket does not have versioning enabled.                                                                                                                                   | Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.                                                                                                                                                                                                                                                                                                                                                               | AU-9(2), CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), PM-11b, PM-17b, SC-5(2), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-13(5)                                                                                                                                                                                                                                                                                                        |
| [NIST.800.53.R5-S3DefaultEncryptionKMS](https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html)                                                   | The S3 Bucket is not encrypted with a KMS Key by default.                                                                                                                         | Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                          | AU-9(3), CP-9d, CP-9(8), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                   |
| [NIST.800.53.R5-SageMakerEndpointConfigurationKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html) | The SageMaker resource endpoint is not encrypted with a KMS key.                                                                                                                  | Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-SageMakerNotebookInstanceKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html)           | The SageMaker notebook is not encrypted with a KMS key.                                                                                                                           | Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-SageMakerNotebookNoDirectInternetAccess](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html)               | The SageMaker notebook does not disable direct internet access.                                                                                                                   | By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |
| [NIST.800.53.R5-SecretsManagerUsingKMSKey](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-using-cmk.html)                                                 | The secret is not encrypted with a KMS Customer managed key.                                                                                                                      | To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                              | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                            |
| [NIST.800.53.R5-SNSEncryptedKMS](https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html)                                                                  | The SNS topic does not have KMS encryption enabled.                                                                                                                               | To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                     | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1)                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [NIST.800.53.R5-VPCNoUnrestrictedRouteToIGW](https://docs.aws.amazon.com/config/latest/developerguide/no-unrestricted-route-to-igw.html)                                           | The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').                                                                                  | Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.                                                                                                                                                                                                                                                                                                                                                | AC-4(21), CM-7b                                                                                                                                                                                                                                                                                                                                                                                                                                                         |
| [NIST.800.53.R5-VPCSubnetAutoAssignPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/subnet-auto-assign-public-ip-disabled.html)                          | The subnet auto-assigns public IP addresses.                                                                                                                                      | Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.                                                                                                                                                                                                                                                 | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                    |

### Excluded Rules

Unimplemented rules from the AWS NIST 800-53 rev 5 Conformance Pack

| AWS Config Rule Name                                                                                                                                                             | Summary                                                                                                                           | Relevant Control ID(s)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  |
| -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| [access-keys-rotated](https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html)                                                                         | IAM active access keys are rotated within the number of days specified in maxAccessKeyAge.                                        | AC-2(1), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-24, CM-5(1)(a), CM-6a, CM-9b, SC-23(3)                                                                                                                                                                                                                                 |
| [account-part-of-organizations](https://docs.aws.amazon.com/config/latest/developerguide/account-part-of-organizations.html)                                                     | The AWS Account is part of AWS Organizations.                                                                                     | CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-3(3), CM-6a, CM-9b                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| [acm-certificate-expiration-check](https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html)                                               | AWS Certificate Manager Certificates in your account are valid and not marked for expiration within the specified number of days. | SC-7(12), SC-7(16)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [alb-waf-enabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html)                                                                                 | Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).                                                   | AC-4(21)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| [api-gw-associated-with-waf](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-associated-with-waf.html)                                                           | Web Application Firewall (WAF) is enabled on API Gateway stages.                                                                  | AC-4(21)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                |
| [cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html)                                                                           | The AWS Account has CloudTrail enabled.                                                                                           | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-3(1), AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)                                                                                                                          |
| [cloudtrail-s3-dataevents-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html)                                               | The AWS Account has at least once AWS CloudTrail that logs Amazon S3 data events for all S3 buckets.                              | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)                                                                                                                     |
| [cmk-backing-key-rotation-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html)                                               | KMS backing keys have key rotation enabled and match to the key ID of the customer managed key.                                   | CM-6a, CM-9b, SA-9(6), SC-12, SC-12(2), SC-12(6)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |
| [cw-loggroup-retention-period-check](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                                           | CloudWatch LogGroup retention period is set to specific number of days and is greater than the configured retention period.       | AC-16b, AT-4b, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-10, AU-11(1), AU-11, AU-12(1), AU-12(2), AU-12(3), AU-14a, AU-14b, CA-7b, PM-14a.1, PM-14b, PM-21b, PM-31, SC-28(2), SI-4(17), SI-12                                                                                                                                                                                                                                                                                                                                              |
| [dynamodb-autoscaling-enabled](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html)                                                       | DynamoDB has autoscaling enabled.                                                                                                 | CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                     |
| [dynamodb-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html)                                                                 | DynamoDB tables are in an AWS Backup Plan.                                                                                        | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [dynamodb-table-encrypted-kms](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html)                                                       | DynamoDB tables are encrypted in KMS.<sup>[1](#dynamodb-table-encrypted-kms)</sup>                                                | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| [dynamodb-throughput-limit-check](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-throughput-limit-check.html)                                                 | DynamoDB throughput is approaching the maximum limit for the AWS Account.                                                         | AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a                                                                                                                                                                                                                                                                                                                                                                                                                                       |
| [ebs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html)                                                                           | EBS volumes are in an AWS Backup Plan.                                                                                            | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [ebs-snapshot-public-restorable-check](https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html)                                       | EBS snapshots can be publicly restored.                                                                                           | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                                                                    |
| [ec2-ebs-encryption-by-default](https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html)                                                     | Default encryption for EBS volumes is enabled at the AWS Account level.                                                           | AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| [ec2-imdsv2-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html)                                                                               | EC2 instances have IMDSV2 (instance metadata service version 2) enabled.                                                          | AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-24, CM-5(1)(a), MP-2, SC-23(3)                                                                                                                                                                                                                    |
| [ec2-instance-managed-by-systems-manager](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html)                                 | EC2 instances are managed by Systems Manager.                                                                                     | CM-2a, CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-2(2), CM-3(3), CM-8a, CM-8a.1, CM-8a.2, CM-8a.3, CM-8a.4, CM-8a.5, CM-8b, CM-8(1), CM-8(2), CM-8(3)(a), CM-8(6), SI-3c.2                                                                                                                                                                                                                                                                                                                                                                    |
| [ec2-managedinstance-association-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html) | Managed EC2 instances are compliant with their association's standards.                                                           | CM-2a, CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-2(2), CM-3(3), CM-8a, CM-8a.1, CM-8a.2, CM-8a.3, CM-8a.4, CM-8a.5, CM-8b, CM-8(1), CM-8(3)(a), CM-8(6), SI-3c.2                                                                                                                                                                                                                                                                                                                                                                             |
| [ec2-managedinstance-patch-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html)             | EC2 instances are compliant with their patch requirements.                                                                        | CM-8(3)(a), RA-3a.1, RA-3a.1, SI-2c, SI-2d, SI-2(2), SI-2(5), SI-3c.2                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| [ec2-stopped-instance](https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html)                                                                       | EC2 instances have not been stopped for more than the allowed number of days.                                                     | CM-2a, CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-2(2), CM-3(3), CM-8(6)                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [ec2-volume-inuse-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html)                                                                   | When an instances is terminated, its associated EBS volumes are marked for deletion.                                              | CM-2a, CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-2(2), CM-3(3), CM-8(6)                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [efs-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html)                                                                           | EFSs are in an AWS Backup plan.                                                                                                   | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [emr-master-no-public-ip](https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html)                                                                 | EMR clusters' master nodes have no public IP.                                                                                     | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                                                                    |
| [encrypted-volumes](https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html)                                                                             | EC2 instances are only attached to encrypted EBS volumes.                                                                         | AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                                                                              |
| [guardduty-enabled-centralized](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html)                                                     | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                         | AC-2(12)(a), AC-3(12)(b), AU-3(1), AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2d, CA-7, CA-7b, CM-8(3)(a), IR-4a, PE-6(2), PE-6(4), PM-14a.1, PM-14b, PM-16, PM-31, RA-1a, RA-1a.1, RA-1a.2, RA-3a.1, RA-3a.1, RA-3(4), RA-5a, RA-5(4), RA-10a, RA-10a.1, RA-10a.2, SC-5a, SC-5b, SC-5(1), SC-5(3)(a), SC-5(3)(b), SC-43b, SI-3(8)(a), SI-4a, SI-4a.1, SI-4a.1, SI-4a.2, SI-4b, SI-4c, SI-4(1), SI-4(1), SI-4(2), SI-4(3), SI-4(4)(a), SI-4(4)(b), SI-4(10), SI-4(13)(a), SI-4(14), SI-4(14), SI-4(23), SI-4(25), SI-5b, SI-5(1)    |
| [iam-password-policy](https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html)                                                                         | The account password policy for IAM users meets the specified requirements indicated in the parameters.                           | AC-2d.1, AC-2(1), AC-2(3)(a), AC-2(3)(b), AC-2(3)(c), AC-2(3)(d), AC-2(3), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-7(4), AC-7(4)(a), AC-24, CM-5(1)(a), CM-6a, CM-9b, CM-12b, IA-4d, IA-5, IA-5b, IA-5c, IA-5d, IA-5f, IA-5h, IA-5(1)(f), IA-5(1)(g), IA-5(1)(h), IA-5(18)(a), IA-5(18)(b), IA-8(2)(b), MA-4c, SC-23(3) |
| [iam-root-access-key-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html)                                                             | The Account IAM Root User has an access key(s).                                                                                   | AC-2(1), AC-2(6), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(2), AC-6(10), AC-24, CM-5(1)(a), CM-6a, CM-6a, CM-9b, IA-2, IA-4b, IA-4(4), IA-4(8), IA-5(8), MP-2, SC-23(3), SC-25                                                                                                                          |
| [iam-user-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html)                                                                       | AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.                                          | AC-2(1), AC-3(2), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-7(4), AC-7(4)(a), AC-24, CM-5(1)(a), IA-2(1), IA-2(2), IA-2(6), IA-2(6), IA-2(6)(a), IA-2(8), SC-23(3)                                                                                                                                                        |
| [iam-user-unused-credentials-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html)                                             | IAM User passwords and active access keys have been used within a specified number of days.                                       | AC-2g, AC-2j, AC-2j, AC-2(1), AC-2(3)(a), AC-2(3)(b), AC-2(3)(c), AC-2(3)(d), AC-2(3), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)                                                                                                               |
| [kms-cmk-not-scheduled-for-deletion](https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html)                                           | AWS KMS keys are not scheduled for deletion in AWS Key Management Service (KMS).                                                  | SA-9(6), SC-12, SC-12(2), SC-12(6)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [lambda-function-public-access-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html)                               | Public access to lambda functions is prohibited.                                                                                  | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                                                                    |
| [mfa-enabled-for-iam-console-access](https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html)                                           | IAM Users have MFA enabled for console access.                                                                                    | AC-2(1), AC-3(2), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-7(4), AC-7(4)(a), AC-24, CM-5(1)(a), CM-6a, CM-9b, IA-2(1), IA-2(2), IA-2(6), IA-2(6), IA-2(6)(a), IA-2(8), SC-23(3)                                                                                                                                          |
| [multi-region-cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html)                                                 | The AWS Account has at least one multi-region CloudTrail enabled.                                                                 | AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)                                                                                                                     |
| [rds-in-backup-plan](https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html)                                                                           | RDS DB instances are in an AWS Backup Plan.                                                                                       | CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                                                      |
| [rds-snapshot-encrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html)                                                                   | RDS snapshots are encrypted.                                                                                                      | AU-9(3), CP-9d, CP-9(8), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                                                                                   |
| [rds-snapshots-public-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html)                                                 | RDS snapshots are not public.                                                                                                     | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                                                                    |
| [redshift-cluster-kms-enabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-kms-enabled.html)                                                       | Redshift clusters are encrypted with one of the specified KMS keys.                                                               | AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |
| [redshift-require-tls-ssl](https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html)                                                               | Redshift clusters require TLS/SSL encryption.                                                                                     | AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                                                                                                                   |
| [root-account-hardware-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html)                                             | The AWS Account root user is hardware MFA enabled.                                                                                | AC-2(1), AC-3(2), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-7(4), AC-7(4)(a), AC-24, CM-5(1)(a), CM-6a, CM-9b, IA-2(1), IA-2(2), IA-2(6), IA-2(6), IA-2(6)(a), IA-2(8), SC-23(3)                                                                                                                                          |
| [root-account-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html)                                                               | The AWS Account root user is MFA enabled.                                                                                         | AC-2(1), AC-3(2), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-7(4), AC-7(4)(a), AC-24, CM-5(1)(a), CM-6a, CM-9b, IA-2(1), IA-2(2), IA-2(6), IA-2(6), IA-2(6)(a), IA-2(8), SC-23(3)                                                                                                                                          |
| [s3-account-level-public-access-blocks-periodic](https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks-periodic.html)                   | The AWS Account blocks S3 public access.                                                                                          | AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25                                                                                                                                                                                                                                      |
| [s3-bucket-ssl-requests-only](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html)                                                         | S3 Buckets only allow SSL requests.                                                                                               | AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, CM-6a, CM-9b, IA-5(1)(c), PM-11b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-16(1), SC-23, SI-1a.2, SI-1a.2, SI-1c.2                                                                                                                                                                                                                                                                                                             |
| [secretsmanager-rotation-enabled-check](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-rotation-enabled-check.html)                                     | Secrets have rotation enabled.                                                                                                    | AC-2(1), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-24, CM-5(1)(a), SC-23(3)                                                                                                                                                                                                                                               |
| [secretsmanager-secret-periodic-rotation](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-periodic-rotation.html)                                 | All secrets have been rotated in the past 90 days.                                                                                | AC-2(1), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-24, CM-5(1)(a), SC-23(3)                                                                                                                                                                                                                                               |
| [secretsmanager-secret-unused](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-secret-unused.html)                                                       | All secrets have been accessed in the past 90 days.                                                                               | AC-2(1), AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-24, CM-5(1)(a), SC-23(3)                                                                                                                                                                                                                                               |
| [securityhub-enabled](https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html)                                                                         | The AWS Account has Security Hub enabled.                                                                                         | AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2d, CA-7, CA-7b, PM-14a.1, PM-14b, PM-31                                                                                                                                                                                                                                                                                                                                                                                                                                                 |
| [ssm-document-not-public](https://docs.aws.amazon.com/config/latest/developerguide/ssm-document-not-public.html)                                                                 | SSM documents are not public.                                                                                                     | AC-3, AC-4(21), CM-6a, SC-7(3)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |
| [vpc-flow-logs-enabled](https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html)                                                                     | Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.                                                      | AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SI-4(17), SI-7(8)                                                                                                                                                                                                                                            |
| [vpc-sg-open-only-to-authorized-ports](https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html)                                       | The VPC Security Group restricts IPv4 TCP traffic on unauthorized ports.<sup>[2](#vpc-sg-open-only-to-authorized-ports)</sup>     | AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), SC-7a, SC-7c, SC-7(5), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b)                                                                                                                                                                                                                                                                                                                                                                                      |
| [vpc-vpn-2-tunnels-up](https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html)                                                                       | At least two redundant Site-to-Site VPN tunnels are implemented.                                                                  | CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)                                                                                                                                                                                                                                                                                                                                                                                                     |
| [wafv2-logging-enabled](https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html)                                                                     | Logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control list (ACLs).                    | AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)                                                                                                                                                                                                                                |

## PCI DSS 3.2.1

The [Operational Best Practices for PCI DSS 3.2.1](https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-pci-dss.html) Conformance Pack offers a collection of AWS Config Rules that enable customers to align to a subset of the Payment Card Industry Data Security Standard (PCI DSS) standards. The PCI DSS 3.2.1 Rules Pack implements many of the checks within the conformance pack.

### Warnings

| Rule ID                                                                                                                                      | Cause                                                                | Explanation                                                                                                                                                                                                                                                                                                        | Relevant Control ID(s)                         |
| -------------------------------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------ | ---------------------------------------------- |
| [PCI.DSS.321-VPCDefaultSecurityGroupClosed](https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html) | The VPC's default security group allows inbound or outbound traffic. | Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources. | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.1, 2.2, 2.2.2 |

### Errors

| Rule ID                                                                                                                                                                         | Cause                                                                                                                                                                             | Explanation                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | Relevant Control ID(s)                                                                                    |
| ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------- |
| [PCI.DSS.321-ALBHttpToHttpsRedirection](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html)                                      | The ALB's HTTP listeners are not configured to redirect to HTTPS.                                                                                                                 | To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                            | 2.3, 4.1, 8.2.1                                                                                           |
| [PCI.DSS.321-APIGWCacheEnabledAndEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html)                                   | The API Gateway stage does not have caching enabled and encrypted for all methods.                                                                                                | To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                               | 3.4                                                                                                       |
| [PCI.DSS.321-APIGWExecutionLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html)                                      | The API Gateway stage does not have execution logging enabled for all methods.                                                                                                    | API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.                                                                                                                                                                                                                                                                                                                                                                                                             | 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4                                              |
| [PCI.DSS.321-APIGWSSLEnabled](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-ssl-enabled.html)                                                                 | The API Gateway REST API stage is not configured with SSL certificates.                                                                                                           | Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.                                                                                                                                                                                                                                                                                                                                                                                                            | 2.3, 4.1, 8.2.1                                                                                           |
| [PCI.DSS.321-AutoscalingGroupELBHealthCheckRequired](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html)                  | The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks.                                                                              | The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.                                                                                                                                                             | 2.2                                                                                                       |
| [PCI.DSS.321-AutoscalingLaunchConfigPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-launch-config-public-ip-disabled.html)               | The Auto Scaling launch configuration does not have public IP addresses disabled.                                                                                                 | If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.                                                                                                                                                                                                                                                                                          | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-CloudTrailCloudWatchLogsEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html)                               | The trail does not have CloudWatch logs enabled.                                                                                                                                  | Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.                                                                                                                                                                                                                                                                                                                                                                                                    | 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.3, 10.5.4 |
| [PCI.DSS.321-CloudTrailEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html)                                         | The trail does not have encryption enabled.                                                                                                                                       | Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.                                                                                                                                                                                                                                                                                                                                                                                                                                              | 2.2, 3.4, 10.5                                                                                            |
| [PCI.DSS.321-CloudTrailLogFileValidationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html)                         | The trail does not have log file validation enabled.                                                                                                                              | Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.                                                                                                                              | 2.2, 10.5.2, 10.5, 10.5.5, 11.5                                                                           |
| [PCI.DSS.321-CloudWatchLogGroupEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html)                                         | The CloudWatch Log Group is not encrypted with an AWS KMS key.                                                                                                                    | To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 3.4                                                                                                       |
| [PCI.DSS.321-CloudWatchLogGroupRetentionPeriod](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                               | The CloudWatch Log Group does not have an explicit retention period configured.                                                                                                   | Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.                                                                                                                                                                                                                                                                                                                | 3.1, 10.7                                                                                                 |
| [PCI.DSS.321-DMSReplicationNotPublic](https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html)                                                 | The DMS replication instance is public.                                                                                                                                           | DMS replication instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-EC2InstanceNoPublicIp](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html)                                                    | The EC2 instance is associated with a public IP address.                                                                                                                          | Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                         | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-EC2InstanceProfileAttached](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-profile-attached.html)                                           | The EC2 instance does not have an instance profile attached.                                                                                                                      | EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.                                                                                                                                                                                                                                                                                                                                                                                                       | 2.2, 7.1.1, 7.2.1                                                                                         |
| [PCI.DSS.321-EC2InstancesInVPC](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html)                                                             | The EC2 instance is not within a VPC.                                                                                                                                             | Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access. | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-EC2RestrictedCommonPorts](https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html)                                                   | The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).                        | Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.                                                                                                                                                                                                                                                                                                                            | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2                                                                 |
| [PCI.DSS.321-EC2RestrictedSsh](https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html)                                                                    | The Security Group allows unrestricted SSH access.                                                                                                                                | Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.                                                                                                                                                                                                                                                                                                                                                                                                                                                | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2                                                                 |
| [PCI.DSS.321-ECSTaskDefinitionUserForHostMode](https://docs.aws.amazon.com/config/latest/developerguide/ecs-task-definition-user-for-host-mode-check.html)                      | The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty. | If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.                                                                                                                                                                                                                                                                                         | 7.2.1                                                                                                     |
| [PCI.DSS.321-EFSEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html)                                                                   | The EFS does not have encryption at rest enabled.                                                                                                                                 | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).                                                                                                                                                                                                                                                                                                                                                                                                                                   | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-ELBACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html)                                             | The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                                         | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | 4.1, 8.2.1                                                                                                |
| [PCI.DSS.321-ELBLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html)                                                              | The ELB does not have logging enabled.                                                                                                                                            | Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client\'s IP address, latencies, request paths, and server responses.                                                                                                                                                                                                                | 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4                                              |
| [PCI.DSS.321-ELBTlsHttpsListenersOnly](https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html)                                              | The CLB does not restrict its listeners to only the SSL and HTTPS protocols.                                                                                                      | Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                         | 2.3, 4.1, 8.2.1                                                                                           |
| [PCI.DSS.321-ELBv2ACMCertificateRequired](https://docs.aws.amazon.com/config/latest/developerguide/elbv2-acm-certificate-required.html)                                         | The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                   | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | 4.1                                                                                                       |
| [PCI.DSS.321-EMRKerberosEnabled](https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html)                                                            | The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager).                                                                   | Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.                                                                                                                                                                                                                                                                                            | 7.2.1                                                                                                     |
| [PCI.DSS.321-IAMNoInlinePolicy](https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html)                                                       | The IAM Group, User, or Role contains an inline policy.                                                                                                                           | AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.                                                                                                                                                                                                                                                                                                                                                                                               | 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                           |
| [PCI.DSS.321-IAMPolicyNoStatementsWithAdminAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html)                    | The IAM policy grants admin access.                                                                                                                                               | AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.                                                                                                                                                                                                                | 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                           |
| [PCI.DSS.321-IAMPolicyNoStatementsWithFullAccess](https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-full-access.html)                      | The IAM policy grants full access.                                                                                                                                                | Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                                                                                                        | 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                                |
| [PCI.DSS.321-IAMUserGroupMembership](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html)                                             | The IAM user does not belong to any group(s).                                                                                                                                     | AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.                                                                                                                                                                                                                                                                           | 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                           |
| [PCI.DSS.321-IAMUserNoPolicies](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html)                                                       | The IAM policy is attached at the user level.                                                                                                                                     | Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.                                                                                                                                                                                                                                                                                                                                                                                                                                   | 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                           |
| [PCI.DSS.321-LambdaInsideVPC](https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html)                                                                  | The Lambda function is not VPC enabled.                                                                                                                                           | Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.                                                                                                                                                                                                                                                                                                                                                                                                             | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 2.2.2                                                               |
| [PCI.DSS.321-OpenSearchEncryptedAtRest](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html)                                          | The OpenSearch Service domain does not have encryption at rest enabled.                                                                                                           | Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                             | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-OpenSearchInVPCOnly](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html)                                                      | The OpenSearch Service domain is not running within a VPC.                                                                                                                        | VPCs help secure your AWS resources and provide an extra layer of protection.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-OpenSearchLogsToCloudWatch](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-logs-to-cloudwatch.html)                                        | The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs.                                                                                | Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.                                                                                                                                                                                                                                                                                                                              | 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6              |
| [PCI.DSS.321-OpenSearchNodeToNodeEncryption](https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html)                         | The OpenSearch Service domain does not have node-to-node encryption enabled.                                                                                                      | Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.                                                                                                                                                                                                                                                                                                                                                                                                                 | 4.1                                                                                                       |
| [PCI.DSS.321-RDSAutomaticMinorVersionUpgradeEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-automatic-minor-version-upgrade-enabled.html)                 | The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').                                                                                  | Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.                                                                                                                                                                                                                                                                                                                                                | 6.2                                                                                                       |
| [PCI.DSS.321-RDSInstancePublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html)                                           | The RDS DB Instance allows public access.                                                                                                                                         | Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                            | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-RDSLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html)                                                              | The RDS DB Instance does not have all CloudWatch log types exported.                                                                                                              | To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.                                                                                                                                                                                                                                                                                                                     | 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6              |
| [PCI.DSS.321-RDSStorageEncrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html)                                                          | The RDS DB Instance or Aurora Cluster does not have storage encrypted.                                                                                                            | Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                       | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-RedshiftClusterConfiguration](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html)                                  | The Redshift cluster does not have encryption or audit logging enabled.                                                                                                           | To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.                                                                                                                                                                                                                                                                  | 3.4, 8.2.1, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6  |
| [PCI.DSS.321-RedshiftClusterMaintenanceSettings](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html)                      | The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured                          | Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                                | 6.2                                                                                                       |
| [PCI.DSS.321-RedshiftClusterPublicAccess](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html)                                   | The Redshift cluster allows public access.                                                                                                                                        | Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-RedshiftEnhancedVPCRoutingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-enhanced-vpc-routing-enabled.html)                            | The Redshift cluster does not have enhanced VPC routing enabled.                                                                                                                  | Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.                                                                                                                                                                                                                                                                | 1.2, 1.3, 1.3.1, 1.3.2                                                                                    |
| [PCI.DSS.321-S3BucketLevelPublicAccessProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-level-public-access-prohibited.html)                       | The S3 bucket does not prohibit public access through bucket level settings.                                                                                                      | Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-S3BucketLoggingEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html)                                                   | The S3 Buckets does not have server access logs enabled.                                                                                                                          | Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.                                                                                                             | 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6 |
| [PCI.DSS.321-S3BucketPublicReadProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html)                                      | The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.                                                                | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2                                                   |
| [PCI.DSS.321-S3BucketPublicWriteProhibited](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html)                                    | The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.                                                               | The management of access should be consistent with the classification of the data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2                                                   |
| [PCI.DSS.321-S3BucketReplicationEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html)                                           | The S3 Bucket does not have replication enabled.                                                                                                                                  | Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.                                                                                                                                                                                                                                                                                                       | 2.2, 10.5.3                                                                                               |
| [PCI.DSS.321-S3BucketServerSideEncryptionEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html)                       | The S3 Bucket does not have default server-side encryption enabled.                                                                                                               | Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | 2.2, 3.4, 8.2.1, 10.5                                                                                     |
| [PCI.DSS.321-S3BucketVersioningEnabled](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html)                                             | The S3 Bucket does not have versioning enabled.                                                                                                                                   | Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.                                                                                                                                                                                                                                                                                                                                                               | 10.5.3                                                                                                    |
| [PCI.DSS.321-S3DefaultEncryptionKMS](https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html)                                                   | The S3 Bucket is not encrypted with a KMS Key by default.                                                                                                                         | Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                          | 3.4, 8.2.1, 10.5                                                                                          |
| [PCI.DSS.321-SageMakerEndpointConfigurationKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html) | The SageMaker resource endpoint is not encrypted with a KMS key.                                                                                                                  | Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-SageMakerNotebookInstanceKMSKeyConfigured](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html)           | The SageMaker notebook is not encrypted with a KMS key.                                                                                                                           | Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                                                                                                                                                                         | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-SageMakerNotebookNoDirectInternetAccess](https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html)               | The SageMaker notebook does not disable direct internet access.                                                                                                                   | By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |
| [PCI.DSS.321-SecretsManagerUsingKMSKey](https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-using-cmk.html)                                                 | The secret is not encrypted with a KMS Customer managed key.                                                                                                                      | To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                                              | 3.4, 8.2.1                                                                                                |
| [PCI.DSS.321-SNSEncryptedKMS](https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html)                                                                  | The SNS topic does not have KMS encryption enabled.                                                                                                                               | To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.                                                                                                                                                                                                                                                                                     | 8.2.1                                                                                                     |
| [PCI.DSS.321-VPCNoUnrestrictedRouteToIGW](https://docs.aws.amazon.com/config/latest/developerguide/no-unrestricted-route-to-igw.html)                                           | The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0').                                                                                  | Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.                                                                                                                                                                                                                                                                                                                                                | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2.2                                                                      |
| [PCI.DSS.321-VPCSubnetAutoAssignPublicIpDisabled](https://docs.aws.amazon.com/config/latest/developerguide/subnet-auto-assign-public-ip-disabled.html)                          | The subnet auto-assigns public IP addresses.                                                                                                                                      | Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.                                                                                                                                                                                                                                                 | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                        |

### Excluded Rules

Unimplemented rules from the AWS PCI DSS 3.2.1 Conformance Pack.

| AWS Config Rule Name                                                                                                                                                             | Summary                                                                                                                             | Relevant Control ID(s)                                                                                            |
| -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ----------------------------------------------------------------------------------------------------------------------------------- | ----------------------------------------------------------------------------------------------------------------- |
| [access-keys-rotated](https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html)                                                                         | IAM active access keys are rotated within the number of days specified in maxAccessKeyAge.                                          | 2.2                                                                                                               |
| [account-part-of-organizations](https://docs.aws.amazon.com/config/latest/developerguide/account-part-of-organizations.html)                                                     | The AWS Account is part of AWS Organizations.                                                                                       | 2.2                                                                                                               |
| [acm-certificate-expiration-check](https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html)                                               | AWS Certificate Manager Certificates in your account are valid and not marked for expiration within the specified number of days.   | 4.1                                                                                                               |
| [alb-http-drop-invalid-header-enabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html)                                       | AWS Certificate Manager Certificates in your account are valid and not marked for expiration within the specified number of days.   | 4.1, 8.2.1                                                                                                        |
| [alb-waf-enabled](https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html)                                                                                 | Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).                                                     | 6.6                                                                                                               |
| [api-gw-associated-with-waf](https://docs.aws.amazon.com/config/latest/developerguide/api-gw-associated-with-waf.html)                                                           | Web Application Firewall (WAF) is enabled on API Gateway stages.                                                                    | 6.6                                                                                                               |
| [cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html)                                                                           | The AWS Account has CloudTrail enabled.                                                                                             | 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.6, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6      |
| [cloudtrail-s3-dataevents-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html)                                               | The AWS Account has at least once AWS CloudTrail that logs Amazon S3 data events for all S3 buckets.                                | 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6                         |
| [cmk-backing-key-rotation-enabled](https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html)                                               | KMS backing keys have key rotation enabled and match to the key ID of the customer managed key.                                     | 2.2, 3.5, 3.6, 3.6.4                                                                                              |
| [codebuild-project-envvar-awscred-check](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html)                                   | KMS backing keys have key rotation enabled and match to the key ID of the customer managed key.                                     | 8.2.1                                                                                                             |
| [codebuild-project-source-repo-url-check](https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html)                                 | KMS backing keys have key rotation enabled and match to the key ID of the customer managed key.                                     | 8.2.1                                                                                                             |
| [cw-loggroup-retention-period-check](https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html)                                           | CloudWatch LogGroup retention period is set to specific number of days and is greater than the configured retention period.         | 3.1, 10.7                                                                                                         |
| [dynamodb-table-encrypted-kms](https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html)                                                       | DynamoDB tables are encrypted in KMS.<sup>[1](#dynamodb-table-encrypted-kms)</sup>                                                  | 3.4, 8.2.1                                                                                                        |
| [ebs-snapshot-public-restorable-check](https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html)                                       | EBS snapshots can be publicly restored.                                                                                             | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                                |
| [ec2-ebs-encryption-by-default](https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html)                                                     | Default encryption for EBS volumes is enabled at the AWS Account level.                                                             | 2.2, 3.4, 8.2.1                                                                                                   |
| [ec2-instance-managed-by-systems-manager](https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html)                                 | EC2 instances are managed by Systems Manager.                                                                                       | 2.4                                                                                                               |
| [ec2-managedinstance-association-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html) | Managed EC2 instances are compliant with their association's standards.                                                             | 6.2                                                                                                               |
| [ec2-managedinstance-patch-compliance-status-check](https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html)             | EC2 instances are compliant with their patch requirements.                                                                          | 6.2                                                                                                               |
| [elb-predefined-security-policy-ssl-check](https://docs.aws.amazon.com/config/latest/developerguide/elb-predefined-security-policy-ssl-check.html)                               | CLB SSL listeners use a specific predefined policy.                                                                                 | 2.3, 4.1, 8.2.1                                                                                                   |
| [emr-master-no-public-ip](https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html)                                                                 | EMR clusters' master nodes have no public IP.                                                                                       | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                                |
| [encrypted-volumes](https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html)                                                                             | EC2 instances are only attached to encrypted EBS volumes.                                                                           | 2.2, 3.4, 8.2.1                                                                                                   |
| [guardduty-enabled-centralized](https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html)                                                     | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                           | 5.1.2, 6.1, 11.4                                                                                                  |
| [iam-customer-policy-blocked-kms-actions](https://docs.aws.amazon.com/config/latest/developerguide/iam-customer-policy-blocked-kms-actions.html)                                 | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                           | 3.5.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                                 |
| [iam-group-has-users-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html)                                                             | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                           | 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                                        |
| [iam-inline-policy-blocked-kms-actions](https://docs.aws.amazon.com/config/latest/developerguide/iam-inline-policy-blocked-kms-actions.html)                                     | The AWS Account has Amazon GuardDuty enabled and results logged in a centralized account.                                           | 3.5.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2                                                                                 |
| [iam-password-policy](https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html)                                                                         | The account password policy for IAM users meets the specified requirements indicated in the parameters.                             | 8.2.3, 8.2.4, 8.2.5                                                                                               |
| [iam-root-access-key-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html)                                                             | The Account IAM Root User has an access key(s)                                                                                      | 2.2, 7.1.2, 7.2.1, 8.1.1                                                                                          |
| [iam-user-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html)                                                                       | AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.                                            | 8.3                                                                                                               |
| [iam-user-unused-credentials-check](https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html)                                             | IAM User passwords and active access keys have been used within a specified number of days.                                         | 2.2, 7.2.1, 8.1.4                                                                                                 |
| [kms-cmk-not-scheduled-for-deletion](https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html)                                           | AWS KMS keys are not scheduled for deletion in AWS Key Management Service (KMS).                                                    | 3.5, 3.6                                                                                                          |
| [lambda-function-public-access-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html)                               | Public access to lambda functions is prohibited.                                                                                    | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 2.2.2                                                                       |
| [mfa-enabled-for-iam-console-access](https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html)                                           | IAM Users have MFA enabled for console access.                                                                                      | 2.2, 8.3                                                                                                          |
| [multi-region-cloudtrail-enabled](https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html)                                                 | The AWS Account has at least one multi-region CloudTrail enabled.                                                                   | 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.6, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6 |
| [rds-snapshot-encrypted](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html)                                                                   | RDS snapshots are encrypted.                                                                                                        | 3.4, 8.2.1                                                                                                        |
| [rds-snapshots-public-prohibited](https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html)                                                 | RDS snapshots are not public.                                                                                                       | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                                |
| [redshift-cluster-kms-enabled](https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-kms-enabled.html)                                                       | Redshift clusters are encrypted with one of the specified KMS keys.                                                                 | 3.4, 8.2.1                                                                                                        |
| [redshift-require-tls-ssl](https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html)                                                               | Redshift clusters require TLS/SSL encryption.                                                                                       | 2.3, 4.1                                                                                                          |
| [root-account-hardware-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html)                                             | The AWS Account root user is hardware MFA enabled.                                                                                  | 2.2, 8.3                                                                                                          |
| [root-account-mfa-enabled](https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html)                                                               | The AWS Account root user is MFA enabled.                                                                                           | 2.2, 8.3                                                                                                          |
| [s3-account-level-public-access-blocks-periodic](https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks-periodic.html)                   | The AWS Account blocks S3 public access.                                                                                            | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2                                                           |
| [s3-bucket-policy-grantee-check](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html)                                                   | Access to S3 buckets is restricted to specific provided AWS principals, federated users, service principals, IP addresses, or VPCs. | 7.2.1                                                                                                             |
| [s3-bucket-ssl-requests-only](https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html)                                                         | S3 Buckets only allow SSL requests.                                                                                                 | 2.2, 4.1, 8.2.1                                                                                                   |
| [ssm-document-not-public](https://docs.aws.amazon.com/config/latest/developerguide/ssm-document-not-public.html)                                                                 | SSM documents are not public.                                                                                                       | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2                                                                |
| [vpc-flow-logs-enabled](https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html)                                                                     | Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.                                                        | 2.2, 10.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6                                                                 |
| [vpc-sg-open-only-to-authorized-ports](https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html)                                       | The VPC Security Group restricts IPv4 TCP traffic on unauthorized ports.<sup>[2](#vpc-sg-open-only-to-authorized-ports)</sup>       | 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2.2                                                                              |
| [wafv2-logging-enabled](https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html)                                                                     | Logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control list (ACLs).                      | 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4                                                      |

## Footnotes

<a id="dynamodb-table-encrypted-kms">1</a>: This rule is intentionally unimplemented. Server-side encryption at rest is enabled on all DynamoDB table data and [cannot be disabled](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/encryption.usagenotes.html#encryption.usagenotes.tabledata).

<a id="vpc-sg-open-only-to-authorized-ports">2</a>: This rule is intentionally unimplemented. The implementation of this rule is highly dependent upon organizational requirements.
