"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    /**
     * (experimental) Associate portfolio with an IAM Role.
     *
     * @experimental
     */
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM User.
     *
     * @experimental
     */
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM Group.
     *
     * @experimental
     */
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    /**
     * (experimental) Associate portfolio with the given product.
     *
     * @experimental
     */
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    /**
     * (experimental) Initiate a portfolio share with another account.
     *
     * @experimental
     */
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new servicecatalog_generated_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    /**
     * (experimental) Associate Tag Options.
     *
     * A TagOption is a key-value pair managed in AWS Service Catalog.
     * It is not an AWS tag, but serves as a template for creating an AWS tag based on the TagOption.
     *
     * @experimental
     */
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, tagOptions);
    }
    /**
     * (experimental) Add a Resource Update Constraint.
     *
     * @experimental
     */
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    /**
     * (experimental) Add notifications for supplied topics on the provisioned product.
     *
     * @experimental
     */
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    /**
     * (experimental) Set provisioning rules for the product.
     *
     * @experimental
     */
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    /**
     * (experimental) Force users to assume a certain role when launching a product.
     *
     * @experimental
     */
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    /**
     * (experimental) Configure deployment options using AWS Cloudformaiton StackSets.
     *
     * @experimental
     */
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new servicecatalog_generated_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * (experimental) A Service Catalog portfolio.
 *
 * @experimental
 */
class Portfolio extends PortfolioBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validatePortfolioProps(props);
        this.portfolio = new servicecatalog_generated_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * (experimental) Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     * @experimental
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    /**
     * (experimental) Create a unique id based off the L1 CfnPortfolio or the arn of an imported portfolio.
     *
     * @experimental
     */
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog.Portfolio", version: "1.119.0" };
//# sourceMappingURL=data:application/json;base64,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