import * as certmgr from '@aws-cdk/aws-certificatemanager';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as rds from '@aws-cdk/aws-rds';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import * as cdk from '@aws-cdk/core';
/**
 * Keycloak  version
 */
export declare class KeycloakVersion {
    readonly version: string;
    /**
     * Keycloak version 12.0.4
     */
    static readonly V12_0_4: KeycloakVersion;
    /**
     * Keycloak version 15.0.0
     */
    static readonly V15_0_0: KeycloakVersion;
    /**
     * Keycloak version 15.0.1
     */
    static readonly V15_0_1: KeycloakVersion;
    /**
     * Keycloak version 15.0.2
     */
    static readonly V15_0_2: KeycloakVersion;
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version: string): KeycloakVersion;
    /**
     *
     * @param version cluster version number
     */
    private constructor();
}
/**
 * The ECS task autoscaling definition
 */
export interface AutoScaleTask {
    /**
     * The minimal count of the task number
     *
     * @default - nodeCount
     */
    readonly min?: number;
    /**
     * The maximal count of the task number
     *
     * @default - min + 5
     */
    readonly max?: number;
    /**
     * The target cpu utilization for the service autoscaling
     *
     * @default 75
     */
    readonly targetCpuUtilization?: number;
}
export interface KeyCloakProps {
    /**
     * The Keycloak version for the cluster.
     */
    readonly keycloakVersion: KeycloakVersion;
    /**
     * The environment variables to pass to the keycloak container
     */
    readonly env?: {
        [key: string]: string;
    };
    /**
     * VPC for the workload
     */
    readonly vpc?: ec2.IVpc;
    /**
     * ACM certificate ARN to import
     */
    readonly certificateArn: string;
    /**
     * Create a bastion host for debugging or trouble-shooting
     *
     * @default false
     */
    readonly bastion?: boolean;
    /**
     * Number of keycloak node in the cluster
     *
     * @default 2
     */
    readonly nodeCount?: number;
    /**
     * VPC public subnets for ALB
     *
     * @default - VPC public subnets
     */
    readonly publicSubnets?: ec2.SubnetSelection;
    /**
     * VPC private subnets for keycloak service
     *
     * @default - VPC private subnets
     */
    readonly privateSubnets?: ec2.SubnetSelection;
    /**
     * VPC subnets for database
     *
     * @default - VPC isolated subnets
     */
    readonly databaseSubnets?: ec2.SubnetSelection;
    /**
     * Database instance type
     *
     * @default r5.large
     */
    readonly databaseInstanceType?: ec2.InstanceType;
    /**
     * The database instance engine
     *
     * @default - MySQL 8.0.21
     */
    readonly instanceEngine?: rds.IInstanceEngine;
    /**
     * The database cluster engine
     *
     * @default rds.AuroraMysqlEngineVersion.VER_2_09_1
     */
    readonly clusterEngine?: rds.IClusterEngine;
    /**
     * Whether to use aurora serverless. When enabled, the `databaseInstanceType` and
     * `engine` will be ignored. The `rds.DatabaseClusterEngine.AURORA_MYSQL` will be used as
     * the default cluster engine instead.
     *
     * @default false
     */
    readonly auroraServerless?: boolean;
    /**
     * Whether to use single RDS instance rather than RDS cluster. Not recommended for production.
     *
     * @default false
     */
    readonly singleDbInstance?: boolean;
    /**
     * database backup retension
     *
     * @default - 7 days
     */
    readonly backupRetention?: cdk.Duration;
    /**
     * The sticky session duration for the keycloak workload with ALB.
     *
     * @default - one day
     */
    readonly stickinessCookieDuration?: cdk.Duration;
    /**
     * Autoscaling for the ECS Service
     *
     * @default - no ecs service autoscaling
     */
    readonly autoScaleTask?: AutoScaleTask;
}
export declare class KeyCloak extends cdk.Construct {
    readonly vpc: ec2.IVpc;
    readonly db?: Database;
    constructor(scope: cdk.Construct, id: string, props: KeyCloakProps);
    addDatabase(props: DatabaseProps): Database;
    addKeyCloakContainerService(props: ContainerServiceProps): ContainerService;
    private _generateKeycloakSecret;
}
export interface DatabaseProps {
    /**
     * The VPC for the database
     */
    readonly vpc: ec2.IVpc;
    /**
     * VPC subnets for database
     */
    readonly databaseSubnets?: ec2.SubnetSelection;
    /**
     * The database instance type
     *
     * @default r5.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The database instance engine
     *
     * @default - MySQL 8.0.21
     */
    readonly instanceEngine?: rds.IInstanceEngine;
    /**
     * The database cluster engine
     *
     * @default rds.AuroraMysqlEngineVersion.VER_2_09_1
     */
    readonly clusterEngine?: rds.IClusterEngine;
    /**
     * enable aurora serverless
     *
     * @default false
     */
    readonly auroraServerless?: boolean;
    /**
     * Whether to use single RDS instance rather than RDS cluster. Not recommended for production.
     *
     * @default false
     */
    readonly singleDbInstance?: boolean;
    /**
     * database backup retension
     *
     * @default - 7 days
     */
    readonly backupRetention?: cdk.Duration;
}
/**
 * Database configuration
 */
export interface DatabaseCofig {
    /**
     * The database secret.
     */
    readonly secret: secretsmanager.ISecret;
    /**
     * The database connnections.
     */
    readonly connections: ec2.Connections;
    /**
     * The endpoint address for the database.
     */
    readonly endpoint: string;
    /**
     * The databasae identifier.
     */
    readonly identifier: string;
}
/**
 * Represents the database instance or database cluster
 */
export declare class Database extends cdk.Construct {
    readonly vpc: ec2.IVpc;
    readonly clusterEndpointHostname: string;
    readonly clusterIdentifier: string;
    readonly secret: secretsmanager.ISecret;
    readonly connections: ec2.Connections;
    private readonly _mysqlListenerPort;
    constructor(scope: cdk.Construct, id: string, props: DatabaseProps);
    private _createRdsInstance;
    private _createRdsCluster;
    private _createServerlessCluster;
}
export interface ContainerServiceProps {
    /**
     * The environment variables to pass to the keycloak container
     */
    readonly env?: {
        [key: string]: string;
    };
    /**
     * Keycloak version for the container image
     */
    readonly keycloakVersion: KeycloakVersion;
    /**
     * The VPC for the service
     */
    readonly vpc: ec2.IVpc;
    /**
     * VPC subnets for keycloak service
     */
    readonly privateSubnets?: ec2.SubnetSelection;
    /**
     * VPC public subnets for ALB
     */
    readonly publicSubnets?: ec2.SubnetSelection;
    /**
     * The RDS database for the service
     */
    readonly database: Database;
    /**
     * The secrets manager secret for the keycloak
     */
    readonly keycloakSecret: secretsmanager.ISecret;
    /**
     * The ACM certificate
     */
    readonly certificate: certmgr.ICertificate;
    /**
     * Whether to create the bastion host
     * @default false
     */
    readonly bastion?: boolean;
    /**
     * Whether to enable the ECS service deployment circuit breaker
     * @default false
     */
    readonly circuitBreaker?: boolean;
    /**
     * Number of keycloak node in the cluster
     *
     * @default 1
     */
    readonly nodeCount?: number;
    /**
     * The sticky session duration for the keycloak workload with ALB.
     *
     * @default - one day
     */
    readonly stickinessCookieDuration?: cdk.Duration;
    /**
     * Autoscaling for the ECS Service
     *
     * @default - no ecs service autoscaling
     */
    readonly autoScaleTask?: AutoScaleTask;
}
export declare class ContainerService extends cdk.Construct {
    readonly service: ecs.FargateService;
    constructor(scope: cdk.Construct, id: string, props: ContainerServiceProps);
    private getImageUriFromMap;
    private getKeyCloakDockerImageUri;
}
