"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsEventSource = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 */
class SqsEventSource {
    /**
     *
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
            throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize})`);
        }
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     */
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${core_1.Names.nodeUniqueId(this.queue.node)}`, {
            batchSize: this.props.batchSize,
            enabled: this.props.enabled,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping.
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
//# sourceMappingURL=data:application/json;base64,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