"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const LATEST_COMMIT_OUTPUT = "latest_commit";
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this.workflowPermissions = {
            contents: workflows_model_1.JobPermission.WRITE,
            ...options.workflowPermissions,
        };
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            workflowRunsOn: options.workflowRunsOn,
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ?? "release",
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ?? `release-${branchName}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.minMajorVersion !== undefined) {
            if (branch.majorVersion !== undefined) {
                throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
            }
            env.MIN_MAJOR = branch.minMajorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: `echo "${LATEST_COMMIT_OUTPUT}=$(git ls-remote origin -h \${{ github.ref }} | cut -f1)" >> $GITHUB_OUTPUT`,
        });
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, {
            name: "Upload artifact",
            if: noNewCommits,
            uses: "actions/upload-artifact@v3",
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: this.artifactsDirectory,
            },
        });
        if (this.github && !this.releaseTrigger.isManual) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseTrigger.schedule
                        ? [{ cron: this.releaseTrigger.schedule }]
                        : undefined,
                    push: this.releaseTrigger.isContinuous
                        ? { branches: [branchName] }
                        : undefined,
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: this.workflowPermissions,
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    "fetch-depth": 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
                runsOn: this.workflowRunsOn,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.67.87" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
//# sourceMappingURL=data:application/json;base64,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