"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const GIT_PATCH_FILE = ".repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded,
     *
     * @param options Options
     * @returns Job steps
     */
    static createUploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const steps = [
            {
                id: options.stepId,
                name: "Find mutations",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "${options.outputName}=true" >> $GITHUB_OUTPUT`,
                ].join("\n"),
            },
            {
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                uses: "actions/upload-artifact@v3",
                with: { name: GIT_PATCH_FILE, path: GIT_PATCH_FILE },
            },
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `createUploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    token: options.token,
                    ref: options.ref,
                    repository: options.repository,
                    ...(options.lfs ? { lfs: true } : {}),
                },
            },
            {
                name: "Download patch",
                uses: "actions/download-artifact@v3",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     */
    static setGitIdentity(id) {
        return [
            {
                name: "Set git identity",
                run: [
                    `git config user.name "${id.name}"`,
                    `git config user.email "${id.email}"`,
                ].join("\n"),
            },
        ];
    }
}
exports.WorkflowActions = WorkflowActions;
//# sourceMappingURL=data:application/json;base64,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